<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Services\Warehouse\ItemService;
use App\Services\Warehouse\InventoryService;
use App\Services\Auth\RoleBasedAccessService;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    protected $itemService;
    protected $inventoryService;
    protected $accessService;

    public function __construct(
        ItemService $itemService,
        InventoryService $inventoryService,
        RoleBasedAccessService $accessService
    ) {
        $this->itemService = $itemService;
        $this->inventoryService = $inventoryService;
        $this->accessService = $accessService;

        $this->middleware('warehouse.access');
    }

    /**
     * Display the warehouse dashboard.
     */
    public function index()
    {
        try {
            // Get basic statistics
            $stats = [
                'total_items' => 0,
                'low_stock_items' => 0,
                'total_locations' => 0,
                'pending_transactions' => 0,
            ];

            // Get recent activities
            $recentActivities = collect();

            // Get low stock items (top 5)
            $lowStockItems = collect();

            // Get inventory summary by location
            $locationSummary = collect();

            // Get recent stock movements
            $recentMovements = collect();

            $response = response()->view('warehouse.dashboard.index', compact(
                'stats',
                'recentActivities',
                'lowStockItems',
                'locationSummary',
                'recentMovements'
            ));

            $response->header('Content-Type', 'text/html; charset=utf-8');
            return $response;
        } catch (\Exception $e) {
            // Log error and show basic dashboard
            \Log::error('Warehouse dashboard error: ' . $e->getMessage());

            return view('warehouse.dashboard.index', [
                'stats' => ['total_items' => 0, 'low_stock_items' => 0, 'total_locations' => 0, 'pending_transactions' => 0],
                'recentActivities' => collect(),
                'lowStockItems' => collect(),
                'locationSummary' => collect(),
                'recentMovements' => collect(),
            ]);
        }
    }

    /**
     * Get quick stats for API.
     */
    public function quickStats()
    {
        return response()->json([
            'total_items' => 0,
            'low_stock_items' => 0,
            'pending_transactions' => 0,
        ]);
    }

    /**
     * Get pending transactions count.
     */
    protected function getPendingTransactionsCount()
    {
        $count = 0;

        // Count pending incoming transactions
        if (class_exists('\App\Models\Warehouse\IncomingTransaction')) {
            $count += \App\Models\Warehouse\IncomingTransaction::where('status', 'pending')->count();
        }

        // Count pending outgoing transactions
        if (class_exists('\App\Models\Warehouse\OutgoingTransaction')) {
            $count += \App\Models\Warehouse\OutgoingTransaction::whereIn('status', ['pending', 'approved'])->count();
        }

        // Count pending returns
        if (class_exists('\App\Models\Warehouse\ReturnTransaction')) {
            $count += \App\Models\Warehouse\ReturnTransaction::where('status', 'pending')->count();
        }

        return $count;
    }

    /**
     * Get recent activities.
     */
    protected function getRecentActivities()
    {
        $activities = collect();

        // Get recent stock movements
        if (class_exists('\App\Models\Warehouse\StockMovement')) {
            $stockMovements = \App\Models\Warehouse\StockMovement::with(['item', 'user'])
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get()
                ->map(function ($movement) {
                    return [
                        'type' => 'stock_movement',
                        'description' => $this->formatStockMovementActivity($movement),
                        'user' => $movement->user?->name ?? 'System',
                        'created_at' => $movement->created_at,
                        'icon' => $this->getStockMovementIcon($movement->type),
                        'color' => $this->getStockMovementColor($movement->type),
                    ];
                });

            $activities = $activities->merge($stockMovements);
        }

        return $activities->sortByDesc('created_at')->take(10)->values();
    }

    /**
     * Format stock movement activity description.
     */
    protected function formatStockMovementActivity($movement)
    {
        $itemName = $movement->item?->name ?? 'Unknown Item';
        $quantity = $movement->quantity;

        switch ($movement->type) {
            case 'increase':
                return "Added {$quantity} units of {$itemName}";
            case 'decrease':
                return "Removed {$quantity} units of {$itemName}";
            case 'adjustment':
                return "Adjusted {$itemName} by {$quantity} units";
            case 'transfer_in':
                return "Received {$quantity} units of {$itemName} (transfer)";
            case 'transfer_out':
                return "Sent {$quantity} units of {$itemName} (transfer)";
            case 'reserve':
                return "Reserved {$quantity} units of {$itemName}";
            case 'release':
                return "Released {$quantity} units of {$itemName}";
            default:
                return "Stock movement: {$quantity} units of {$itemName}";
        }
    }

    /**
     * Get stock movement icon.
     */
    protected function getStockMovementIcon($type)
    {
        switch ($type) {
            case 'increase':
            case 'transfer_in':
                return 'add_circle';
            case 'decrease':
            case 'transfer_out':
                return 'remove_circle';
            case 'adjustment':
                return 'tune';
            case 'reserve':
                return 'bookmark';
            case 'release':
                return 'bookmark_border';
            default:
                return 'swap_horiz';
        }
    }

    /**
     * Get stock movement color.
     */
    protected function getStockMovementColor($type)
    {
        switch ($type) {
            case 'increase':
            case 'transfer_in':
                return 'success';
            case 'decrease':
            case 'transfer_out':
                return 'danger';
            case 'adjustment':
                return 'warning';
            case 'reserve':
            case 'release':
                return 'info';
            default:
                return 'secondary';
        }
    }

    /**
     * Get location summary.
     */
    protected function getLocationSummary()
    {
        if (!class_exists('\App\Models\Warehouse\Inventory')) {
            return collect();
        }

        return \App\Models\Warehouse\Inventory::selectRaw('
            location,
            COUNT(*) as total_items,
            SUM(quantity_available) as total_quantity,
            SUM(quantity_reserved) as total_reserved
        ')
        ->groupBy('location')
        ->get()
        ->map(function ($location) {
            return [
                'name' => $location->location ?: 'Default',
                'total_items' => $location->total_items,
                'total_quantity' => $location->total_quantity,
                'total_reserved' => $location->total_reserved,
                'available_quantity' => $location->total_quantity - $location->total_reserved,
            ];
        });
    }

    /**
     * Get recent stock movements.
     */
    protected function getRecentStockMovements()
    {
        if (!class_exists('\App\Models\Warehouse\StockMovement')) {
            return collect();
        }

        return \App\Models\Warehouse\StockMovement::with(['item', 'user'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();
    }
}