<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Services\Warehouse\ItemService;
use App\Services\Auth\RoleBasedAccessService;
use Illuminate\Http\Request;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ItemController extends Controller
{
    protected $itemService;
    protected $accessService;

    public function __construct(ItemService $itemService, RoleBasedAccessService $accessService)
    {
        $this->itemService = $itemService;
        $this->accessService = $accessService;

        $this->middleware('warehouse.access');
        $this->middleware('warehouse.permission:warehouse.items.view')->only(['index', 'show']);
        $this->middleware('warehouse.permission:warehouse.items.create')->only(['create', 'store']);
        $this->middleware('warehouse.permission:warehouse.items.edit')->only(['edit', 'update']);
        $this->middleware('warehouse.permission:warehouse.items.delete')->only(['destroy']);
    }

    /**
     * Display a listing of the items.
     */
    public function index(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.view

        if ($request->expectsJson()) {
            return $this->getDataForTable($request);
        }

        return view('warehouse.items.index');
    }

    /**
     * Get data for DataTable.
     */
    private function getDataForTable(Request $request)
    {
        $start = $request->get('start', 0);
        $length = $request->get('length', 25);
        $search = $request->get('search.value');

        $query = \App\Models\Warehouse\Item::with('category');

        // Apply search
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('item_code', 'LIKE', "%{$search}%")
                  ->orWhere('item_description', 'LIKE', "%{$search}%")
                  ->orWhereHas('category', function ($q) use ($search) {
                      $q->where('category_name', 'LIKE', "%{$search}%");
                  });
            });
        }

        // Apply filters
        if ($request->filled('status')) {
            $query->where('status', $request->get('status'));
        }
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->get('category_id'));
        }

        // Get total count
        $totalRecords = \App\Models\Warehouse\Item::count();
        $filteredRecords = $query->count();

        // Apply pagination
        $items = $query->skip($start)
                      ->take($length)
                      ->orderBy('created_at', 'desc')
                      ->get();

        // Format data for DataTable
        $data = $items->map(function ($item) {
            return [
                'id' => $item->id,
                'item_code' => $item->item_code,
                'item_description' => $item->item_description,
                'category_name' => $item->category ? $item->category->category_name : 'No Category',
                'unit_of_measure' => $item->unit_of_measure,
                'status' => $item->status,
                'min_stock_level' => $item->min_stock_level,
                'max_stock_level' => $item->max_stock_level,
                'reorder_point' => $item->reorder_point,
                'created_at' => $item->created_at->format('M d, Y H:i'),
                'created_by' => 'Admin', // You can add created_by field later
            ];
        });

        return response()->json([
            'draw' => intval($request->get('draw')),
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data
        ]);
    }

    /**
     * Show the form for creating a new item.
     */
    public function create()
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        $categories = \App\Models\Warehouse\ItemCategory::where('status', 'active')->get();

        return view('warehouse.items.create', compact('categories'));
    }

    /**
     * Store a newly created item in storage.
     */
    public function store(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        $validatedData = $request->validate([
            'item_code' => 'nullable|string|max:50|unique:items,item_code',
            'item_description' => 'required|string|max:255',
            'category_id' => 'required|exists:item_categories,id',
            'unit_of_measure' => 'required|string|max:50',
            'min_stock_level' => 'nullable|numeric|min:0',
            'max_stock_level' => 'nullable|numeric|min:0',
            'reorder_point' => 'nullable|numeric|min:0',
            'status' => 'required|in:active,inactive,discontinued',
            'notes' => 'nullable|string|max:1000'
        ]);

        // Generate item code if not provided
        if (empty($validatedData['item_code'])) {
            $validatedData['item_code'] = $this->generateItemCode($validatedData['category_id']);
        }

        try {
            $item = \App\Models\Warehouse\Item::create($validatedData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $item->load('category'),
                    'message' => 'Item created successfully'
                ], 201);
            }

            return redirect()->route('warehouse.items.index')
                           ->with('success', 'Item created successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Generate item code based on category
     */
    private function generateItemCode($categoryId)
    {
        $category = \App\Models\Warehouse\ItemCategory::find($categoryId);
        $prefix = $category ? strtoupper(substr($category->category_name, 0, 3)) : 'ITM';
        $timestamp = now()->format('ymd');
        $sequence = str_pad(\App\Models\Warehouse\Item::where('item_code', 'LIKE', $prefix . '%')->count() + 1, 3, '0', STR_PAD_LEFT);

        return $prefix . $timestamp . $sequence;
    }

    /**
     * Display the specified item.
     */
    public function show($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.view

        $item->load(['category', 'inventories', 'stockMovements.user']);

        return view('warehouse.items.show', compact('item'));
    }

    /**
     * Show the form for editing the specified item.
     */
    public function edit($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.edit

        $categories = \App\Models\Warehouse\ItemCategory::where('status', 'active')->get();

        return view('warehouse.items.edit', compact('item', 'categories'));
    }

    /**
     * Update the specified item in storage.
     */
    public function update(Request $request, $id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.edit

        $request->validate([
            'name' => 'required|string|max:255',
            'category_id' => 'nullable|exists:item_categories,id',
            'description' => 'nullable|string',
            'unit' => 'required|string|max:50',
            'unit_cost' => 'nullable|numeric|min:0',
            'selling_price' => 'nullable|numeric|min:0',
            'reorder_point' => 'nullable|integer|min:0',
            'max_stock_level' => 'nullable|integer|min:0',
            'location' => 'nullable|string|max:255',
            'status' => 'required|in:active,inactive'
        ]);

        try {
            $updatedItem = $this->itemService->update($id, $request->validated());

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'data' => $updatedItem,
                    'message' => 'Item updated successfully'
                ]);
            }

            return redirect()->route('warehouse.items.show', $updatedItem)
                           ->with('success', 'Item updated successfully');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => $e->getMessage()
                ], 400);
            }

            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Remove the specified item from storage.
     */
    public function destroy($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.delete

        try {
            $this->itemService->delete($id);

            return response()->json([
                'success' => true,
                'message' => 'Item deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Generate barcode for item.
     */
    public function generateBarcode($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        try {
            $barcodeUrl = $this->itemService->generateBarcode($item);

            return response()->json([
                'success' => true,
                'barcode_url' => $barcodeUrl,
                'message' => 'Barcode generated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Generate QR code for item.
     */
    public function generateQrCode($id)
    {
        $item = $this->itemService->findById($id);

        if (!$item) {
            abort(404, 'Item not found');
        }

        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        try {
            $qrCodeUrl = $this->itemService->generateQrCode($item);

            return response()->json([
                'success' => true,
                'qr_code_url' => $qrCodeUrl,
                'message' => 'QR code generated successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Bulk import items.
     */
    public function bulkImport(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        $request->validate([
            'items' => 'required|array|min:1',
            'items.*.name' => 'required|string|max:255',
            'items.*.category_id' => 'nullable|exists:item_categories,id',
            'items.*.unit' => 'required|string|max:50',
        ]);

        try {
            $results = $this->itemService->bulkImport($request->input('items'));

            return response()->json([
                'success' => true,
                'data' => $results,
                'message' => "Successfully imported {$results['success']} items"
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Export items.
     */
    public function export(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.view

        $filters = $request->only([
            'category_id', 'status', 'location', 'low_stock', 'out_of_stock',
            'created_from', 'created_to'
        ]);

        try {
            $itemsData = $this->itemService->exportItems($filters);

            return response()->json([
                'success' => true,
                'data' => $itemsData,
                'message' => 'Items exported successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Download the item import template.
     */
    public function downloadTemplate()
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.view

        try {
            $filePath = $this->generateImportTemplate();
            $filename = 'items_import_template.xlsx';

            return response()->download($filePath, $filename)->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 400);
        }
    }

    /**
     * Import items from Excel file.
     */
    public function import(Request $request)
    {
        // Authorization is handled by middleware: warehouse.permission:warehouse.items.create

        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:10240', // 10MB max
            'category_id' => 'required|exists:item_categories,id'
        ]);

        try {
            $results = $this->importFromExcel($request->file('file'), $request->category_id);

            return response()->json([
                'success' => true,
                'imported' => $results['success'],
                'updated' => 0, // Currently we only create new items
                'skipped' => count($results['errors']),
                'data' => $results,
                'message' => "Import completed. {$results['success']} items imported successfully."
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 422);
        }
    }

    /**
     * Generate the import template Excel file (without category column).
     */
    private function generateImportTemplate()
    {
        $spreadsheet = new Spreadsheet();

        // Data Sheet
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Data');

        // Headers (NO category column - it will be selected in dropdown)
        $headers = [
            'A1' => 'item_code',
            'B1' => 'item_description',
            'C1' => 'unit_of_measure',
            'D1' => 'min_stock_level',
            'E1' => 'max_stock_level',
            'F1' => 'reorder_point',
            'G1' => 'status',
            'H1' => 'notes'
        ];

        foreach ($headers as $cell => $header) {
            $sheet->setCellValue($cell, $header);
            $sheet->getStyle($cell)->getFont()->setBold(true);
        }

        // Sample data
        $sampleData = [
            2 => [
                'A' => 'ELE241201001',
                'B' => 'LED Light Bulb 10W',
                'C' => 'piece',
                'D' => '10',
                'E' => '100',
                'F' => '20',
                'G' => 'active',
                'H' => 'Energy efficient LED bulb'
            ],
            3 => [
                'A' => 'CON241201002',
                'B' => 'Cement Bag 50kg',
                'C' => 'bag',
                'D' => '5',
                'E' => '50',
                'F' => '10',
                'G' => 'active',
                'H' => 'Premium quality cement'
            ],
            4 => [
                'A' => '', // Empty code will be auto-generated
                'B' => 'Office Paper A4',
                'C' => 'ream',
                'D' => '2',
                'E' => '20',
                'F' => '5',
                'G' => 'active',
                'H' => 'Standard office printing paper'
            ]
        ];

        foreach ($sampleData as $row => $data) {
            foreach ($data as $col => $value) {
                $sheet->setCellValue($col . $row, $value);
            }
        }

        // Add comments/notes
        $sheet->getComment('A1')->getText()->createTextRun('Optional. Leave empty for auto-generation. Must be unique if provided.');
        $sheet->getComment('B1')->getText()->createTextRun('Required. Item description (max 255 characters)');
        $sheet->getComment('C1')->getText()->createTextRun('Required. Unit of measure (piece, kg, meter, etc.)');
        $sheet->getComment('D1')->getText()->createTextRun('Optional. Minimum stock level (numeric)');
        $sheet->getComment('E1')->getText()->createTextRun('Optional. Maximum stock level (numeric)');
        $sheet->getComment('F1')->getText()->createTextRun('Optional. Reorder point threshold (numeric)');
        $sheet->getComment('G1')->getText()->createTextRun('Required. Status: active, inactive, or discontinued');
        $sheet->getComment('H1')->getText()->createTextRun('Optional. Additional notes about the item');

        // Auto-size columns
        foreach (range('A', 'H') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        // Instructions Sheet
        $instructionsSheet = $spreadsheet->createSheet();
        $instructionsSheet->setTitle('Instructions');

        $instructions = [
            'ITEMS IMPORT INSTRUCTIONS',
            '',
            '1. CATEGORY SELECTION:',
            '   - Category is selected in the import modal dropdown',
            '   - All items in this file will be assigned to the selected category',
            '   - Do NOT include category information in the Excel file',
            '',
            '2. REQUIRED FIELDS:',
            '   - item_description: Item name/description (required)',
            '   - unit_of_measure: Unit of measure (required)',
            '   - status: active, inactive, or discontinued (required)',
            '',
            '3. OPTIONAL FIELDS:',
            '   - item_code: Leave empty for auto-generation',
            '   - min_stock_level, max_stock_level, reorder_point: Numeric values',
            '   - notes: Additional information',
            '',
            '4. VALIDATION RULES:',
            '   - Item codes must be unique (if provided)',
            '   - Item descriptions are required and max 255 characters',
            '   - Stock levels must be numeric (can be decimals)',
            '   - Status must be exactly: active, inactive, or discontinued',
            '',
            '5. AUTO-GENERATION:',
            '   - Item codes are auto-generated if left empty',
            '   - Format: [CATEGORY_PREFIX][DATE][SEQUENCE]',
            '   - Example: ELE241201001 (Electronics, Dec 1, 2024, item #1)',
            '',
            '6. DATA FORMAT:',
            '   - Do not modify the header row',
            '   - Delete the sample data rows before importing your data',
            '   - Ensure proper spelling for status values',
            '   - Use numeric values for stock levels',
            '',
            '7. IMPORT PROCESS:',
            '   - Select category in the import modal first',
            '   - Upload only .xlsx, .xls, or .csv files',
            '   - Maximum file size: 10MB',
            '   - Duplicate item codes will be skipped',
            '   - Invalid rows will be reported in import results',
            '',
            '8. EXAMPLE DATA:',
            '   LED Light Bulb 10W | piece | 10 | 100 | 20 | active',
            '   Cement Bag 50kg | bag | 5 | 50 | 10 | active',
            '   Office Paper A4 | ream | 2 | 20 | 5 | active'
        ];

        foreach ($instructions as $index => $instruction) {
            $instructionsSheet->setCellValue('A' . ($index + 1), $instruction);
            if ($index === 0) {
                $instructionsSheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
            }
        }

        $instructionsSheet->getColumnDimension('A')->setWidth(80);

        // Set active sheet back to data
        $spreadsheet->setActiveSheetIndex(0);

        // Save file
        $tempDir = storage_path('app/temp');
        if (!file_exists($tempDir)) {
            mkdir($tempDir, 0755, true);
        }

        $fileName = 'items_import_template.xlsx';
        $filePath = $tempDir . '/' . $fileName;

        $writer = new Xlsx($spreadsheet);
        $writer->save($filePath);

        return $filePath;
    }

    /**
     * Import items from Excel file with category from dropdown.
     */
    private function importFromExcel($file, $categoryId)
    {
        $spreadsheet = IOFactory::load($file->getPathname());
        $sheet = $spreadsheet->getActiveSheet();
        $rows = $sheet->toArray();

        // Remove header row
        array_shift($rows);

        $imported = 0;
        $errors = [];

        // Verify category exists
        $category = \App\Models\Warehouse\ItemCategory::find($categoryId);
        if (!$category) {
            throw new \Exception('Selected category not found');
        }

        foreach ($rows as $index => $row) {
            $rowNumber = $index + 2; // +2 because we removed header and arrays are 0-indexed

            // Skip empty rows
            if (empty(array_filter($row))) {
                continue;
            }

            try {
                // Map columns (NO category column)
                $itemData = [
                    'item_code' => trim($row[0] ?? ''),
                    'item_description' => trim($row[1] ?? ''),
                    'unit_of_measure' => trim($row[2] ?? ''),
                    'min_stock_level' => is_numeric($row[3]) ? (float)$row[3] : null,
                    'max_stock_level' => is_numeric($row[4]) ? (float)$row[4] : null,
                    'reorder_point' => is_numeric($row[5]) ? (float)$row[5] : null,
                    'status' => trim($row[6] ?? 'active'),
                    'notes' => trim($row[7] ?? ''),
                    'category_id' => $categoryId // Set from dropdown selection
                ];

                // Validate required fields
                if (empty($itemData['item_description'])) {
                    $errors[] = "Row {$rowNumber}: item_description is required";
                    continue;
                }

                if (empty($itemData['unit_of_measure'])) {
                    $errors[] = "Row {$rowNumber}: unit_of_measure is required";
                    continue;
                }

                // Validate status
                if (!in_array($itemData['status'], ['active', 'inactive', 'discontinued'])) {
                    $errors[] = "Row {$rowNumber}: status must be 'active', 'inactive', or 'discontinued'";
                    continue;
                }

                // Generate item code if empty
                if (empty($itemData['item_code'])) {
                    $itemData['item_code'] = $this->generateItemCode($categoryId);
                } else {
                    // Check for duplicate item code
                    if (\App\Models\Warehouse\Item::where('item_code', $itemData['item_code'])->exists()) {
                        $errors[] = "Row {$rowNumber}: Item code '{$itemData['item_code']}' already exists";
                        continue;
                    }
                }

                // Create item
                \App\Models\Warehouse\Item::create($itemData);
                $imported++;

            } catch (\Exception $e) {
                $errors[] = "Row {$rowNumber}: " . $e->getMessage();
            }
        }

        return [
            'success' => $imported,
            'errors' => $errors,
            'category' => $category->category_name
        ];
    }
}