<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\OutgoingItem;
use App\Models\Warehouse\OutgoingFulfillment;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectInventory;
use App\Models\Warehouse\Supplier;
use App\Models\Warehouse\Inventory;
use App\Models\DirectDelivery;
use App\Models\DirectDeliveryItem;
use App\Models\Warehouse\MaterialTransferRequest;
use App\Models\Warehouse\MaterialTransferRequestItem;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class MaterialIssueController extends Controller
{
    /**
     * Display the material issue decision page
     */
    public function index(Request $request)
    {
        Gate::authorize('viewAny', \App\Models\Warehouse\Inventory::class);

        // Get pending material requests that need issue decisions
        $query = OutgoingTransaction::with(['project', 'division', 'requester', 'items.item'])
                                   ->where('status', 'pending')
                                   ->latest('created_at');

        // Apply filters
        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('date_from')) {
            $query->where('transaction_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->where('transaction_date', '<=', $request->date_to);
        }

        $pendingRequests = $query->paginate(15);

        // Debug: Log what we found
        \Log::info('Material Issue Index - Pending Requests', [
            'count' => $pendingRequests->count(),
            'status_check' => OutgoingTransaction::select('id', 'status', 'material_request_number')->get()->toArray()
        ]);

        // Get filter options
        $projects = Project::all();

        // Get statistics
        $stats = [
            'pending_decisions' => OutgoingTransaction::where('status', 'pending')->count(),
            'inventory_issues' => OutgoingTransaction::where('status', 'successful')->count(),
            'direct_deliveries' => \App\Models\DirectDelivery::whereIn('status', ['pending', 'delivered'])->count(),
            'project_transfers' => \App\Models\Warehouse\MaterialTransferRequest::whereIn('status', ['approved', 'partially_approved'])->count()
        ];

        return view('warehouse.material-issue.index', compact(
            'pendingRequests', 'projects', 'stats'
        ));
    }

    /**
     * Show the issue decision form for a specific material request
     */
    public function show(OutgoingTransaction $transaction)
    {
        Gate::authorize('view', $transaction);

        $transaction->load(['project', 'division', 'requester', 'items.item']);

        // Get issue history
        $fulfillments = OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
            ->get()
            ->groupBy('item_id');

        // Calculate remaining quantities for each item
        foreach ($transaction->items as $item) {
            $issuedQty = 0;
            if (isset($fulfillments[$item->item_id])) {
                $issuedQty = $fulfillments[$item->item_id]->sum('quantity_fulfilled');
            }
            $item->quantity_fulfilled = $issuedQty;
            $item->quantity_remaining = $item->quantity_requested - $issuedQty;
        }

        // Get options for different issue types
        $projects = Project::where('id', '!=', $transaction->project_id)->get();
        $suppliers = Supplier::where('status', 'active')->get();

        // Use multi-source view for better flexibility
        return view('warehouse.material-issue.decision-multi', compact(
            'transaction', 'projects', 'suppliers'
        ));
    }

    /**
     * Process the issue decision
     */
    public function processDecision(Request $request, OutgoingTransaction $transaction)
    {
        $request->validate([
            'issue_type' => 'required|in:inventory_dispatch,direct_delivery,project_transfer',
            'notes' => 'nullable|string',
        ]);

        $issueType = $request->issue_type;

        switch ($issueType) {
            case 'inventory_dispatch':
                return $this->processInventoryDispatch($request, $transaction);

            case 'direct_delivery':
                return $this->processDirectDelivery($request, $transaction);

            case 'project_transfer':
                return $this->processProjectTransfer($request, $transaction);

            default:
                return back()->with('error', 'Invalid issue type selected.');
        }
    }

    /**
     * Process inventory dispatch decision
     */
    private function processInventoryDispatch(Request $request, OutgoingTransaction $transaction)
    {
        $request->validate([
            'dispatch_date' => 'required|date|after_or_equal:today',
            'receiver_name' => 'required|string|max:255',
            'items' => 'required|array',
            'items.*.quantity_to_issue' => 'required|numeric|min:0.01',
        ]);

        // Update transaction items with quantities to issue
        foreach ($request->items as $itemId => $itemData) {
            $outgoingItem = $transaction->items()->where('item_id', $itemId)->first();
            if ($outgoingItem) {
                $outgoingItem->update([
                    'quantity_released' => $itemData['quantity_to_issue']
                ]);
            }
        }

        // Update transaction status and dispatch
        $transaction->update([
            'status' => 'processing',
            'receiver_name' => $request->receiver_name,
            'notes' => $request->notes
        ]);

        // Process the dispatch
        $transaction->approveAndDispatch(auth()->id());

        return redirect()->route('warehouse.material-issue.index')
            ->with('success', 'Material request processed for inventory dispatch successfully.');
    }

    /**
     * Process direct delivery decision
     */
    private function processDirectDelivery(Request $request, OutgoingTransaction $transaction)
    {
        $request->validate([
            'supplier_id' => 'nullable|exists:suppliers,id',
            'delivery_date' => 'required|date',
            'purchase_order_number' => 'nullable|string|max:255',
        ]);

        // Create Direct Delivery based on the material request
        $directDelivery = \App\Models\DirectDelivery::create([
            'delivery_date' => $request->delivery_date,
            'project_id' => $transaction->project_id,
            'project_division_id' => $transaction->division_id,
            'supplier_id' => $request->supplier_id,
            'purchase_order_number' => $request->purchase_order_number,
            'operation_type' => 'supplier_delivery',
            'delivery_type' => 'purchase_order',
            'workflow_status' => 'pending_supplier',
            'status' => 'pending',
            'material_request_id' => null,
            'currency' => 'AED',
            'exchange_rate' => 1.0000,
            'vat_rate' => 5.00,
        ]);

        // Create direct delivery items
        foreach ($transaction->items as $outgoingItem) {
            \App\Models\DirectDeliveryItem::create([
                'direct_delivery_id' => $directDelivery->id,
                'item_id' => $outgoingItem->item_id,
                'quantity_ordered' => $outgoingItem->quantity_requested,
                'ordered_quantity' => $outgoingItem->quantity_requested,
                'unit_price' => $outgoingItem->unit_price ?: 0,
                'unit_price_aed' => $outgoingItem->unit_price ?: 0,
                'currency' => 'AED',
                'status' => 'pending',
            ]);
        }

        // Update original transaction status
        $transaction->update([
            'status' => 'converted_to_direct_delivery',
            'notes' => $request->notes
        ]);

        return redirect()->route('warehouse.direct-deliveries.show', $directDelivery->id)
            ->with('success', 'Material request converted to direct delivery. Please update supplier and delivery details.');
    }

    /**
     * Process project transfer decision
     */
    private function processProjectTransfer(Request $request, OutgoingTransaction $transaction)
    {
        $request->validate([
            'source_project_id' => 'required|exists:projects,id|different:' . $transaction->project_id,
            'transfer_reason' => 'required|string|max:255',
            'required_date' => 'required|date|after:today',
            'priority' => 'required|in:low,medium,high,urgent',
        ]);

        // Create Material Transfer Request
        $transferRequest = \App\Models\Warehouse\MaterialTransferRequest::create([
            'request_date' => now(),
            'transferrer_project_id' => $request->source_project_id,
            'receiver_project_id' => $transaction->project_id,
            'transfer_reason' => $request->transfer_reason,
            'transfer_notes' => $request->notes,
            'priority' => $request->priority,
            'required_date' => $request->required_date,
            'requested_by' => auth()->id(),
            'status' => 'pending_transferrer_approval'
        ]);

        // Create transfer request items
        foreach ($transaction->items as $outgoingItem) {
            // Check available quantity in source project
            $projectInventory = \App\Models\Warehouse\ProjectInventory::where('project_id', $request->source_project_id)
                ->where('item_id', $outgoingItem->item_id)
                ->first();

            $availableQuantity = $projectInventory ? $projectInventory->available_for_transfer : 0;
            $estimatedUnitPrice = $projectInventory ? $projectInventory->unit_price : ($outgoingItem->unit_price ?: 0);

            \App\Models\Warehouse\MaterialTransferRequestItem::create([
                'transfer_request_id' => $transferRequest->id,
                'item_id' => $outgoingItem->item_id,
                'quantity_requested' => $outgoingItem->quantity_requested,
                'quantity_available' => $availableQuantity,
                'estimated_unit_price' => $estimatedUnitPrice,
                'transfer_notes' => "Generated from material request: {$transaction->material_request_number}"
            ]);
        }

        // Update original transaction status
        $transaction->update([
            'status' => 'converted_to_transfer',
            'notes' => $request->notes
        ]);

        return redirect()->route('warehouse.transfer-requests.show', $transferRequest->id)
            ->with('success', 'Material request converted to project transfer. Awaiting source project approval.');
    }

    /**
     * Process multi-source material issue decision
     */
    public function processMultiSource(Request $request, OutgoingTransaction $transaction)
    {
        // Debug: Check if method is being called
        if ($request->has('debug')) {
            return response()->json([
                'message' => 'Method reached successfully',
                'transaction_id' => $transaction->id,
                'request_data' => $request->all()
            ]);
        }

        \Log::info('processMultiSource called', [
            'transaction_id' => $transaction->id,
            'request_data' => $request->all()
        ]);

        $request->validate([
            'enable_inventory' => 'required|in:0,1',
            'enable_delivery' => 'required|in:0,1',
            'enable_transfer' => 'required|in:0,1',
            'inventory' => 'nullable|array',
            'delivery' => 'nullable|array',
            'transfer' => 'nullable|array',
            'inventory.*' => 'nullable|numeric|min:0',
            'delivery.*' => 'nullable|numeric|min:0',
            'transfer.*' => 'nullable|numeric|min:0',
        ]);

        \Log::info('Validation passed');

        // Convert string values to boolean
        $enableInventory = (bool) $request->input('enable_inventory');
        $enableDelivery = (bool) $request->input('enable_delivery');
        $enableTransfer = (bool) $request->input('enable_transfer');

        // Ensure at least one issue method is enabled
        if (!$enableInventory && !$enableDelivery && !$enableTransfer) {
            return back()->with('error', 'Please enable at least one issue method.');
        }

        \Log::info('Multi-source form data:', [
            'enable_inventory' => $request->input('enable_inventory'),
            'enable_delivery' => $request->input('enable_delivery'),
            'enable_transfer' => $request->input('enable_transfer'),
            'converted_booleans' => [$enableInventory, $enableDelivery, $enableTransfer],
            'inventory_data' => $request->inventory,
            'delivery_data' => $request->delivery,
            'transfer_data' => $request->transfer
        ]);

        try {
            \DB::beginTransaction();

            $results = [];
            $inventoryTotal = 0;
            $deliveryTotal = 0;
            $transferTotal = 0;

            // Initialize arrays to track what's being fulfilled
            $inventoryItems = [];
            $deliveryItems = [];
            $transferItems = [];

        // Process Inventory Dispatch if enabled
        if ($enableInventory && !empty($request->inventory)) {
            $inventoryItems = array_filter($request->inventory, function($qty) { return $qty > 0; });
            if (!empty($inventoryItems)) {
                $inventoryResult = $this->processInventoryPortion($request, $transaction, $inventoryItems);
                $results[] = $inventoryResult;
                $inventoryTotal = array_sum($inventoryItems);
            }
        }

        // Process Direct Delivery if enabled
        if ($enableDelivery && !empty($request->delivery)) {
            $deliveryItems = array_filter($request->delivery, function($qty) { return $qty > 0; });
            if (!empty($deliveryItems)) {
                $deliveryResult = $this->processDirectDeliveryPortion($request, $transaction, $deliveryItems);
                $results[] = $deliveryResult;
                $deliveryTotal = array_sum($deliveryItems);
            }
        }

        // Process Project Transfer if enabled
        if ($enableTransfer && !empty($request->transfer)) {
            $transferItems = array_filter($request->transfer, function($qty) { return $qty > 0; });
            if (!empty($transferItems)) {
                $transferResult = $this->processProjectTransferPortion($request, $transaction, $transferItems);
                $results[] = $transferResult;
                $transferTotal = array_sum($transferItems);
            }
        }

        // Check if all items are fully fulfilled
        $fullyFulfilled = true;
        foreach ($transaction->items as $item) {
            $totalFulfilled = 0;

            // Calculate total fulfilled for this item across all methods
            if (isset($inventoryItems[$item->item_id])) {
                $totalFulfilled += $inventoryItems[$item->item_id];
            }
            if (isset($deliveryItems[$item->item_id])) {
                $totalFulfilled += $deliveryItems[$item->item_id];
            }
            if (isset($transferItems[$item->item_id])) {
                $totalFulfilled += $transferItems[$item->item_id];
            }

            // Check if this item is fully fulfilled
            if ($totalFulfilled < $item->quantity_requested) {
                $fullyFulfilled = false;
                break;
            }
        }

        // Update original transaction status based on issue progress
        if ($fullyFulfilled) {
            $transaction->update([
                'status' => 'processed_multi_source',
                'notes' => "Fully fulfilled via multi-source: Inventory({$inventoryTotal}), Delivery({$deliveryTotal}), Transfer({$transferTotal})"
            ]);
        } else {
            // Keep as pending if partially fulfilled
            $existingNotes = $transaction->notes ?? '';
            $newNote = "Partially issued: Inventory({$inventoryTotal}), Delivery({$deliveryTotal}), Transfer({$transferTotal})";
            $transaction->update([
                'status' => 'pending',
                'notes' => $existingNotes . "\n" . date('Y-m-d H:i') . " - " . $newNote
            ]);
        }

        // Update project inventory for ALL fulfilled items
        $this->updateProjectInventoryForMaterialIssue($transaction, $inventoryItems, $deliveryItems, $transferItems);

            // Create summary message
            $messages = [];
            if ($inventoryTotal > 0) $messages[] = "Inventory Dispatch: {$inventoryTotal} units processed";
            if ($deliveryTotal > 0) $messages[] = "Direct Delivery: {$deliveryTotal} units ordered";
            if ($transferTotal > 0) $messages[] = "Project Transfer: {$transferTotal} units requested";

            \DB::commit();

            \Log::info('Multi-source material issue completed successfully', [
                'transaction_id' => $transaction->id,
                'inventory_total' => $inventoryTotal,
                'delivery_total' => $deliveryTotal,
                'transfer_total' => $transferTotal
            ]);

            return redirect()->route('warehouse.material-issue.index')
                ->with('success', 'Multi-source material issue processed successfully! ' . implode(', ', $messages));

        } catch (\Exception $e) {
            \DB::rollback();
            \Log::error('Multi-source material issue failed', [
                'transaction_id' => $transaction->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return back()
                ->with('error', 'Failed to process multi-source material issue: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Process inventory portion of multi-source material issue
     */
    private function processInventoryPortion(Request $request, OutgoingTransaction $originalTransaction, array $inventoryItems)
    {
        // Create a new outgoing transaction for inventory portion
        $inventoryTransaction = OutgoingTransaction::create([
            'transaction_date' => $request->inventory_dispatch_date ?: now(),
            'project_id' => $originalTransaction->project_id,
            'division_id' => $originalTransaction->division_id,
            'material_request_number' => $originalTransaction->material_request_number . '-INV',
            'type_of_goods' => 'Material Request - Inventory',
            'requested_by' => $originalTransaction->requested_by,
            'receiver_name' => $request->inventory_receiver_name,
            'status' => 'processing',
            'notes' => $request->inventory_notes ?: 'Multi-source: Inventory portion'
        ]);

        // Create inventory transaction items
        foreach ($inventoryItems as $itemId => $quantity) {
            if ($quantity > 0) {
                $originalItem = $originalTransaction->items()->where('item_id', $itemId)->first();
                $averagePrice = Inventory::where('item_id', $itemId)
                    ->where('quantity_available', '>', 0)
                    ->avg('unit_price') ?: 0;

                OutgoingItem::create([
                    'outgoing_transaction_id' => $inventoryTransaction->id,
                    'item_id' => $itemId,
                    'quantity_requested' => $quantity,
                    'quantity_released' => $quantity,
                    'unit_price' => $averagePrice,
                ]);
            }
        }

        // Process the dispatch immediately
        $inventoryTransaction->approveAndDispatch(auth()->id());

        // Record issue records
        foreach ($inventoryItems as $itemId => $quantity) {
            if ($quantity > 0) {
                OutgoingFulfillment::create([
                    'outgoing_transaction_id' => $originalTransaction->id,
                    'item_id' => $itemId,
                    'quantity_fulfilled' => $quantity,
                    'fulfillment_type' => 'inventory',
                    'reference_id' => $inventoryTransaction->id,
                    'reference_type' => 'OutgoingTransaction',
                    'notes' => 'Inventory dispatch'
                ]);
            }
        }

        return ['type' => 'inventory', 'id' => $inventoryTransaction->id, 'total' => array_sum($inventoryItems)];
    }

    /**
     * Process direct delivery portion of multi-source material issue
     */
    private function processDirectDeliveryPortion(Request $request, OutgoingTransaction $originalTransaction, array $deliveryItems)
    {
        // Create Direct Delivery
        $directDelivery = DirectDelivery::create([
            'delivery_date' => $request->delivery_date ?: now()->addDays(7),
            'project_id' => $originalTransaction->project_id,
            'project_division_id' => $originalTransaction->division_id,
            'supplier_id' => $request->delivery_supplier_id,
            'purchase_order_number' => $request->delivery_po_number,
            'operation_type' => 'supplier_delivery',
            'delivery_type' => 'purchase_order',
            'workflow_status' => 'pending_supplier',
            'status' => 'pending',
            'material_request_id' => null,
            'material_request_number' => $originalTransaction->material_request_number,
            'outgoing_transaction_id' => $originalTransaction->id,
            'currency' => 'AED',
            'exchange_rate' => 1.0000,
            'vat_rate' => 5.00,
        ]);

        // Create direct delivery items
        foreach ($deliveryItems as $itemId => $quantity) {
            if ($quantity > 0) {
                DirectDeliveryItem::create([
                    'direct_delivery_id' => $directDelivery->id,
                    'item_id' => $itemId,
                    'quantity_ordered' => $quantity,
                    'ordered_quantity' => $quantity,
                    'currency' => 'AED',
                    'status' => 'pending',
                ]);

                // Create issue record for direct delivery
                OutgoingFulfillment::create([
                    'outgoing_transaction_id' => $originalTransaction->id,
                    'item_id' => $itemId,
                    'quantity_fulfilled' => $quantity,
                    'fulfillment_type' => 'direct_delivery',
                    'reference_id' => $directDelivery->id,
                    'reference_type' => 'DirectDelivery',
                    'notes' => 'Direct delivery issue'
                ]);
            }
        }

        return ['type' => 'delivery', 'id' => $directDelivery->id, 'total' => array_sum($deliveryItems)];
    }

    /**
     * Process project transfer portion of multi-source material issue
     */
    private function processProjectTransferPortion(Request $request, OutgoingTransaction $originalTransaction, array $transferItems)
    {
        // Parse the selected reference to get type and ID
        $referenceData = $this->parseTransferReference($request->transfer_reference_id ?? null);

        // Create Material Transfer Request
        $transferRequest = MaterialTransferRequest::create([
            'request_date' => now(),
            'transferrer_project_id' => $request->transfer_source_project_id,
            'receiver_project_id' => $originalTransaction->project_id,
            'transfer_reason' => $request->transfer_reason ?: 'Multi-source material issue',
            'transfer_notes' => $request->transfer_notes ?: "Part of multi-source material request. Reference: " . ($request->transfer_reference_id ?: 'N/A'),
            'priority' => $request->transfer_priority ?: 'medium',
            'required_date' => $request->transfer_date ?: now()->addDays(3),
            'requested_by' => auth()->id(),
            'status' => 'approved' // Auto-approve for immediate transfer since quantities are verified
        ]);

        // Create transfer request items
        foreach ($transferItems as $itemId => $quantity) {
            if ($quantity > 0) {
                // Check available quantity in source project
                $projectInventory = ProjectInventory::where('project_id', $request->transfer_source_project_id)
                    ->where('item_id', $itemId)
                    ->first();

                $availableQuantity = $projectInventory ? $projectInventory->available_for_transfer : 0;
                $estimatedUnitPrice = $projectInventory ? $projectInventory->unit_price : 0;

                // Link to the original reference based on type
                $linkingFields = $this->getLinkingFieldsForReference($referenceData, $itemId);

                MaterialTransferRequestItem::create([
                    'transfer_request_id' => $transferRequest->id,
                    'item_id' => $itemId,
                    'quantity_requested' => $quantity,
                    'quantity_approved' => $quantity, // Auto-approve the full quantity
                    'quantity_available' => $availableQuantity,
                    'estimated_unit_price' => $estimatedUnitPrice,
                    'transfer_notes' => "Multi-source material issue portion",
                    'approval_status' => 'approved'
                ] + $linkingFields);

                // CRITICAL: Decrease source project inventory immediately
                $this->decreaseSourceProjectInventory($request->transfer_source_project_id, $itemId, $quantity);

                // Create issue record for project transfer
                OutgoingFulfillment::create([
                    'outgoing_transaction_id' => $originalTransaction->id,
                    'item_id' => $itemId,
                    'quantity_fulfilled' => $quantity,
                    'fulfillment_type' => 'project_transfer',
                    'reference_id' => $transferRequest->id,
                    'reference_type' => 'MaterialTransferRequest',
                    'notes' => 'Project transfer issue'
                ]);
            }
        }

        return ['type' => 'transfer', 'id' => $transferRequest->id, 'total' => array_sum($transferItems)];
    }

    /**
     * Parse transfer reference ID to get type and ID
     */
    private function parseTransferReference($referenceId)
    {
        if (!$referenceId) {
            return ['type' => null, 'id' => null];
        }

        $parts = explode('_', $referenceId);
        if (count($parts) == 2) {
            return ['type' => $parts[0], 'id' => $parts[1]];
        }

        return ['type' => null, 'id' => null];
    }

    /**
     * Get linking fields for reference based on type
     */
    private function getLinkingFieldsForReference($referenceData, $itemId)
    {
        $fields = [];

        if (!$referenceData['type'] || !$referenceData['id']) {
            return $fields;
        }

        switch ($referenceData['type']) {
            case 'MR':
                // Link to Material Request (Outgoing Transaction)
                $outgoingItem = \App\Models\Warehouse\OutgoingItem::where('outgoing_transaction_id', $referenceData['id'])
                    ->where('item_id', $itemId)
                    ->first();
                if ($outgoingItem) {
                    $fields['outgoing_item_id'] = $outgoingItem->id;
                }
                break;

            case 'DD':
                // Link to Direct Delivery
                $deliveryItem = \App\Models\DirectDeliveryItem::where('direct_delivery_id', $referenceData['id'])
                    ->where('item_id', $itemId)
                    ->first();
                if ($deliveryItem) {
                    $fields['direct_delivery_item_id'] = $deliveryItem->id;
                }
                break;

            case 'MT':
                // Link to Material Transfer
                $transferItem = \App\Models\Warehouse\MaterialTransferRequestItem::where('transfer_request_id', $referenceData['id'])
                    ->where('item_id', $itemId)
                    ->first();
                if ($transferItem) {
                    $fields['material_transfer_item_id'] = $transferItem->id;
                }
                break;
        }

        return $fields;
    }

    /**
     * Decrease source project inventory
     */
    private function decreaseSourceProjectInventory($sourceProjectId, $itemId, $quantity)
    {
        $projectInventory = \App\Models\Warehouse\ProjectInventory::where('project_id', $sourceProjectId)
            ->where('item_id', $itemId)
            ->first();

        if ($projectInventory) {
            // Decrease available quantity
            $projectInventory->updateQuantity($quantity, 'subtract');

            \Log::info("Decreased source project inventory", [
                'project_id' => $sourceProjectId,
                'item_id' => $itemId,
                'quantity_decreased' => $quantity,
                'new_available' => $projectInventory->quantity_available
            ]);
        } else {
            \Log::warning("No project inventory found to decrease", [
                'project_id' => $sourceProjectId,
                'item_id' => $itemId,
                'quantity' => $quantity
            ]);
        }
    }

    /**
     * Update project inventory for all issued items across all issue methods
     */
    private function updateProjectInventoryForMaterialIssue(OutgoingTransaction $transaction, array $inventoryItems, array $deliveryItems, array $transferItems)
    {
        if (!$transaction->project_id) {
            return; // No project to update inventory for
        }

        // Combine all issued quantities by item
        $allIssuedItems = [];

        // Add inventory dispatch quantities
        foreach ($inventoryItems as $itemId => $quantity) {
            $allIssuedItems[$itemId] = ($allIssuedItems[$itemId] ?? 0) + $quantity;
        }

        // Add direct delivery quantities
        foreach ($deliveryItems as $itemId => $quantity) {
            $allIssuedItems[$itemId] = ($allIssuedItems[$itemId] ?? 0) + $quantity;
        }

        // Add project transfer quantities (these go to project inventory when transfer is completed)
        foreach ($transferItems as $itemId => $quantity) {
            $allIssuedItems[$itemId] = ($allIssuedItems[$itemId] ?? 0) + $quantity;
        }

        // Update project inventory for each issued item
        foreach ($allIssuedItems as $itemId => $totalIssued) {
            if ($totalIssued > 0) {
                $projectInventory = ProjectInventory::where('project_id', $transaction->project_id)
                    ->where('item_id', $itemId)
                    ->first();

                if ($projectInventory) {
                    // Update existing record
                    $projectInventory->updateQuantity($totalIssued, 'add');
                } else {
                    // Create new record - get unit price from original request
                    $originalItem = $transaction->items()->where('item_id', $itemId)->first();
                    $unitPrice = $originalItem->unit_price ?? 0;

                    ProjectInventory::create([
                        'project_id' => $transaction->project_id,
                        'item_id' => $itemId,
                        'quantity_available' => $totalIssued,
                        'allocated_quantity' => 0,
                        'unit_price' => $unitPrice,
                        'total_value' => $totalIssued * $unitPrice
                    ]);
                }
            }
        }
    }
}