<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserController extends Controller
{
    /**
     * Display a listing of warehouse users.
     */
    public function index(Request $request)
    {
        if (!auth()->user()->hasPermission('warehouse.users.view')) {
            abort(403, 'Unauthorized');
        }

        if ($request->ajax()) {
            $query = User::with('role');

            // Apply search filter
            if ($request->has('search') && $request->search['value']) {
                $search = $request->search['value'];
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%")
                      ->orWhere('employee_id', 'like', "%{$search}%")
                      ->orWhere('department', 'like', "%{$search}%");
                });
            }

            // Apply role filter
            if ($request->filled('role_filter')) {
                $query->where('role_id', $request->role_filter);
            }

            // Apply status filter
            if ($request->filled('status_filter')) {
                $query->where('status', $request->status_filter);
            }

            $totalRecords = $query->count();

            // Apply ordering
            if ($request->has('order')) {
                $columnIndex = $request->order[0]['column'];
                $columns = ['id', 'name', 'email', 'department', 'role_id', 'status', 'created_at'];
                $columnName = $columns[$columnIndex] ?? 'created_at';
                $direction = $request->order[0]['dir'] === 'asc' ? 'asc' : 'desc';
                $query->orderBy($columnName, $direction);
            }

            // Apply pagination
            $start = $request->start ?? 0;
            $length = $request->length ?? 25;
            $users = $query->skip($start)->take($length)->get();

            $data = $users->map(function ($user) {
                return [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'employee_id' => $user->employee_id,
                    'department' => $user->department,
                    'role' => $user->role->display_name ?? 'No Role',
                    'status' => $user->status,
                    'last_login_at' => $user->last_login_at ? $user->last_login_at->format('M d, Y H:i') : 'Never',
                    'created_at' => $user->created_at->format('M d, Y'),
                ];
            });

            return response()->json([
                'draw' => $request->draw,
                'recordsTotal' => User::count(),
                'recordsFiltered' => $totalRecords,
                'data' => $data
            ]);
        }

        $roles = Role::where('name', 'like', 'warehouse_%')->active()->get();

        return view('warehouse.users.index', compact('roles'));
    }

    /**
     * Store a newly created user.
     */
    public function store(Request $request)
    {
        if (!auth()->user()->hasPermission('warehouse.users.create')) {
            abort(403, 'Unauthorized');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'employee_id' => 'nullable|string|max:50|unique:users',
            'department' => 'required|string|max:100',
            'phone' => 'nullable|string|max:20',
            'role_id' => 'required|exists:roles,id',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = User::create([
                'name' => $request->name,
                'email' => $request->email,
                'employee_id' => $request->employee_id,
                'department' => $request->department,
                'phone' => $request->phone,
                'role_id' => $request->role_id,
                'password' => Hash::make($request->password),
                'status' => 'active',
                'created_by_admin_id' => auth()->id(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'User created successfully',
                'user' => $user->load('role')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating user: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified user.
     */
    public function update(Request $request, User $user)
    {
        if (!auth()->user()->hasPermission('warehouse.users.edit')) {
            abort(403, 'Unauthorized');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => ['required', 'string', 'email', 'max:255', Rule::unique('users')->ignore($user->id)],
            'employee_id' => ['nullable', 'string', 'max:50', Rule::unique('users')->ignore($user->id)],
            'department' => 'required|string|max:100',
            'phone' => 'nullable|string|max:20',
            'role_id' => 'required|exists:roles,id',
            'status' => 'required|in:active,inactive',
            'password' => 'nullable|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $updateData = [
                'name' => $request->name,
                'email' => $request->email,
                'employee_id' => $request->employee_id,
                'department' => $request->department,
                'phone' => $request->phone,
                'role_id' => $request->role_id,
                'status' => $request->status,
            ];

            if ($request->filled('password')) {
                $updateData['password'] = Hash::make($request->password);
            }

            $user->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'User updated successfully',
                'user' => $user->load('role')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating user: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified user.
     */
    public function destroy(User $user)
    {
        if (!auth()->user()->hasPermission('warehouse.users.delete')) {
            abort(403, 'Unauthorized');
        }

        try {
            // Prevent self-deletion
            if ($user->id === auth()->id()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete your own account'
                ], 400);
            }

            $user->delete();

            return response()->json([
                'success' => true,
                'message' => 'User deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting user: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user details for editing.
     */
    public function show(User $user)
    {
        if (!auth()->user()->hasPermission('warehouse.users.view')) {
            abort(403, 'Unauthorized');
        }

        return response()->json([
            'success' => true,
            'user' => $user->load(['role.permissions'])
        ]);
    }

    /**
     * Update user role.
     */
    public function updateRole(Request $request, User $user)
    {
        if (!auth()->user()->hasPermission('warehouse.users.roles')) {
            abort(403, 'Unauthorized');
        }

        $validator = Validator::make($request->all(), [
            'role_id' => 'required|exists:roles,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user->update(['role_id' => $request->role_id]);

            return response()->json([
                'success' => true,
                'message' => 'User role updated successfully',
                'user' => $user->load('role')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating user role: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get warehouse statistics for users.
     */
    public function stats()
    {
        $warehouseRoleIds = Role::where('name', 'like', 'warehouse_%')->pluck('id');

        return response()->json([
            'total_users' => User::whereIn('role_id', $warehouseRoleIds)->count(),
            'active_users' => User::whereIn('role_id', $warehouseRoleIds)->where('status', 'active')->count(),
            'inactive_users' => User::whereIn('role_id', $warehouseRoleIds)->where('status', 'inactive')->count(),
            'roles' => Role::where('name', 'like', 'warehouse_%')->withCount('users')->get(),
        ]);
    }
}
