<?php

namespace App\Repositories\Warehouse;

use App\Repositories\BaseRepository;
use App\Models\Warehouse\Item;
use Illuminate\Database\Eloquent\Builder;

class ItemRepository extends BaseRepository
{
    /**
     * Create a new repository instance.
     */
    public function __construct(Item $model)
    {
        $this->model = $model;
    }

    /**
     * Get items below reorder point.
     */
    public function getItemsBelowReorderPoint()
    {
        return $this->model->where('available_quantity', '<=', 'reorder_point')
                          ->where('reorder_point', '>', 0)
                          ->where('status', 'active')
                          ->with('category')
                          ->get();
    }

    /**
     * Get items by category with filters.
     */
    public function getByCategory($categoryId, array $filters = [])
    {
        $query = $this->model->where('category_id', $categoryId);
        return $this->applyFilters($query, $filters)->get();
    }

    /**
     * Search items by barcode or item code.
     */
    public function findByCode(string $code): ?Item
    {
        return $this->model->where('item_code', $code)
                          ->orWhere('sku', $code)
                          ->first();
    }

    /**
     * Get items with low stock levels.
     */
    public function getLowStockItems(int $threshold = null)
    {
        $query = $this->model->with('category');

        if ($threshold) {
            $query->where('available_quantity', '<=', $threshold);
        } else {
            $query->whereColumn('available_quantity', '<=', 'reorder_point')
                  ->where('reorder_point', '>', 0);
        }

        return $query->where('status', 'active')->get();
    }

    /**
     * Get items by location.
     */
    public function getByLocation(string $location)
    {
        return $this->model->where('location', $location)
                          ->where('status', 'active')
                          ->with('category')
                          ->get();
    }

    /**
     * Apply item-specific search filters.
     */
    protected function applySearchFilter(Builder $query, $search)
    {
        return $query->where(function ($q) use ($search) {
            $q->where('name', 'LIKE', "%{$search}%")
              ->orWhere('item_code', 'LIKE', "%{$search}%")
              ->orWhere('sku', 'LIKE', "%{$search}%")
              ->orWhere('description', 'LIKE', "%{$search}%");
        });
    }

    /**
     * Apply item-specific filters.
     */
    protected function applyFilters(Builder $query, array $filters)
    {
        foreach ($filters as $key => $value) {
            if ($value === null || $value === '') {
                continue;
            }

            switch ($key) {
                case 'category_id':
                    $query->where('category_id', $value);
                    break;
                case 'status':
                    $query->where('status', $value);
                    break;
                case 'location':
                    $query->where('location', $value);
                    break;
                case 'low_stock':
                    if ($value) {
                        $query->whereColumn('available_quantity', '<=', 'reorder_point')
                              ->where('reorder_point', '>', 0);
                    }
                    break;
                case 'out_of_stock':
                    if ($value) {
                        $query->where('available_quantity', 0);
                    }
                    break;
                case 'price_min':
                    $query->where('unit_cost', '>=', $value);
                    break;
                case 'price_max':
                    $query->where('unit_cost', '<=', $value);
                    break;
                case 'created_from':
                    $query->where('created_at', '>=', $value);
                    break;
                case 'created_to':
                    $query->where('created_at', '<=', $value);
                    break;
                case 'search':
                    $this->applySearchFilter($query, $value);
                    break;
            }
        }

        return $query;
    }

    /**
     * Get item statistics.
     */
    public function getStatistics()
    {
        return [
            'total_items' => $this->model->count(),
            'active_items' => $this->model->where('status', 'active')->count(),
            'low_stock_items' => $this->getLowStockItems()->count(),
            'out_of_stock_items' => $this->model->where('available_quantity', 0)->count(),
            'total_value' => $this->model->selectRaw('SUM(available_quantity * unit_cost) as total_value')->value('total_value') ?? 0,
        ];
    }
}