<?php

namespace App\Services\Warehouse;

use App\Services\BaseService;
use App\Repositories\Warehouse\ItemRepository;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\ItemCategory;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Picqer\Barcode\BarcodeGeneratorPNG;

class ItemService extends BaseService
{
    /**
     * Create a new service instance.
     */
    public function __construct(ItemRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Create a new item with automatic code generation.
     */
    public function create(array $data)
    {
        // Generate item code if not provided
        if (!isset($data['item_code'])) {
            $data['item_code'] = $this->generateItemCode($data['category_id'] ?? null);
        }

        // Generate SKU if not provided
        if (!isset($data['sku'])) {
            $data['sku'] = $this->generateSKU($data);
        }

        // Set initial quantities
        $data['total_quantity'] = 0;
        $data['available_quantity'] = 0;
        $data['reserved_quantity'] = 0;

        // Validate business rules
        $this->validateItemCreation($data);

        return parent::create($data);
    }

    /**
     * Update item with validation.
     */
    public function update($id, array $data)
    {
        $item = $this->findById($id);
        if (!$item) {
            throw new \Exception("Item not found with ID: {$id}");
        }

        // Don't allow changing item code if item has stock
        if (isset($data['item_code']) && $data['item_code'] !== $item->item_code) {
            if ($item->total_quantity > 0) {
                throw new \Exception("Cannot change item code for items with existing stock.");
            }
        }

        // Validate business rules
        $this->validateItemUpdate($data, $item);

        return parent::update($id, $data);
    }

    /**
     * Generate unique item code.
     */
    public function generateItemCode($categoryId = null): string
    {
        $prefix = 'ITM';

        if ($categoryId) {
            $category = ItemCategory::find($categoryId);
            if ($category && $category->code) {
                $prefix = strtoupper($category->code);
            }
        }

        $lastItem = Item::where('item_code', 'LIKE', "{$prefix}%")
                       ->orderBy('item_code', 'desc')
                       ->first();

        if ($lastItem) {
            $lastNumber = (int) substr($lastItem->item_code, strlen($prefix));
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }

        return $prefix . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Generate SKU.
     */
    public function generateSKU(array $data): string
    {
        $sku = '';

        // Add category prefix
        if (isset($data['category_id'])) {
            $category = ItemCategory::find($data['category_id']);
            if ($category) {
                $sku .= strtoupper(substr($category->name, 0, 3));
            }
        }

        // Add name prefix
        if (isset($data['name'])) {
            $sku .= strtoupper(substr(str_replace(' ', '', $data['name']), 0, 3));
        }

        // Add random suffix
        $sku .= strtoupper(Str::random(4));

        return $sku;
    }

    /**
     * Generate barcode for item.
     */
    public function generateBarcode(Item $item): string
    {
        $generator = new BarcodeGeneratorPNG();
        $barcode = $generator->getBarcode($item->item_code, $generator::TYPE_CODE_128);

        $filename = "barcodes/{$item->item_code}.png";
        Storage::disk('public')->put($filename, $barcode);

        return Storage::disk('public')->url($filename);
    }

    /**
     * Generate QR code for item.
     */
    public function generateQrCode(Item $item): string
    {
        $qrData = [
            'item_code' => $item->item_code,
            'name' => $item->name,
            'sku' => $item->sku,
            'url' => route('warehouse.items.show', $item->id)
        ];

        $qrCode = QrCode::format('png')
                       ->size(200)
                       ->generate(json_encode($qrData));

        $filename = "qrcodes/{$item->item_code}.png";
        Storage::disk('public')->put($filename, $qrCode);

        return Storage::disk('public')->url($filename);
    }

    /**
     * Search items by barcode.
     */
    public function findByBarcode(string $barcode): ?Item
    {
        return $this->repository->findBy('item_code', $barcode);
    }

    /**
     * Get items below reorder point.
     */
    public function getItemsBelowReorderPoint()
    {
        return $this->repository->getItemsBelowReorderPoint();
    }

    /**
     * Get items by category.
     */
    public function getItemsByCategory($categoryId, array $filters = [])
    {
        $filters['category_id'] = $categoryId;
        return $this->repository->getAll($filters);
    }

    /**
     * Update item stock levels.
     */
    public function updateStockLevels(Item $item)
    {
        $inventoryTotal = $item->inventories()->sum('quantity_available');
        $reservedTotal = $item->inventories()->sum('quantity_reserved');

        $item->update([
            'total_quantity' => $inventoryTotal,
            'available_quantity' => $inventoryTotal - $reservedTotal,
            'reserved_quantity' => $reservedTotal
        ]);

        return $item->fresh();
    }

    /**
     * Check item availability.
     */
    public function checkAvailability(Item $item, int $quantity): bool
    {
        return $item->available_quantity >= $quantity;
    }

    /**
     * Reserve item quantity.
     */
    public function reserveQuantity(Item $item, int $quantity, string $reason = ''): bool
    {
        if (!$this->checkAvailability($item, $quantity)) {
            throw new \Exception("Insufficient available quantity. Available: {$item->available_quantity}, Requested: {$quantity}");
        }

        // Update item quantities
        $item->increment('reserved_quantity', $quantity);
        $item->decrement('available_quantity', $quantity);

        // Log the reservation
        $this->logActivity('reserved', $item, [
            'quantity' => $quantity,
            'reason' => $reason
        ]);

        return true;
    }

    /**
     * Release reserved quantity.
     */
    public function releaseReservedQuantity(Item $item, int $quantity, string $reason = ''): bool
    {
        if ($item->reserved_quantity < $quantity) {
            throw new \Exception("Cannot release more than reserved quantity. Reserved: {$item->reserved_quantity}, Requested: {$quantity}");
        }

        // Update item quantities
        $item->decrement('reserved_quantity', $quantity);
        $item->increment('available_quantity', $quantity);

        // Log the release
        $this->logActivity('released', $item, [
            'quantity' => $quantity,
            'reason' => $reason
        ]);

        return true;
    }

    /**
     * Bulk import items from array.
     */
    public function bulkImport(array $itemsData): array
    {
        $results = [
            'success' => 0,
            'errors' => [],
            'created_items' => []
        ];

        foreach ($itemsData as $index => $itemData) {
            try {
                // Validate required fields
                if (!isset($itemData['name']) || empty($itemData['name'])) {
                    $results['errors'][] = "Row {$index}: Name is required";
                    continue;
                }

                // Create item
                $item = $this->create($itemData);
                $results['created_items'][] = $item;
                $results['success']++;

            } catch (\Exception $e) {
                $results['errors'][] = "Row {$index}: " . $e->getMessage();
            }
        }

        return $results;
    }

    /**
     * Export items to array.
     */
    public function exportItems(array $filters = []): array
    {
        $items = $this->getAll($filters, ['category', 'inventories']);

        return $items->map(function ($item) {
            return [
                'item_code' => $item->item_code,
                'name' => $item->name,
                'description' => $item->description,
                'sku' => $item->sku,
                'category' => $item->category?->name,
                'unit' => $item->unit,
                'unit_cost' => $item->unit_cost,
                'selling_price' => $item->selling_price,
                'total_quantity' => $item->total_quantity,
                'available_quantity' => $item->available_quantity,
                'reserved_quantity' => $item->reserved_quantity,
                'reorder_point' => $item->reorder_point,
                'max_stock_level' => $item->max_stock_level,
                'location' => $item->location,
                'status' => $item->status,
                'created_at' => $item->created_at?->format('Y-m-d H:i:s'),
            ];
        })->toArray();
    }

    /**
     * Validate item creation.
     */
    protected function validateItemCreation(array $data)
    {
        // Check for duplicate item code
        if (isset($data['item_code'])) {
            $existingItem = $this->repository->findBy('item_code', $data['item_code']);
            if ($existingItem) {
                throw new \Exception("Item code '{$data['item_code']}' already exists.");
            }
        }

        // Check for duplicate SKU
        if (isset($data['sku'])) {
            $existingItem = $this->repository->findBy('sku', $data['sku']);
            if ($existingItem) {
                throw new \Exception("SKU '{$data['sku']}' already exists.");
            }
        }

        // Validate category exists
        if (isset($data['category_id'])) {
            $category = ItemCategory::find($data['category_id']);
            if (!$category) {
                throw new \Exception("Category not found.");
            }
        }

        // Validate numeric values
        if (isset($data['unit_cost']) && $data['unit_cost'] < 0) {
            throw new \Exception("Unit cost cannot be negative.");
        }

        if (isset($data['selling_price']) && $data['selling_price'] < 0) {
            throw new \Exception("Selling price cannot be negative.");
        }
    }

    /**
     * Validate item update.
     */
    protected function validateItemUpdate(array $data, Item $item)
    {
        // Check for duplicate item code (excluding current item)
        if (isset($data['item_code']) && $data['item_code'] !== $item->item_code) {
            $existingItem = $this->repository->findBy('item_code', $data['item_code']);
            if ($existingItem && $existingItem->id !== $item->id) {
                throw new \Exception("Item code '{$data['item_code']}' already exists.");
            }
        }

        // Check for duplicate SKU (excluding current item)
        if (isset($data['sku']) && $data['sku'] !== $item->sku) {
            $existingItem = $this->repository->findBy('sku', $data['sku']);
            if ($existingItem && $existingItem->id !== $item->id) {
                throw new \Exception("SKU '{$data['sku']}' already exists.");
            }
        }

        // Other validation rules...
        $this->validateItemCreation($data);
    }
}