#!/bin/bash

# Database backup script for Docker MySQL container
# Usage: ./backup-database.sh

# Configuration
CONTAINER_NAME="aufg_mysql"
DATABASE_NAME="aufg_laravel"
USERNAME="laravel"
PASSWORD="laravel123"
BACKUP_DIR="./docker/mysql/backup"
TIMESTAMP=$(date +"%Y%m%d_%H%M%S")
BACKUP_FILE="${BACKUP_DIR}/aufg_laravel_${TIMESTAMP}.sql"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

echo -e "${YELLOW}Starting database backup...${NC}"

# Check if Docker container is running
if ! docker ps | grep -q "$CONTAINER_NAME"; then
    echo -e "${RED}Error: MySQL container '$CONTAINER_NAME' is not running!${NC}"
    echo "Please start the container with: docker-compose up -d mysql"
    exit 1
fi

# Create backup directory if it doesn't exist
mkdir -p "$BACKUP_DIR"

# Create database backup
echo -e "${YELLOW}Creating backup of database '$DATABASE_NAME'...${NC}"

docker exec "$CONTAINER_NAME" mysqldump \
    -u"$USERNAME" \
    -p"$PASSWORD" \
    --single-transaction \
    --routines \
    --triggers \
    --no-tablespaces \
    --databases "$DATABASE_NAME" > "$BACKUP_FILE"

# Check if backup was successful
if [ $? -eq 0 ]; then
    echo -e "${GREEN}✓ Backup completed successfully!${NC}"
    echo -e "${GREEN}✓ Backup saved to: $BACKUP_FILE${NC}"
    
    # Display backup file size
    BACKUP_SIZE=$(du -h "$BACKUP_FILE" | cut -f1)
    echo -e "${GREEN}✓ Backup size: $BACKUP_SIZE${NC}"
    
    # Compress backup (optional)
    echo -e "${YELLOW}Compressing backup...${NC}"
    gzip "$BACKUP_FILE"
    COMPRESSED_FILE="${BACKUP_FILE}.gz"
    COMPRESSED_SIZE=$(du -h "$COMPRESSED_FILE" | cut -f1)
    echo -e "${GREEN}✓ Compressed backup: ${COMPRESSED_FILE}${NC}"
    echo -e "${GREEN}✓ Compressed size: $COMPRESSED_SIZE${NC}"
    
else
    echo -e "${RED}✗ Backup failed!${NC}"
    exit 1
fi

echo -e "${GREEN}Database backup completed!${NC}"
echo ""
echo "To restore this backup:"
echo "  1. Copy the backup file to Docker container:"
echo "     docker cp ${COMPRESSED_FILE} ${CONTAINER_NAME}:/tmp/"
echo "  2. Restore the database:"
echo "     docker exec ${CONTAINER_NAME} gunzip -c /tmp/$(basename ${COMPRESSED_FILE}) | mysql -u${USERNAME} -p${PASSWORD}"
echo ""
echo "Or use the restore script: ./restore-database.sh $(basename ${COMPRESSED_FILE})"