<?php

require_once 'bootstrap/app.php';

use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectInventory;
use App\Models\Warehouse\OutgoingItem;
use App\Models\DirectDeliveryItem;
use App\Models\Warehouse\IncomingOperationItem;
use Illuminate\Support\Facades\DB;

echo "Starting Project Inventory Fix...\n";

// Get all projects that have inventory
$projects = Project::whereExists(function ($query) {
    $query->select(DB::raw(1))
        ->from('project_inventory')
        ->whereColumn('project_inventory.project_id', 'projects.id')
        ->where('project_inventory.quantity_available', '>', 0);
})->get();

foreach ($projects as $project) {
    echo "\nProcessing Project: {$project->project_name} (ID: {$project->id})\n";

    // Get all unique items for this project
    $itemIds = ProjectInventory::where('project_id', $project->id)
        ->where('quantity_available', '>', 0)
        ->pluck('item_id')
        ->unique();

    foreach ($itemIds as $itemId) {
        $projectInventory = ProjectInventory::where('project_id', $project->id)
            ->where('item_id', $itemId)
            ->first();

        if (!$projectInventory) continue;

        $item = $projectInventory->item;
        echo "  Item: {$item->item_description} (ID: {$itemId})\n";

        // Calculate total delivered from material requests
        $materialDispatched = OutgoingItem::whereHas('transaction', function($q) use ($project) {
            $q->where('project_id', $project->id)
              ->where('status', 'successful');
        })
        ->where('item_id', $itemId)
        ->sum('quantity_released');

        // Calculate total returned from material requests
        $materialReturned = IncomingOperationItem::join('incoming_operations', 'incoming_operation_items.incoming_operation_id', '=', 'incoming_operations.id')
            ->join('outgoing_items', 'incoming_operation_items.outgoing_item_id', '=', 'outgoing_items.id')
            ->join('outgoing_transactions', 'outgoing_items.outgoing_transaction_id', '=', 'outgoing_transactions.id')
            ->where('incoming_operations.operation_type', 'site_return')
            ->where('incoming_operations.status', 'completed')
            ->where('outgoing_transactions.project_id', $project->id)
            ->where('incoming_operation_items.item_id', $itemId)
            ->sum('incoming_operation_items.quantity_accepted');

        // Calculate total delivered directly
        $directDelivered = DirectDeliveryItem::whereHas('directDelivery', function($q) use ($project) {
            $q->where('project_id', $project->id)
              ->whereIn('status', ['received', 'delivered']);
        })
        ->where('item_id', $itemId)
        ->sum('delivered_quantity');

        // Calculate total returned from direct deliveries
        $directReturned = DirectDeliveryItem::whereHas('directDelivery', function($q) use ($project) {
            $q->where('project_id', $project->id)
              ->whereIn('status', ['received', 'delivered']);
        })
        ->where('item_id', $itemId)
        ->sum('returned_quantity');

        $totalDelivered = $materialDispatched + $directDelivered;
        $totalReturned = $materialReturned + $directReturned;
        $correctStock = max(0, $totalDelivered - $totalReturned);

        echo "    Material Dispatched: {$materialDispatched}\n";
        echo "    Material Returned: {$materialReturned}\n";
        echo "    Direct Delivered: {$directDelivered}\n";
        echo "    Direct Returned: {$directReturned}\n";
        echo "    Total Delivered: {$totalDelivered}\n";
        echo "    Total Returned: {$totalReturned}\n";
        echo "    Current in DB: {$projectInventory->quantity_available}\n";
        echo "    Correct Stock: {$correctStock}\n";

        if ($projectInventory->quantity_available != $correctStock) {
            $projectInventory->quantity_available = $correctStock;
            $projectInventory->save();
            echo "    ✓ UPDATED to correct value\n";
        } else {
            echo "    ✓ Already correct\n";
        }
    }
}

echo "\nProject Inventory Fix Complete!\n";