/**
 * Admin Custom JavaScript
 * Custom functionality for the admin panel
 * Extracted from admin-simple.blade.php for better maintainability
 */

// Ensure preloader is hidden and layout is initialized
document.addEventListener('DOMContentLoaded', function() {
    // Hide preloader after short delay
    setTimeout(function() {
        document.body.classList.add('loaded');
        const preloader = document.querySelector('.preloader');
        if (preloader) {
            preloader.style.display = 'none';
        }
    }, 1000);

    // Force initialize MDK layout if it hasn't been initialized
    setTimeout(function() {
        try {
            if (typeof domFactory !== 'undefined') {
                domFactory.handler.upgrade();
            }
        } catch (error) {
            console.warn('MDK initialization warning:', error);
        }
    }, 1500);
});

$(document).ready(function() {
    // Prevent interference with Bootstrap navbar dropdowns
    $('.navbar [data-toggle="dropdown"]').off('click.sidebar');

    // Additional dropdown fixes
    console.log('🔧 Admin custom: Initializing dropdown functionality...');

    // Ensure dropdown toggle works properly - remove duplicate handlers
    $('#navbarDropdown').off('click').on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        console.log('🖱️ Admin-custom: Dropdown clicked');

        const $menu = $(this).next('.dropdown-menu');
        const isVisible = $menu.is(':visible');

        // Close all other dropdowns first
        $('.dropdown-menu').not($menu).hide().removeClass('show');

        // Toggle this dropdown
        if (!isVisible) {
            // Use CSS positioning instead of JavaScript positioning
            $menu.addClass('show');
            console.log('✅ Admin-custom: Dropdown opened with CSS positioning');
            console.log('Dropdown menu offset:', $menu.offset());
            console.log('Dropdown menu CSS:', {
                display: $menu.css('display'),
                position: $menu.css('position'),
                top: $menu.css('top'),
                right: $menu.css('right'),
                zIndex: $menu.css('z-index')
            });
        } else {
            $menu.removeClass('show');
            console.log('✅ Admin-custom: Dropdown closed');
        }
    });

    // Close dropdown when clicking outside
    $(document).on('click', function(e) {
        if (!$(e.target).closest('.dropdown').length) {
            $('.dropdown-menu').hide().removeClass('show');
            console.log('✅ Admin-custom: Dropdown closed (clicked outside)');
        }
    });

    // Handle dropdown item clicks
    $(document).on('click', '.dropdown-item', function(e) {
        // Only close dropdown for regular links, not form buttons
        if (!$(this).is('button[type="submit"]') && !$(this).hasClass('no-close')) {
            setTimeout(() => {
                $('.dropdown-menu').hide().removeClass('show');
                console.log('✅ Admin-custom: Dropdown closed (item clicked)');
            }, 100);
        }
    });
    // ============================
    // PERFECT SCROLLBAR INITIALIZATION
    // ============================
    
    // Make PerfectScrollbar available globally if needed
    window.PerfectScrollbar = PerfectScrollbar;
    
    // Initialize PerfectScrollbar if not already initialized by app.js
    const scrollContainers = document.querySelectorAll('[data-perfect-scrollbar]');
    scrollContainers.forEach(container => {
        if (!container.classList.contains('ps')) {
            new PerfectScrollbar(container, {
                wheelSpeed: 2,
                wheelPropagation: true,
                minScrollbarLength: 20
            });
        }
    });

    // ============================
    // CUSTOM PROFILE DROPDOWN
    // ============================

    let dropdownVisible = false;
    const $profileDropdown = $('.profile-dropdown');
    const $profileToggle = $('#profileDropdown');
    const $profileMenu = $('#profileMenu');

    // Toggle dropdown on click
    $profileToggle.on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        console.log('Profile dropdown clicked');

        if (dropdownVisible) {
            console.log('Hiding dropdown');
            $profileMenu.hide();
            $profileDropdown.removeClass('active');
            dropdownVisible = false;
        } else {
            console.log('Showing dropdown');
            $profileMenu.show();
            $profileDropdown.addClass('active');
            dropdownVisible = true;
        }

        console.log('Dropdown visible state:', dropdownVisible);

    });

    // Close dropdown when clicking outside
    $(document).on('click', function(e) {
        if (!$(e.target).closest('.profile-dropdown').length && dropdownVisible) {
            console.log('Clicking outside - closing dropdown');
            $profileMenu.hide();
            $profileDropdown.removeClass('active');
            dropdownVisible = false;
        }
    });

    // Prevent dropdown from closing when clicking inside
    $profileMenu.on('click', function(e) {
        e.stopPropagation();
    });

    // Close dropdown on escape key
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape' && dropdownVisible) {
            $profileMenu.hide();
            $profileDropdown.removeClass('active');
            dropdownVisible = false;
        }
    });
    
    // ============================
    // HEADER POSITIONING FIXES
    // ============================
    
    // Ensure header stays fixed
    const header = document.querySelector('.mdk-header[data-fixed]');
    if (header) {
        // Force fixed positioning
        header.style.position = 'fixed';
        header.style.top = '0';
        header.style.left = '0';
        header.style.right = '0';
        header.style.width = '100%';
        header.style.zIndex = '1030';
        
        // Ensure content has proper spacing
        const content = document.querySelector('.mdk-header-layout__content');
        if (content) {
            content.style.paddingTop = '80px';
        }
    }
    
    // ============================
    // SIDEBAR MENU FUNCTIONALITY
    // ============================

    // Initialize sidebar menu state
    $('.sidebar-submenu.show').each(function() {
        $(this).closest('.sidebar-menu-item').addClass('open');
    });

    // Handle sidebar menu collapsing - only target sidebar elements, not navbar dropdowns
    $('.sidebar .sidebar-menu-button[data-toggle="collapse"]').off('click.sidebar').on('click.sidebar', function(e) {
        e.preventDefault();
        e.stopPropagation();

        const target = $(this).attr('href');
        const $target = $(target);
        const $parent = $(this).closest('.sidebar-menu-item');

        console.log('Sidebar toggle clicked', target);

        // Check if target exists and has a valid href
        if (!target || target === '#' || $target.length === 0) {
            console.error('Collapse target not found or invalid:', target);
            return false;
        }

        // Toggle the submenu
        if ($target.hasClass('show')) {
            $target.removeClass('show');
            $parent.removeClass('open');
            console.log('Closing sidebar submenu:', target);
        } else {
            // Close other open submenus first
            $('.sidebar-submenu.show').not($target).removeClass('show');
            $('.sidebar-menu-item.open').not($parent).removeClass('open');

            // Open this submenu
            $target.addClass('show');
            $parent.addClass('open');
            console.log('Opening sidebar submenu:', target);
        }

        return false;
    });

    // Simple fallback using plain event delegation
    $(document).on('click', '.sidebar-menu-button[href="#items_menu"]', function(e) {
        e.preventDefault();
        e.stopPropagation();

        console.log('Fallback handler for items menu triggered');

        const $submenu = $('#items_menu');
        const $parent = $(this).closest('.sidebar-menu-item');

        if ($submenu.hasClass('show')) {
            $submenu.removeClass('show');
            $parent.removeClass('open');
            console.log('Closing items submenu');
        } else {
            $submenu.addClass('show');
            $parent.addClass('open');
            console.log('Opening items submenu');
        }
    });

    // Projects menu fallback handler
    $(document).on('click', '.sidebar-menu-button[href="#projects_menu"]', function(e) {
        e.preventDefault();
        e.stopPropagation();

        console.log('Fallback handler for projects menu triggered');

        const $submenu = $('#projects_menu');
        const $parent = $(this).closest('.sidebar-menu-item');

        if ($submenu.hasClass('show')) {
            $submenu.removeClass('show');
            $parent.removeClass('open');
            console.log('Closing projects submenu');
        } else {
            $submenu.addClass('show');
            $parent.addClass('open');
            console.log('Opening projects submenu');
        }
    });

    // Inventory menu fallback handler
    $(document).on('click', '.sidebar-menu-button[href="#inventory_menu"]', function(e) {
        e.preventDefault();
        e.stopPropagation();

        console.log('Fallback handler for inventory menu triggered');

        const $submenu = $('#inventory_menu');
        const $parent = $(this).closest('.sidebar-menu-item');

        if ($submenu.hasClass('show')) {
            $submenu.removeClass('show');
            $parent.removeClass('open');
            console.log('Closing inventory submenu');
        } else {
            $submenu.addClass('show');
            $parent.addClass('open');
            console.log('Opening inventory submenu');
        }
    });

    // Prevent submenu area clicks from bubbling up
    $(document).on('click', '.sidebar-submenu', function(e) {
        e.stopPropagation();
        console.log('Clicked in submenu area');
    });

    // Handle clicks on submenu links properly
    $(document).on('click', '.sidebar-submenu .sidebar-menu-button', function(e) {
        // Don't prevent default here as we want navigation to work
        e.stopPropagation();
        console.log('Clicked submenu link:', $(this).attr('href'));
    });
    
    // ============================
    // PAGE LOADING IMPLEMENTATION
    // ============================
    
    const $loadingOverlay = $('#pageLoadingOverlay');
    const $pageContainer = $('.page__container');
    
    // Show loading on navigation links
    function showPageLoading() {
        $loadingOverlay.addClass('show');
        $pageContainer.addClass('loading');
    }
    
    // Hide loading overlay
    function hidePageLoading() {
        $loadingOverlay.removeClass('show');
        $pageContainer.removeClass('loading');
    }
    
    // Handle navigation clicks
    $(document).on('click', 'a[href]:not([href^="#"]):not([href^="javascript:"]):not([href^="mailto:"]):not([href^="tel:"]):not([data-toggle]):not([onclick]):not(.no-loading)', function(e) {
        const href = $(this).attr('href');
        
        // Skip if it's the current page or external link
        if (href === window.location.pathname || 
            href.startsWith('http') && !href.includes(window.location.hostname)) {
            return;
        }
        
        // Skip if it's a form submission or AJAX request
        if ($(this).closest('form').length > 0) {
            return;
        }
        
        // Show loading for internal navigation
        showPageLoading();
        
        // Hide loading after a timeout as fallback
        setTimeout(hidePageLoading, 5000);
    });
    
    // Handle form submissions that navigate
    $(document).on('submit', 'form:not(.ajax-form)', function() {
        const action = $(this).attr('action');
        const method = $(this).attr('method') || 'GET';
        
        // Only show loading for GET forms or forms that navigate
        if (method.toLowerCase() === 'get' || $(this).hasClass('navigate-form')) {
            showPageLoading();
            setTimeout(hidePageLoading, 5000);
        }
    });
    
    // Handle browser back/forward buttons
    $(window).on('popstate', function() {
        showPageLoading();
        setTimeout(hidePageLoading, 3000);
    });
    
    // Hide loading when page is fully loaded
    $(window).on('load', function() {
        hidePageLoading();
    });
    
    // Hide loading on page show (back button)
    $(window).on('pageshow', function() {
        hidePageLoading();
    });
    
    // ============================
    // TABLE DROPDOWN POSITIONING
    // ============================
    
    // Fix dropdown positioning for table rows
    $('.dropdown-toggle').on('click', function() {
        const $dropdown = $(this).closest('.dropdown');
        const $menu = $dropdown.find('.dropdown-menu');
        const $table = $dropdown.closest('table');
        const $row = $dropdown.closest('tr');
        
        // Check if this is one of the last two rows
        const rowIndex = $row.index();
        const totalRows = $table.find('tbody tr').length;
        
        if (rowIndex >= totalRows - 2) {
            $dropdown.addClass('dropup');
        } else {
            $dropdown.removeClass('dropup');
        }
    });
    
    // ============================
    // CLEANUP
    // ============================
    
    // Hide any unwanted app settings
    $('#app-settings, .app-settings').remove();
});

// ============================
// GLOBAL UTILITY FUNCTIONS
// ============================

/**
 * Show toast notification
 * @param {string} message - The message to display
 * @param {string} type - Type of alert: 'success' or 'error'
 */
function showToastMessage(message, type = 'success') {
    const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
    const toast = `<div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
        ${message}
        <button type="button" class="close" data-dismiss="alert">&times;</button>
    </div>`;
    $('body').append(toast);
    setTimeout(() => $('.alert').alert('close'), 5000);
}

// ============================
// TIME PICKER PAST TIME BLOCKING
// ============================

/**
 * Block past times in time picker based on selected date
 */
function blockPastTimes() {
    const now = new Date();
    const currentHour = now.getHours();
    const currentMinute = now.getMinutes();
    const currentTime = currentHour * 60 + currentMinute; // Convert to minutes

    // Get today's date in YYYY-MM-DD format
    const today = now.toISOString().split('T')[0];

    // Process all time pickers
    document.querySelectorAll('.time-picker').forEach(function(timePicker) {
        // Find the associated date picker
        const dateInput = timePicker.closest('.form-group')?.querySelector('.date-picker') ||
                         timePicker.closest('.datetime-selection-row')?.querySelector('.date-picker');

        if (!dateInput) return;

        const selectedDate = dateInput.value;
        const isToday = selectedDate === today;

        // Process all time options
        Array.from(timePicker.options).forEach(function(option) {
            if (option.value === '') return; // Skip empty option

            // Parse time value (e.g., "14:30")
            const timeParts = option.value.split(':');
            const optionHour = parseInt(timeParts[0]);
            const optionMinute = parseInt(timeParts[1]);
            const optionTime = optionHour * 60 + optionMinute; // Convert to minutes

            // Disable if it's today and the time has passed
            if (isToday && optionTime <= currentTime) {
                option.disabled = true;
                option.style.color = '#ccc';
                option.style.backgroundColor = '#f5f5f5';
            } else {
                option.disabled = false;
                option.style.color = '';
                option.style.backgroundColor = '';
            }
        });

        // If current selection is now invalid, clear it
        const currentSelection = timePicker.value;
        if (currentSelection && isToday) {
            const currentParts = currentSelection.split(':');
            const currentHour = parseInt(currentParts[0]);
            const currentMinute = parseInt(currentParts[1]);
            const currentOptionTime = currentHour * 60 + currentMinute;

            if (currentOptionTime <= currentTime) {
                timePicker.value = '';
            }
        }
    });
}

/**
 * Initialize time blocking functionality
 */
function initializeTimeBlocking() {
    // Block past times on page load
    blockPastTimes();

    // Update time options when date changes
    document.addEventListener('change', function(e) {
        if (e.target.classList.contains('date-picker')) {
            blockPastTimes();
        }
    });

    // Update every minute to handle real-time blocking
    setInterval(blockPastTimes, 60000); // Update every minute
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(initializeTimeBlocking, 1000); // Delay to ensure elements are loaded
});

// Also initialize when jQuery is ready (fallback)
$(document).ready(function() {
    setTimeout(initializeTimeBlocking, 1000);
});