// Carousel Debug and Fix Script
document.addEventListener('DOMContentLoaded', function() {
    console.log('=== CAROUSEL FIX APPLYING ===');
    
    // Check if we're on mobile
    function isMobile() {
        return window.innerWidth <= 768;
    }
    
    // Skip complex carousel fixes on mobile
    if (isMobile()) {
        console.log('Mobile detected - applying simple mobile fixes only');
        
        // Apply basic mobile-friendly CSS
        const style = document.createElement('style');
        style.textContent = `
            @media (max-width: 768px) {
                .products-carousel {
                    display: flex !important;
                    flex-direction: column !important;
                    overflow: visible !important;
                    gap: 20px !important;
                }
                
                .products-carousel > * {
                    width: 100% !important;
                    opacity: 1 !important;
                    transform: none !important;
                }
                
                .carousel-nav {
                    display: none !important;
                }
            }
        `;
        document.head.appendChild(style);
        return;
    }
    
    // Add CSS fix for carousel overflow and fade effects
    const style = document.createElement('style');
    style.textContent = `
        /* Fix carousel overflow */
        .products-carousel {
            overflow-x: auto !important;
            overflow-y: hidden !important;
        }
        
        /* Ensure wrapper allows scrolling */
        .products-carousel-wrapper {
            overflow: visible !important;
        }
        
        /* Ensure products maintain their layout */
        .products-carousel > * {
            flex-shrink: 0 !important;
        }
        
        /* Product filter transitions */
        .product-card-clear-float,
        .product-card-silver-mirror, 
        .product-card-pattern-glass,
        .product-card-decorative-mirror {
            transition: opacity 0.6s cubic-bezier(0.4, 0, 0.2, 1), 
                        transform 0.6s cubic-bezier(0.4, 0, 0.2, 1),
                        order 0.6s cubic-bezier(0.4, 0, 0.2, 1) !important;
            opacity: 1;
            transform: scale(1);
        }
        
        .product-card-clear-float.aufg-hidden,
        .product-card-silver-mirror.aufg-hidden,
        .product-card-pattern-glass.aufg-hidden,
        .product-card-decorative-mirror.aufg-hidden {
            opacity: 0 !important;
            transform: scale(0.95) !important;
            pointer-events: none !important;
            order: 999 !important;
        }
        
        .product-card-clear-float.aufg-fade-in,
        .product-card-silver-mirror.aufg-fade-in,
        .product-card-pattern-glass.aufg-fade-in,
        .product-card-decorative-mirror.aufg-fade-in {
            opacity: 1 !important;
            transform: scale(1) !important;
            pointer-events: auto !important;
        }
        
        /* Positioning for filtered products */
        .aufg-position-1 { order: 1 !important; }
        .aufg-position-2 { order: 2 !important; }
        .aufg-position-3 { order: 3 !important; }
        .aufg-position-4 { order: 4 !important; }
    `;
    document.head.appendChild(style);
    console.log('Applied overflow fix CSS');
    
    // Wait for everything to load
    setTimeout(function() {
        // Find the carousel container
        const carousel = document.querySelector('.products-carousel');
        const wrapper = document.querySelector('.products-carousel-wrapper');
        
        console.log('Carousel found:', !!carousel);
        console.log('Wrapper found:', !!wrapper);
        
        if (carousel) {
            // Check current styles
            const computedStyle = window.getComputedStyle(carousel);
            console.log('Carousel display:', computedStyle.display);
            console.log('Carousel overflow-x:', computedStyle.overflowX);
            console.log('Carousel width:', carousel.scrollWidth, 'Client width:', carousel.clientWidth);
            
            // Find all product cards
            const products = carousel.querySelectorAll('.product-card-clear-float, .product-card-silver-mirror, .product-card-pattern-glass, .product-card-decorative-mirror');
            console.log('Products found:', products.length);
            
            // Clean up each product
            products.forEach((product, index) => {
                console.log(`Product ${index + 1}:`, {
                    display: product.style.display,
                    opacity: product.style.opacity,
                    visibility: product.style.visibility
                });
                
                // Force reset all styles
                product.setAttribute('style', '');
                
                // Ensure proper display
                const productStyle = window.getComputedStyle(product);
                console.log(`Product ${index + 1} computed display:`, productStyle.display);
            });
            
            // Force enable scrolling
            carousel.style.overflowX = 'auto';
            carousel.style.overflowY = 'hidden';
            carousel.style.scrollBehavior = 'smooth';
            
            // Test scroll
            console.log('Testing scroll...');
            carousel.scrollLeft = 100;
            setTimeout(() => {
                console.log('Scroll position after test:', carousel.scrollLeft);
                if (carousel.scrollLeft === 0) {
                    console.log('WARNING: Scroll not working, applying additional fix...');
                    // Try to fix by ensuring the carousel has proper width
                    carousel.style.width = '100%';
                    carousel.style.maxWidth = '100%';
                }
                carousel.scrollLeft = 0;
            }, 500);
        }
        
        // Check for navigation buttons
        const prevBtn = document.querySelector('.carousel-prev');
        const nextBtn = document.querySelector('.carousel-next');
        console.log('Prev button found:', !!prevBtn);
        console.log('Next button found:', !!nextBtn);
        
        // Add tab navigation functionality
        const allTab = document.querySelector('.tab-all');
        const floatGlassTab = document.querySelector('.tab-float-glass'); 
        const patternGlassTab = document.querySelector('.tab-pattern-glass');
        const mirrorTab = document.querySelector('.tab-mirror');
        
        const filterTabs = [allTab, floatGlassTab, patternGlassTab, mirrorTab];
        const validTabs = filterTabs.filter(tab => tab !== null);
        
        if (validTabs.length > 0) {
            console.log('Setting up tab navigation');
            
            // Set active tab styling
            function setActiveTab(activeTab) {
                validTabs.forEach(tab => {
                    tab.classList.remove('active-product-tab', 'active-tab-wrapper');
                });
                if (activeTab) {
                    activeTab.classList.add('active-product-tab', 'active-tab-wrapper');
                }
            }
            
            // Product filtering with fade effects
            const clearFloatContainer = document.querySelector('.product-card-clear-float');
            const silverMirrorContainer = document.querySelector('.product-card-silver-mirror');
            const patternGlassContainer = document.querySelector('.product-card-pattern-glass');
            const decorativeMirrorContainer = document.querySelector('.product-card-decorative-mirror');
            const allProducts = [clearFloatContainer, silverMirrorContainer, patternGlassContainer, decorativeMirrorContainer].filter(p => p !== null);
            
            let isAnimating = false;
            
            function filterProducts(category) {
                if (isAnimating) return;
                isAnimating = true;
                
                console.log(`Filtering for category: ${category}`);
                
                let visibleProducts = [];
                
                switch(category) {
                    case 'all':
                        visibleProducts = allProducts;
                        break;
                    case 'float':
                        if (clearFloatContainer) visibleProducts = [clearFloatContainer];
                        break;
                    case 'pattern':
                        if (patternGlassContainer) visibleProducts = [patternGlassContainer];
                        break;
                    case 'mirror':
                        visibleProducts = [silverMirrorContainer, decorativeMirrorContainer].filter(p => p !== null);
                        break;
                }
                
                const productsToHide = allProducts.filter(p => !visibleProducts.includes(p));
                
                // Step 1: Fade out products to hide and reset all positions
                allProducts.forEach(product => {
                    // Remove all position classes first
                    product.classList.remove('aufg-position-1', 'aufg-position-2', 'aufg-position-3', 'aufg-position-4');
                });
                
                productsToHide.forEach(product => {
                    product.classList.add('aufg-hidden');
                    product.classList.remove('aufg-fade-in');
                });
                
                // Step 2: After fade out, show visible products with staggered fade-in and new positions
                setTimeout(() => {
                    // Scroll to start first
                    carousel.scrollTo({ left: 0, behavior: 'smooth' });
                    
                    visibleProducts.forEach((product, index) => {
                        // Set new position order (1st position, 2nd position, etc.)
                        product.classList.add(`aufg-position-${index + 1}`);
                        
                        setTimeout(() => {
                            product.classList.remove('aufg-hidden');
                            product.classList.add('aufg-fade-in');
                        }, index * 200); // Stagger the fade-in
                    });
                    
                    // Reset animation lock
                    setTimeout(() => {
                        isAnimating = false;
                    }, visibleProducts.length * 200 + 600);
                    
                }, 600); // Wait for fade out
            }
            
            // Tab click handlers with filtering
            if (allTab) {
                allTab.style.cursor = 'pointer';
                allTab.addEventListener('click', function() {
                    setActiveTab(this);
                    filterProducts('all');
                    console.log('All tab - showing all products');
                });
            }
            
            if (floatGlassTab) {
                floatGlassTab.style.cursor = 'pointer';
                floatGlassTab.addEventListener('click', function() {
                    setActiveTab(this);
                    filterProducts('float');
                    console.log('Float tab - showing float glass only');
                });
            }
            
            if (patternGlassTab) {
                patternGlassTab.style.cursor = 'pointer';
                patternGlassTab.addEventListener('click', function() {
                    setActiveTab(this);
                    filterProducts('pattern');
                    console.log('Pattern tab - showing pattern glass only');
                });
            }
            
            if (mirrorTab) {
                mirrorTab.style.cursor = 'pointer';
                mirrorTab.addEventListener('click', function() {
                    setActiveTab(this);
                    filterProducts('mirror');
                    console.log('Mirror tab - showing mirror products only');
                });
            }
            
            // Initialize all products as visible with their default positions
            allProducts.forEach((product, index) => {
                product.classList.remove('aufg-hidden');
                product.classList.add('aufg-fade-in');
                product.classList.add(`aufg-position-${index + 1}`);
            });
            
            // Initialize with All tab active
            if (allTab) {
                setActiveTab(allTab);
            }
            
            console.log('Tab filtering with fade effects and carousel scrolling initialized');
        }
        
        console.log('=== CAROUSEL DEBUG END ===');
    }, 2000);
});