/* ===================================
   AUFG GLOBAL HEADER JAVASCRIPT
   Handles all header interactions
   =================================== */

(function() {
	'use strict';

	// Wait for DOM to be ready
	document.addEventListener('DOMContentLoaded', function() {
		initializeGlobalHeader();
	});

	function initializeGlobalHeader() {
		const header = document.getElementById('aufgGlobalHeader');
		if (!header) return;

		// Initialize all header components
		initStickyHeader(header);
		initMobileMenu(header);
		initLanguageSelector(header);
		initDropdownMenus(header);
		initActiveNavigation(header);
	}

	// ===== STICKY HEADER FUNCTIONALITY =====
	function initStickyHeader(header) {
		let lastScrollTop = 0;
		let ticking = false;

		function updateHeader() {
			const scrollTop = window.pageYOffset || document.documentElement.scrollTop;

			// Add scrolled class when scrolled down
			if (scrollTop > 100) {
				header.classList.add('scrolled');
			} else {
				header.classList.remove('scrolled');
			}

			// Hide/show header on scroll
			if (scrollTop > lastScrollTop && scrollTop > 400) {
				// Scrolling down
				header.classList.add('hidden');
			} else if (scrollTop < lastScrollTop || scrollTop <= 400) {
				// Scrolling up or at top
				header.classList.remove('hidden');
			}

			lastScrollTop = scrollTop;
			ticking = false;
		}

		function requestTick() {
			if (!ticking) {
				window.requestAnimationFrame(updateHeader);
				ticking = true;
			}
		}

		window.addEventListener('scroll', requestTick);
	}

	// ===== MOBILE MENU FUNCTIONALITY =====
	function initMobileMenu(header) {
		const menuToggle = document.getElementById('aufgMobileToggle') || document.getElementById('aufgMenuToggle');
		const mobileNav = document.getElementById('aufgMobileNav');
		
		if (!menuToggle || !mobileNav) return;

		menuToggle.addEventListener('click', function() {
			const isActive = this.classList.contains('active');
			
			if (isActive) {
				this.classList.remove('active');
				mobileNav.classList.remove('active');
				this.setAttribute('aria-expanded', 'false');
				document.body.style.overflow = '';
			} else {
				this.classList.add('active');
				mobileNav.classList.add('active');
				this.setAttribute('aria-expanded', 'true');
				document.body.style.overflow = 'hidden';
			}
		});

		// Handle submenu toggles in mobile
		const hasSubmenu = mobileNav.querySelectorAll('.aufg-has-submenu > a');
		hasSubmenu.forEach(link => {
			link.addEventListener('click', function(e) {
				const submenu = this.nextElementSibling;
				if (submenu) {
					e.preventDefault();
					submenu.style.display = submenu.style.display === 'block' ? 'none' : 'block';
					this.classList.toggle('active');
				}
			});
		});

		// Close mobile menu on window resize
		let resizeTimer;
		window.addEventListener('resize', function() {
			clearTimeout(resizeTimer);
			resizeTimer = setTimeout(function() {
				if (window.innerWidth > 768) {
					menuToggle.classList.remove('active');
					mobileNav.classList.remove('active');
					document.body.style.overflow = '';
				}
			}, 250);
		});
	}

	// ===== LANGUAGE SELECTOR FUNCTIONALITY =====
	function initLanguageSelector(header) {
		const langToggle = document.getElementById('aufgLangToggle');
		const langDropdown = document.getElementById('aufgLangDropdown');
		const langSelector = header.querySelector('.aufg-language-selector');
		
		if (!langToggle || !langDropdown) return;

		// Toggle dropdown
		langToggle.addEventListener('click', function(e) {
			e.stopPropagation();
			langSelector.classList.toggle('active');
		});

		// Handle language selection
		const langOptions = langDropdown.querySelectorAll('.aufg-lang-option');
		langOptions.forEach(option => {
			option.addEventListener('click', function() {
				const lang = this.getAttribute('data-lang');
				const currentLang = langToggle.querySelector('.aufg-current-lang');
				
				// Update current language display
				if (lang === 'ar') {
					currentLang.textContent = 'AR';
					document.documentElement.setAttribute('dir', 'rtl');
					document.documentElement.setAttribute('lang', 'ar');
				} else {
					currentLang.textContent = 'EN';
					document.documentElement.removeAttribute('dir');
					document.documentElement.setAttribute('lang', 'en');
				}
				
				// Store preference
				localStorage.setItem('preferredLanguage', lang);
				
				// Close dropdown
				langSelector.classList.remove('active');
			});
		});

		// Load saved language preference
		const savedLang = localStorage.getItem('preferredLanguage');
		if (savedLang) {
			const savedOption = langDropdown.querySelector(`[data-lang="${savedLang}"]`);
			if (savedOption) {
				savedOption.click();
			}
		}

		// Close dropdown on outside click
		document.addEventListener('click', function(e) {
			if (!langSelector.contains(e.target)) {
				langSelector.classList.remove('active');
			}
		});
	}

	// ===== DROPDOWN MENUS FUNCTIONALITY =====
	function initDropdownMenus(header) {
		const dropdownItems = header.querySelectorAll('.aufg-has-dropdown');
		
		dropdownItems.forEach(item => {
			let timeout;
			
			// Show dropdown on hover (desktop)
			item.addEventListener('mouseenter', function() {
				if (window.innerWidth > 768) {
					clearTimeout(timeout);
					this.classList.add('active');
				}
			});
			
			// Hide dropdown with delay
			item.addEventListener('mouseleave', function() {
				if (window.innerWidth > 768) {
					timeout = setTimeout(() => {
						this.classList.remove('active');
					}, 200);
				}
			});
			
			// Handle keyboard navigation
			const link = item.querySelector('.aufg-nav-link');
			link.addEventListener('keydown', function(e) {
				if (e.key === 'Enter' || e.key === ' ') {
					e.preventDefault();
					item.classList.toggle('active');
				}
			});
		});

		// Close dropdowns on ESC key
		document.addEventListener('keydown', function(e) {
			if (e.key === 'Escape') {
				dropdownItems.forEach(item => {
					item.classList.remove('active');
				});
			}
		});
	}

	// ===== ACTIVE NAVIGATION HIGHLIGHTING =====
	function initActiveNavigation(header) {
		const currentPath = window.location.pathname;
		const currentFile = currentPath.split('/').pop() || 'index.html';
		const navLinks = header.querySelectorAll('.aufg-nav-link');
		
		// Remove existing active classes first
		navLinks.forEach(link => link.classList.remove('active'));
		
		navLinks.forEach(link => {
			const href = link.getAttribute('href');
			if (href) {
				// Extract filename from href
				const hrefFile = href.split('/').pop();
				
				// Special handling for home page
				if ((currentPath === '/' || currentFile === 'index.html' || currentFile === '') && 
					(href === '/' || hrefFile === 'index.html')) {
					link.classList.add('active');
					return;
				}
				
				// Check for exact filename match
				if (currentFile === hrefFile) {
					link.classList.add('active');
					
					// Also highlight parent menu if in dropdown
					const parentItem = link.closest('.aufg-has-dropdown');
					if (parentItem) {
						const parentLink = parentItem.querySelector('.aufg-nav-link');
						if (parentLink && !parentLink.classList.contains('active')) {
							parentLink.classList.add('active');
						}
					}
					return;
				}
				
				// For product pages, also highlight the main Products menu
				if (currentFile.includes('glass') && href === 'products.html') {
					link.classList.add('active');
				}
			}
		});
	}

	// ===== SMOOTH SCROLL FOR ANCHOR LINKS =====
	function initSmoothScroll() {
		const anchorLinks = document.querySelectorAll('a[href^="#"]:not([href="#"])');
		
		anchorLinks.forEach(link => {
			link.addEventListener('click', function(e) {
				const targetId = this.getAttribute('href');
				const targetElement = document.querySelector(targetId);
				
				if (targetElement) {
					e.preventDefault();
					const headerHeight = document.getElementById('aufgGlobalHeader').offsetHeight;
					const targetPosition = targetElement.offsetTop - headerHeight;
					
					window.scrollTo({
						top: targetPosition,
						behavior: 'smooth'
					});
				}
			});
		});
	}

	// ===== ACCESSIBILITY ENHANCEMENTS =====
	function initAccessibility() {
		// Add ARIA labels dynamically
		const header = document.getElementById('aufgGlobalHeader');
		if (!header) return;

		// Set role for navigation
		const nav = header.querySelector('.aufg-nav-section');
		if (nav) {
			nav.setAttribute('role', 'navigation');
		}

		// Handle focus trap in mobile menu when open
		const mobileNav = document.getElementById('aufgMobileNav');
		const menuToggle = document.getElementById('aufgMobileToggle') || document.getElementById('aufgMenuToggle');
		
		if (mobileNav && menuToggle) {
			const focusableElements = mobileNav.querySelectorAll(
				'a, button, [tabindex]:not([tabindex="-1"])'
			);
			
			if (focusableElements.length > 0) {
				const firstFocusable = focusableElements[0];
				const lastFocusable = focusableElements[focusableElements.length - 1];
				
				mobileNav.addEventListener('keydown', function(e) {
					if (e.key === 'Tab') {
						if (e.shiftKey) {
							if (document.activeElement === firstFocusable) {
								e.preventDefault();
								menuToggle.focus();
							}
						} else {
							if (document.activeElement === lastFocusable) {
								e.preventDefault();
								menuToggle.focus();
							}
						}
					}
				});
			}
		}
	}

	// Initialize accessibility features
	initAccessibility();
	
	// Initialize smooth scroll
	initSmoothScroll();

	// ===== PERFORMANCE OPTIMIZATION =====
	// Debounce function for scroll events
	function debounce(func, wait, immediate) {
		let timeout;
		return function() {
			const context = this, args = arguments;
			const later = function() {
				timeout = null;
				if (!immediate) func.apply(context, args);
			};
			const callNow = immediate && !timeout;
			clearTimeout(timeout);
			timeout = setTimeout(later, wait);
			if (callNow) func.apply(context, args);
		};
	}

	// ===== PUBLIC API =====
	window.AUFGHeader = {
		// Expose methods for external use
		hideHeader: function() {
			const header = document.getElementById('aufgGlobalHeader');
			if (header) header.classList.add('hidden');
		},
		showHeader: function() {
			const header = document.getElementById('aufgGlobalHeader');
			if (header) header.classList.remove('hidden');
		},
		closeMobileMenu: function() {
			const menuToggle = document.getElementById('aufgMobileToggle') || document.getElementById('aufgMenuToggle');
			const mobileNav = document.getElementById('aufgMobileNav');
			if (menuToggle && mobileNav) {
				menuToggle.classList.remove('active');
				mobileNav.classList.remove('active');
				document.body.style.overflow = '';
			}
		}
	};

})();