// Direct Language Selection from Header Dropdown
document.addEventListener('DOMContentLoaded', function() {
    console.log('Initializing language selector...');
    
    // Get elements
    const languageToggle = document.getElementById('languageToggle');
    const languageContainer = document.querySelector('.language-selector-container');
    const languageMenu = document.getElementById('languageMenu');
    const currentLangSpan = document.querySelector('.current-lang');
    const languageOptions = document.querySelectorAll('.language-option');
    
    // Get saved language preference
    let selectedLanguage = localStorage.getItem('selectedLanguage') || 'en';
    
    // Apply initial language
    applyLanguage(selectedLanguage);
    
    if (!languageToggle || !languageContainer || !languageMenu) {
        console.error('Language selector elements not found');
        return;
    }
    
    // Toggle dropdown
    languageToggle.addEventListener('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        const isOpen = languageContainer.classList.contains('open');
        
        if (isOpen) {
            closeDropdown();
        } else {
            openDropdown();
        }
    });
    
    // Handle language selection
    languageOptions.forEach(option => {
        option.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const selectedLang = this.dataset.lang;
            
            // Save selection
            localStorage.setItem('selectedLanguage', selectedLang);
            
            // Apply language
            applyLanguage(selectedLang);
            
            // Close dropdown
            closeDropdown();
            
            console.log(`Language changed to: ${selectedLang}`);
        });
    });
    
    // Close dropdown when clicking outside
    document.addEventListener('click', function(e) {
        if (!languageContainer.contains(e.target) && !languageMenu.contains(e.target)) {
            closeDropdown();
        }
    });
    
    // Close dropdown on escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            closeDropdown();
        }
    });
    
    // Functions
    function openDropdown() {
        languageContainer.classList.add('open');
        languageMenu.classList.add('show');
        languageToggle.setAttribute('aria-expanded', 'true');
    }
    
    function closeDropdown() {
        languageContainer.classList.remove('open');
        languageMenu.classList.remove('show');
        languageToggle.setAttribute('aria-expanded', 'false');
    }
    
    function applyLanguage(lang) {
        // Update current language display
        if (currentLangSpan) {
            currentLangSpan.textContent = lang.toUpperCase();
        }
        
        // Update document language attributes
        document.documentElement.setAttribute('lang', lang);
        document.body.classList.remove('lang-en', 'lang-ar');
        document.body.classList.add(`lang-${lang}`);
        
        // Handle RTL for Arabic
        if (lang === 'ar') {
            document.documentElement.setAttribute('dir', 'rtl');
            document.body.classList.add('rtl');
        } else {
            document.documentElement.setAttribute('dir', 'ltr');
            document.body.classList.remove('rtl');
        }
        
        // Update selected language variable
        selectedLanguage = lang;
        
        // Trigger custom event for other components
        window.dispatchEvent(new CustomEvent('languageChanged', { 
            detail: { language: lang } 
        }));
        
        console.log(`Applied language: ${lang}`);
    }
    
    // Make language change function available globally
    window.changeLanguage = function(lang) {
        if (lang && (lang === 'en' || lang === 'ar')) {
            localStorage.setItem('selectedLanguage', lang);
            applyLanguage(lang);
        }
    };
    
    // Initialize language options active state
    function updateActiveState() {
        languageOptions.forEach(option => {
            option.classList.remove('active');
            if (option.dataset.lang === selectedLanguage) {
                option.classList.add('active');
            }
        });
    }
    
    // Initial active state update
    updateActiveState();
    
    // Update active state when language changes
    window.addEventListener('languageChanged', updateActiveState);
    
    console.log('Language selector initialized successfully');
});