@extends('layouts.admin-simple')

@section('title', 'Direct Deliveries')

@push('styles')
<style>
/* Adjust Select2 dropdown widths for better proportions */
.select2-container {
    width: 100% !important;
}

/* Make dropdowns auto-width based on content */
.select2-dropdown {
    min-width: 200px;
    max-width: 400px;
    width: auto !important;
}

/* Ensure the selection box doesn't get too wide */
.select2-container .select2-selection--single {
    min-width: 200px;
    max-width: 100%;
}

/* Allow dropdown to expand based on content */
.select2-container--open .select2-dropdown--below {
    width: auto !important;
    min-width: 200px;
}
</style>
@endpush

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="card-title mb-0">
                        <i class="material-icons mr-2">local_shipping</i>
                        Direct Deliveries
                    </h4>
                    @can('warehouse.outgoing.create')
                    <a href="{{ route('warehouse.direct-deliveries.create') }}" class="btn btn-primary">
                        <i class="material-icons mr-1">add</i>
                        New Direct Delivery
                    </a>
                    @endcan
                </div>

                <div class="card-body">
                    <!-- Filters -->
                    <div class="row mb-4">
                        <div class="col-md-2">
                            <x-warehouse.select2-dropdown
                                name="status"
                                id="status_filter"
                                label="Status"
                                placeholder="All Status"
                                :options="[
                                    ['value' => '', 'text' => 'All Status'],
                                    ['value' => 'pending', 'text' => 'Pending'],
                                    ['value' => 'partially_received', 'text' => 'Partially Received'],
                                    ['value' => 'received', 'text' => 'Received'],
                                    ['value' => 'cancelled', 'text' => 'Cancelled']
                                ]"
                                :selected="request('status')"
                                allowClear="true"
                            />
                        </div>
                        <div class="col-md-2">
                            <x-warehouse.select2-dropdown
                                name="supplier_id"
                                id="supplier_filter"
                                label="Supplier"
                                placeholder="All Suppliers"
                                :options="collect([['value' => '', 'text' => 'All Suppliers']])->merge($suppliers->map(function($supplier) {
                                    return ['value' => $supplier->id, 'text' => $supplier->name];
                                }))"
                                :selected="request('supplier_id')"
                                allowClear="true"
                            />
                        </div>
                        <div class="col-md-2">
                            <x-warehouse.select2-dropdown
                                name="project_id"
                                id="project_filter"
                                label="Project"
                                placeholder="All Projects"
                                :options="collect([['value' => '', 'text' => 'All Projects']])->merge($projects->map(function($project) {
                                    return ['value' => $project->id, 'text' => $project->project_name];
                                }))"
                                :selected="request('project_id')"
                                allowClear="true"
                            />
                        </div>
                        <div class="col-md-2">
                            <x-warehouse.select2-dropdown
                                name="delivery_type"
                                id="delivery_type_filter"
                                label="Type"
                                placeholder="All Types"
                                :options="[
                                    ['value' => '', 'text' => 'All Types'],
                                    ['value' => 'purchase_order', 'text' => 'Purchase Order'],
                                    ['value' => 'direct_purchase', 'text' => 'Direct Purchase'],
                                    ['value' => 'emergency', 'text' => 'Emergency']
                                ]"
                                :selected="request('delivery_type')"
                                allowClear="true"
                            />
                        </div>
                        <div class="col-md-2">
                            <label for="date_from">Date From</label>
                            <input type="date" name="date_from" id="date_from" class="form-control" value="{{ request('date_from') }}">
                        </div>
                        <div class="col-md-2">
                            <label for="date_to">Date To</label>
                            <input type="date" name="date_to" id="date_to" class="form-control" value="{{ request('date_to') }}">
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="row mb-4">
                        <div class="col-12">
                            <button type="button" onclick="applyFilters()" class="btn btn-info mr-2">
                                <i class="material-icons mr-1">filter_list</i>
                                Apply Filters
                            </button>
                            <a href="{{ route('warehouse.direct-deliveries.index') }}" class="btn btn-outline-secondary">
                                <i class="material-icons mr-1">clear</i>
                                Clear
                            </a>
                        </div>
                    </div>

                    <!-- Direct Deliveries Table -->
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="thead-light">
                                <tr>
                                    <th>Delivery #</th>
                                    <th>Date</th>
                                    <th>Supplier</th>
                                    <th>Project</th>
                                    <th>Type</th>
                                    <th>Status</th>
                                    <th>Items</th>
                                    <th>Total Value</th>
                                    <th>Delivered By</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($directDeliveries as $delivery)
                                <tr>
                                    <td>
                                        <strong>{{ $delivery->delivery_number }}</strong>
                                        @if($delivery->purchase_order_number)
                                        <br><small class="text-muted">PO: {{ $delivery->purchase_order_number }}</small>
                                        @endif
                                    </td>
                                    <td>{{ $delivery->delivery_date->format('M d, Y') }}</td>
                                    <td>
                                        @if($delivery->supplier)
                                            <span class="text-primary">
                                                {{ $delivery->supplier->name }}
                                            </span>
                                            @if($delivery->supplier_delivery_note)
                                            <br><small class="text-muted">Note: {{ $delivery->supplier_delivery_note }}</small>
                                            @endif
                                        @else
                                            <span class="text-muted">
                                                <i class="material-icons small" style="font-size: 14px;">edit</i>
                                                Pending Supplier Selection
                                            </span>
                                            <br><small class="text-warning">Material Request - Needs supplier details</small>
                                        @endif
                                    </td>
                                    <td>
                                        <span class="text-success">
                                            {{ $delivery->project->project_name }}
                                        </span>
                                        @if($delivery->projectDivision)
                                        <br><small class="text-muted">{{ $delivery->projectDivision->division_name }}</small>
                                        @endif
                                    </td>
                                    <td>
                                        <span class="badge badge-{{ $delivery->delivery_type_color }}">
                                            {{ $delivery->delivery_type_display }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-{{ $delivery->status_color }}">
                                            {{ ucfirst(str_replace('_', ' ', $delivery->status)) }}
                                        </span>
                                        @if($delivery->needsApproval())
                                        <br><small class="text-warning">Needs Approval</small>
                                        @endif
                                    </td>
                                    <td>{{ $delivery->items->count() }}</td>
                                    <td>
                                        @if($delivery->total_amount)
                                        {{ $delivery->currency }} {{ number_format($delivery->total_amount, 2) }}
                                        @else
                                        <span class="text-muted">-</span>
                                        @endif
                                    </td>
                                    <td>
                                        @if($delivery->delivered_by)
                                        {{ $delivery->delivered_by }}
                                        @if($delivery->vehicle_number)
                                        <br><small class="text-muted">{{ $delivery->vehicle_number }}</small>
                                        @endif
                                        @else
                                        <span class="text-muted">-</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('warehouse.direct-deliveries.show', $delivery) }}"
                                               class="btn btn-sm btn-outline-primary" title="View Details">
                                                <i class="material-icons">visibility</i>
                                            </a>
                                            @if($delivery->canEdit())
                                            <a href="{{ route('warehouse.direct-deliveries.edit', $delivery) }}"
                                               class="btn btn-sm btn-outline-warning" title="Edit">
                                                <i class="material-icons">edit</i>
                                            </a>
                                            @endif
                                            @if($delivery->canReceive())
                                            <a href="{{ route('warehouse.direct-deliveries.receipt', $delivery) }}"
                                               class="btn btn-sm btn-outline-success" title="Process Receipt">
                                                <i class="material-icons">receipt</i>
                                            </a>
                                            @endif
                                            @if($delivery->needsApproval())
                                            <button type="button" class="btn btn-sm btn-outline-info"
                                                    onclick="approveDelivery({{ $delivery->id }})" title="Approve">
                                                <i class="material-icons">approval</i>
                                            </button>
                                            @endif
                                            <button type="button" class="btn btn-sm btn-outline-info"
                                                    onclick="printDirectDelivery({{ $delivery->id }})" title="Print Delivery Note">
                                                <i class="material-icons">print</i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="10" class="text-center py-4">
                                        <div class="text-muted">
                                            <i class="material-icons mb-2" style="font-size: 48px;">local_shipping</i>
                                            <p>No direct deliveries found</p>
                                            @can('warehouse.outgoing.create')
                                            <a href="{{ route('warehouse.direct-deliveries.create') }}" class="btn btn-primary">
                                                Create First Direct Delivery
                                            </a>
                                            @endcan
                                        </div>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    <!-- Pagination -->
                    @if($directDeliveries->hasPages())
                    <div class="d-flex justify-content-center mt-4">
                        {{ $directDeliveries->appends(request()->query())->links() }}
                    </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Approval Modal -->
<div class="modal fade" id="approvalModal" tabindex="-1" role="dialog" aria-labelledby="approvalModalLabel" aria-hidden="true">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="approvalModalLabel">Approve Direct Delivery</h5>
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
            <form id="approvalForm" method="POST">
                @csrf
                <div class="modal-body">
                    <div class="form-group">
                        <label for="approval_notes">Approval Notes</label>
                        <textarea name="approval_notes" id="approval_notes" class="form-control" rows="3" placeholder="Optional notes for this approval..."></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-success">Approve Delivery</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
function applyFilters() {
    const params = new URLSearchParams();

    // Get filter values
    const status = $('#status_filter').val();
    const supplier = $('#supplier_filter').val();
    const project = $('#project_filter').val();
    const deliveryType = $('#delivery_type_filter').val();
    const dateFrom = document.getElementById('date_from').value;
    const dateTo = document.getElementById('date_to').value;

    // Add non-empty parameters
    if (status) params.append('status', status);
    if (supplier) params.append('supplier_id', supplier);
    if (project) params.append('project_id', project);
    if (deliveryType) params.append('delivery_type', deliveryType);
    if (dateFrom) params.append('date_from', dateFrom);
    if (dateTo) params.append('date_to', dateTo);

    // Redirect with filters
    window.location.href = '{{ route("warehouse.direct-deliveries.index") }}?' + params.toString();
}

function approveDelivery(deliveryId) {
    document.getElementById('approvalForm').action = `/warehouse/direct-deliveries/${deliveryId}/approve`;
    $('#approvalModal').modal('show');
}

// Print function for direct deliveries
function printDirectDelivery(id) {
    console.log('printDirectDelivery called with id:', id);

    // Open print view in new window - using Laravel route helper
    const printUrl = '{{ route("warehouse.direct-deliveries.print", ":id") }}'.replace(':id', id);
    console.log('Opening print URL:', printUrl);
    const printWindow = window.open(printUrl, '_blank', 'width=800,height=600,scrollbars=yes,resizable=yes');

    if (printWindow) {
        printWindow.focus();

        // Wait for content to load then print
        printWindow.addEventListener('load', function() {
            setTimeout(() => {
                printWindow.print();
            }, 1000);
        });
    } else {
        // Fallback: navigate to print page if popup blocked
        window.open(printUrl, '_blank');
    }
}
</script>
@endpush