@extends('layouts.admin-simple')

@section('title', 'Material Request')

@section('meta')
<meta name="csrf-token" content="{{ csrf_token() }}">
@endsection

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item">Inventory</li>
    <li class="breadcrumb-item active">Material Request</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Material Request</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Track and manage material requests from inventory</p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            @can('warehouse.outgoing.create')
            <a href="{{ route('warehouse.outgoing.create') }}" class="btn btn-sm btn-primary">
                <i class="fas fa-plus mr-1"></i>
                Create New Material Request
            </a>
            @endcan
        </div>
    </div>
@endsection

@section('content')

    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Requests"
                :value="$stats['total_requests'] ?? 25"
                icon="assignment"
                color="primary"
                subtitle="All material requests"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Pending"
                :value="$stats['pending'] ?? 8"
                icon="pending"
                color="warning"
                subtitle="Awaiting approval"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Successful"
                :value="$stats['dispatched'] ?? 12"
                icon="check_circle"
                color="success"
                subtitle="Successfully dispatched"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Canceled"
                :value="$stats['cancelled'] ?? 3"
                icon="cancel"
                color="danger"
                subtitle="Canceled requests"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-xl-3 col-md-6 mb-3">
                    <x-warehouse.select2-dropdown
                        name="filter_status"
                        id="filter_status"
                        label="Status"
                        placeholder="All Status"
                        :options="[
                            ['value' => '', 'text' => 'All Status'],
                            ['value' => 'pending', 'text' => 'Pending'],
                            ['value' => 'successful', 'text' => 'Successful'],
                            ['value' => 'canceled', 'text' => 'Canceled']
                        ]"
                        allowClear="true"
                    />
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Date From</label>
                    <input type="date" id="filter_date_from" class="form-control">
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Date To</label>
                    <input type="date" id="filter_date_to" class="form-control">
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <x-warehouse.select2-dropdown
                        name="filter_project"
                        id="filter_project"
                        label="Project"
                        placeholder="All Projects"
                        :options="[['value' => '', 'text' => 'All Projects']]"
                        allowClear="true"
                    />
                </div>
            </div>

            <div class="d-flex justify-content-between align-items-center mt-3 pt-3 border-top">
                <button type="button" onclick="applyFilters()" class="btn btn-primary" id="apply_filters">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">filter_list</i>
                    Apply Filters
                </button>

                <button type="button" onclick="clearFilters()" class="btn btn-outline-secondary" id="clear_filters">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">clear</i>
                    Clear Filters
                </button>
            </div>
        </div>
    </div>

    <!-- Material Requests Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">list</i>
                Material Request List
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="materialRequestsTable" class="table table-bordered table-hover">
                    <thead class="thead-light">
                        <tr>
                            <th>Reference</th>
                            <th>Date</th>
                            <th>Project</th>
                            <th>Division</th>
                            <th>Requested By</th>
                            <th>Items</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Data will be loaded via AJAX -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- View Details Modal -->
    <div class="modal fade" id="viewDetailsModal" tabindex="-1" role="dialog" aria-labelledby="viewDetailsModalLabel" aria-hidden="true" data-backdrop="static" data-keyboard="false">
        <div class="modal-dialog modal-lg modal-dialog-centered" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="viewDetailsModalLabel">
                        <i class="material-icons align-middle mr-1">assignment</i>
                        Material Request Details
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div id="requestDetailsContent">
                        <!-- Content will be loaded dynamically -->
                        <div class="text-center py-4">
                            <i class="material-icons text-muted" style="font-size: 48px;">assignment</i>
                            <div class="text-muted">Loading request details...</div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

@endsection

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <style>
        /* Adjust Select2 dropdown widths for better proportions */
        .select2-container {
            width: 100% !important;
        }

        /* Make dropdowns auto-width based on content */
        .select2-dropdown {
            min-width: 200px;
            max-width: 400px;
            width: auto !important;
        }

        /* Ensure the selection box doesn't get too wide */
        .select2-container .select2-selection--single {
            min-width: 200px;
            max-width: 100%;
        }

        /* Allow dropdown to expand based on content */
        .select2-container--open .select2-dropdown--below {
            width: auto !important;
            min-width: 200px;
        }
        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .status-badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.75rem;
            font-size: 0.75rem;
            font-weight: 500;
            line-height: 1;
            border-radius: 0.375rem;
        }

        .status-pending {
            color: #856404;
            background-color: #fff3cd;
            border: 1px solid #ffeaa7;
        }

        .status-dispatched {
            color: #155724;
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
        }

        .status-cancelled {
            color: #721c24;
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
        }

        /* Modal positioning fixes */
        .modal {
            z-index: 1050 !important;
        }

        .modal-backdrop {
            z-index: 1040 !important;
        }

        .modal-dialog-centered {
            display: flex;
            align-items: center;
            min-height: calc(100% - 1rem);
        }

        .modal-content {
            position: relative;
            width: 100%;
            margin: 1.75rem auto;
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>
<script>
let materialRequestsTable;

$(document).ready(function() {
    initializeDataTable();

    // Entries per page change handler
    $('#entries-per-page').change(function() {
        materialRequestsTable.page.len($(this).val()).draw();
    });
});

function initializeDataTable() {
    materialRequestsTable = $('#materialRequestsTable').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '{{ route('warehouse.inventory.material-requests') }}',
            data: function(d) {
                d.status = $('#filter_status').val();
                d.date_from = $('#filter_date_from').val();
                d.date_to = $('#filter_date_to').val();
                d.project = $('#filter_project').val();
            }
        },
        columns: [
            {
                data: 'request_number',
                name: 'request_number',
                render: function(data) {
                    return `<strong>${data}</strong>`;
                }
            },
            {
                data: 'date',
                name: 'date',
                render: function(data) {
                    return data;
                }
            },
            {
                data: 'project',
                name: 'project',
                render: function(data) {
                    return data || '<span class="text-muted">N/A</span>';
                }
            },
            {
                data: 'division',
                name: 'division',
                render: function(data) {
                    return data || '<span class="text-muted">N/A</span>';
                }
            },
            {
                data: 'requester',
                name: 'requester',
                render: function(data) {
                    return data || '<span class="text-muted">N/A</span>';
                }
            },
            {
                data: 'items_count',
                name: 'items_count',
                render: function(data) {
                    return `<span class="badge badge-light">${data} items</span>`;
                }
            },
            {
                data: 'status',
                name: 'status',
                render: function(data, type, row) {
                    let badgeClass = 'badge-secondary';
                    let icon = 'info';

                    switch(data.label.toLowerCase()) {
                        case 'pending':
                            badgeClass = 'badge-warning';
                            icon = 'pending';
                            break;
                        case 'successful':
                            badgeClass = 'badge-success';
                            icon = 'check_circle';
                            break;
                        case 'canceled':
                            badgeClass = 'badge-danger';
                            icon = 'cancel';
                            break;
                    }

                    return `<span class="badge ${badgeClass}">
                        <i class="material-icons" style="font-size: 12px;">${icon}</i>
                        ${data.label}
                    </span>`;
                }
            },
            {
                data: 'actions',
                name: 'actions',
                orderable: false,
                searchable: false,
                render: function(data) {
                    return data;
                }
            }
        ],
        order: [[1, 'desc']],
        pageLength: 25,
        lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
        language: {
            processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
            emptyTable: 'No material requests found',
            zeroRecords: 'No matching material requests found'
        },
        drawCallback: function(settings) {
            $('#entries-per-page').val(settings._iDisplayLength);
        }
    });
}

function applyFilters() {
    materialRequestsTable.draw();
}

function clearFilters() {
    $('#filter_status, #filter_date_from, #filter_date_to, #filter_project').val('');
    materialRequestsTable.draw();
}

// Action functions
function dispatchRequest(id) {
    // Redirect to approval/dispatch page
    window.location.href = `/warehouse/inventory/material-requests/${id}/dispatch`;
}

function cancelRequest(id) {
    if (confirm('Are you sure you want to cancel this material request? This action cannot be undone.')) {
        // Send AJAX request to cancel the request
        fetch(`/warehouse/inventory/material-requests/${id}/dispatch`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '{{ csrf_token() }}'
            },
            body: JSON.stringify({
                action: 'cancel'
            })
        })
        .then(response => {
            if (response.ok) {
                // Reload table after successful cancellation
                materialRequestsTable.ajax.reload();

                // Show success message
                showAlert('Material request cancelled successfully!', 'success');
            } else {
                showAlert('Failed to cancel material request. Please try again.', 'danger');
            }
        })
        .catch(error => {
            console.error('Error cancelling request:', error);
            showAlert('Failed to cancel material request. Please try again.', 'danger');
        });
    }
}

function viewRequest(id) {
    console.log('viewRequest called with id:', id);

    // Hide any existing modals first
    $('.modal').modal('hide');

    // Wait a moment then show the new modal
    setTimeout(() => {
        // Load request details in modal
        $('#viewDetailsModal').modal({
            backdrop: 'static',
            keyboard: true,
            show: true
        });

        // Ensure modal appears in correct position
        $('#viewDetailsModal').on('shown.bs.modal', function () {
            $(this).css('z-index', 1050);
            $('.modal-backdrop').css('z-index', 1040);
        });

        // Show loading state
        $('#requestDetailsContent').html(`
            <div class="text-center py-4">
                <div class="spinner-border text-primary" role="status">
                    <span class="sr-only">Loading...</span>
                </div>
                <div class="text-muted mt-2">Loading request details...</div>
            </div>
        `);

        // Fetch request details after modal is shown
        const url = `/warehouse/inventory/material-requests/${id}`;
        console.log('Making request to:', url);

        fetch(url, {
        method: 'GET',
        headers: {
            'Accept': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]') ? document.querySelector('meta[name="csrf-token"]').getAttribute('content') : '{{ csrf_token() }}'
        }
    })
    .then(response => {
        console.log('Response status:', response.status);
        console.log('Response headers:', response.headers);

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.json();
    })
    .then(data => {
        console.log('Response data:', data);

        if (data.success) {
            const request = data.data;
            $('#requestDetailsContent').html(`
                <div class="row mb-3">
                    <div class="col-md-6">
                        <strong>Request Number:</strong><br>
                        <span class="text-primary">${request.request_number}</span>
                    </div>
                    <div class="col-md-6">
                        <strong>Date:</strong><br>
                        ${request.request_date}
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-md-6">
                        <strong>Project:</strong><br>
                        ${request.project}
                    </div>
                    <div class="col-md-6">
                        <strong>Requested By:</strong><br>
                        ${request.requested_by}
                    </div>
                </div>
                <div class="row mb-3">
                    <div class="col-12">
                        <strong>Items:</strong>
                        <div class="table-responsive mt-2">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Code</th>
                                        <th>Description</th>
                                        <th>Unit</th>
                                        <th>Requested</th>
                                        <th>Available</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${request.items.map(item => `
                                        <tr>
                                            <td>${item.item_code}</td>
                                            <td>${item.description}</td>
                                            <td>${item.unit}</td>
                                            <td>${item.quantity_requested}</td>
                                            <td>${item.quantity_available}</td>
                                        </tr>
                                    `).join('')}
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            `);
        } else {
            $('#requestDetailsContent').html(`
                <div class="alert alert-danger">
                    Error loading request details: ${data.message || 'Unknown error'}
                </div>
            `);
        }
    })
    .catch(error => {
        console.error('Fetch error:', error);
        $('#requestDetailsContent').html(`
            <div class="alert alert-danger">
                Error loading request details: ${error.message}<br>
                <small>Check console for more details</small>
            </div>
        `);
    });
    }, 100);
}

function deleteRequest(id) {
    if (confirm('Are you sure you want to delete this material request? This action cannot be undone.')) {
        fetch(`{{ route('warehouse.inventory.material-requests', '') }}/${id}`, {
            method: 'DELETE',
            headers: {
                'Accept': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Material request deleted successfully!', 'success');
                materialRequestsTable.ajax.reload();
            } else {
                showAlert('Error: ' + (data.message || 'Failed to delete request'), 'danger');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showAlert('Error deleting material request. Please try again.', 'danger');
        });
    }
}

function showAlert(message, type) {
    const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
    const alert = `
        <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
            ${message}
            <button type="button" class="close" data-dismiss="alert">&times;</button>
        </div>
    `;
    $('body').append(alert);
    setTimeout(() => $('.alert').alert('close'), 5000);
}
</script>
@endpush