@extends('layouts.admin-simple')

@section('title', 'Item Categories')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item active">Item Categories</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Item Categories</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Manage item categories and types of goods</p>
        </div>
        @can('warehouse.item-categories.create')
        <div class="page-actions-right d-flex align-items-center">
            <button onclick="showImportModal()" class="btn btn-sm btn-info mr-2" title="Import item categories">
                <i class="material-icons mr-1" style="font-size: 16px;">file_upload</i>
                Import
            </button>
            <button onclick="showAddCategoryModal()" class="btn btn-sm btn-primary" title="Add new item category">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                Add Category
            </button>
        </div>
        @endcan
    </div>
@endsection

@section('content')
    <!-- Filters Row -->
    <div class="card mb-3">
        <div class="card-body py-2">
            <div class="row align-items-center">
                <div class="col-md-3">
                    <div style="font-size: 12px; margin-bottom: 8px;">
                        <x-warehouse.select2-dropdown
                            name="filter-status"
                            id="filter-status"
                            label="Status Filter"
                            placeholder="All Statuses"
                            :options="[
                                ['value' => '', 'text' => 'All Statuses'],
                                ['value' => 'active', 'text' => 'Active'],
                                ['value' => 'inactive', 'text' => 'Inactive']
                            ]"
                            class="form-control-sm"
                        />
                    </div>
                </div>
                <div class="col-md-3">
                    <div style="font-size: 12px; margin-bottom: 8px;">
                        <x-warehouse.select2-dropdown
                            name="filter-type"
                            id="filter-type"
                            label="Type of Goods"
                            placeholder="All Types"
                            :options="[
                                ['value' => '', 'text' => 'All Types'],
                                ['value' => 'Raw Material', 'text' => 'Raw Material'],
                                ['value' => 'Finished Goods', 'text' => 'Finished Goods'],
                                ['value' => 'Work in Progress', 'text' => 'Work in Progress'],
                                ['value' => 'Consumables', 'text' => 'Consumables']
                            ]"
                            class="form-control-sm"
                        />
                    </div>
                </div>
                <div class="col-md-6 text-right">
                    <button onclick="applyFilters()" class="btn btn-sm btn-secondary mr-2">
                        <i class="material-icons mr-1" style="font-size: 14px;">filter_list</i>Apply Filters
                    </button>
                    <button onclick="clearFilters()" class="btn btn-sm btn-outline-secondary">
                        <i class="material-icons mr-1" style="font-size: 14px;">clear</i>Clear
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Categories Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">category</i>
                Item Categories List
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="categories-table" class="table table-hover">
                    <thead>
                        <tr>
                            <th>Category Info</th>
                            <th>Type of Goods</th>
                            <th>Status</th>
                            <th>Items Count</th>
                            <th>Created Date</th>
                            <th class="text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Data will be populated by DataTables -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Add Category Modal -->
    <div class="modal fade" id="addCategoryModal" tabindex="-1" role="dialog" aria-labelledby="addCategoryModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addCategoryModalLabel">
                        <i class="material-icons mr-2">category</i>
                        Add New Item Category
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <form id="addCategoryForm">
                        @csrf
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="category_name">Category Name <span class="text-danger">*</span></label>
                                    <input type="text" name="category_name" id="category_name" class="form-control" required>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <x-warehouse.select2-dropdown
                                    name="type_of_goods"
                                    id="type_of_goods"
                                    label="Type of Goods"
                                    placeholder="Loading types..."
                                    :options="[]"
                                    required="true"
                                />
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <x-warehouse.select2-dropdown
                                    name="status"
                                    id="status"
                                    label="Status"
                                    placeholder="Select Status"
                                    :options="[
                                        ['value' => 'active', 'text' => 'Active'],
                                        ['value' => 'inactive', 'text' => 'Inactive']
                                    ]"
                                    selected="active"
                                    required="true"
                                />
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <div class="form-group">
                                    <label for="description">Description</label>
                                    <textarea name="description" id="description" class="form-control" rows="3" placeholder="Optional description for this category"></textarea>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="button" onclick="saveCategory()" class="btn btn-primary">
                        <i class="material-icons mr-1" style="font-size: 16px;">save</i>
                        Save Category
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Import Categories Modal -->
    <div class="modal fade" id="importModal" tabindex="-1" role="dialog" aria-labelledby="importModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="importModalLabel">
                        <i class="material-icons mr-2">file_upload</i>
                        Import Item Categories
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <!-- Step 1: Download Template -->
                    <div class="card mb-3">
                        <div class="card-body py-3">
                            <div class="d-flex justify-content-between align-items-center mb-3">
                                <h6 class="mb-0">Step 1: Download Template</h6>
                                <button type="button" onclick="downloadTemplate()" class="btn btn-sm btn-outline-info">
                                    <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                                    Download Template
                                </button>
                            </div>
                            <small class="text-muted">Download the Excel template with the correct column format before uploading your category data.</small>
                        </div>
                    </div>

                    <hr>

                    <!-- Step 2: Upload File -->
                    <form id="importForm" enctype="multipart/form-data">
                        @csrf
                        <div class="form-group">
                            <h6 class="mb-3">Step 2: Upload Your File</h6>
                            <div class="custom-file">
                                <input type="file" name="file" class="custom-file-input" id="importFile" accept=".xlsx,.xls,.csv" required>
                                <label class="custom-file-label" for="importFile">Choose file...</label>
                            </div>
                            <small class="form-text text-muted mt-2">
                                <i class="material-icons" style="font-size: 14px;">info</i>
                                Supported formats: Excel (.xlsx, .xls) or CSV files. Maximum size: 10MB.
                            </small>
                        </div>

                        <!-- Progress Bar -->
                        <div class="progress mb-3" id="uploadProgress" style="display: none;">
                            <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%"></div>
                        </div>

                        <!-- Import Results -->
                        <div id="importResults" style="display: none;">
                            <div class="alert alert-info">
                                <h6><i class="material-icons mr-1">assessment</i> Import Results</h6>
                                <div id="importSummary"></div>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="button" onclick="importCategories()" class="btn btn-primary" id="importBtn">
                        <i class="material-icons mr-1" style="font-size: 16px;">cloud_upload</i>
                        Import Categories
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <style>
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
            border-radius: 4px;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: 600;
            text-transform: uppercase;
        }

        .status-active {
            background-color: #d4edda;
            color: #155724;
        }

        .status-inactive {
            background-color: #f8d7da;
            color: #721c24;
        }

        /* Modal z-index fix - simplified approach */
        .modal,
        #addCategoryModal,
        #importModal {
            z-index: 999999 !important;
        }

        .modal-backdrop {
            z-index: 999998 !important;
        }

        .modal-dialog {
            z-index: 1000000 !important;
            position: relative !important;
        }

        .modal-content {
            z-index: 1000001 !important;
            position: relative !important;
        }

        /* Ensure body doesn't scroll when modal is open */
        body.modal-open {
            overflow: hidden !important;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>

    <script>
        let categoriesTable;

        $(document).ready(function() {
            initializeDataTable();

            // Entries per page change handler
            $('#entries-per-page').change(function() {
                categoriesTable.page.len($(this).val()).draw();
            });
        });

        function initializeDataTable() {
            categoriesTable = $('#categories-table').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: '{{ route("warehouse.item-categories.index") }}',
                    data: function(d) {
                        d.status = $('#filter-status').val();
                        d.type_of_goods = $('#filter-type').val();
                    }
                },
                columns: [
                    {
                        data: null,
                        name: 'category_name',
                        render: function(data, type, row) {
                            return `
                                <div class="d-flex align-items-center">
                                    <div class="mr-3">
                                        <div class="bg-primary text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                            <i class="material-icons" style="font-size: 20px;">category</i>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="font-weight-medium">${row.category_name}</div>
                                        <small class="text-muted">${row.description || 'No description'}</small>
                                    </div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'type_of_goods',
                        name: 'type_of_goods',
                        render: function(data) {
                            return `<span class="badge badge-info">${data}</span>`;
                        }
                    },
                    {
                        data: 'status',
                        name: 'status',
                        render: function(data) {
                            const statusClasses = {
                                'active': 'status-active',
                                'inactive': 'status-inactive'
                            };
                            return `<span class="status-badge ${statusClasses[data] || 'status-inactive'}">${data.charAt(0).toUpperCase() + data.slice(1)}</span>`;
                        }
                    },
                    {
                        data: 'items_count',
                        name: 'items_count',
                        render: function(data) {
                            return `<span class="badge badge-secondary">${data} items</span>`;
                        }
                    },
                    {
                        data: null,
                        name: 'created_at',
                        render: function(data) {
                            return `
                                <div>
                                    <div class="small">${data.created_at}</div>
                                    <div class="small text-muted">by ${data.created_by}</div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'actions',
                        orderable: false,
                        searchable: false,
                        render: function(data) {
                            return `
                                <div class="btn-group" role="group">
                                    <a href="{{ url('warehouse/item-categories') }}/${data.id}" class="btn btn-sm btn-outline-primary" title="View">
                                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                                    </a>
                                    @can('warehouse.item-categories.edit')
                                    <a href="{{ url('warehouse/item-categories') }}/${data.id}/edit" class="btn btn-sm btn-outline-success" title="Edit">
                                        <i class="material-icons" style="font-size: 16px;">edit</i>
                                    </a>
                                    @endcan
                                    @can('warehouse.item-categories.delete')
                                    <button onclick="deleteCategory(${data.id})" class="btn btn-sm btn-outline-danger" title="Delete">
                                        <i class="material-icons" style="font-size: 16px;">delete</i>
                                    </button>
                                    @endcan
                                </div>
                            `;
                        }
                    }
                ],
                order: [[4, 'desc']],
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                language: {
                    processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
                    emptyTable: 'No item categories found',
                    zeroRecords: 'No matching categories found'
                },
                drawCallback: function(settings) {
                    $('#entries-per-page').val(settings._iDisplayLength);
                }
            });
        }

        function applyFilters() {
            categoriesTable.draw();
        }

        function clearFilters() {
            $('#filter-status, #filter-type').val('');
            categoriesTable.draw();
        }

        function deleteCategory(id) {
            if (confirm('Are you sure you want to delete this category?')) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: `{{ url('warehouse/item-categories') }}/${id}`,
                    type: 'DELETE',
                    success: function(response) {
                        if (response.success) {
                            categoriesTable.draw();
                            showAlert('Category deleted successfully', 'success');
                        } else {
                            showAlert(response.message, 'error');
                        }
                    },
                    error: function(xhr) {
                        showAlert('Error deleting category', 'error');
                    }
                });
            }
        }

        function showAddCategoryModal() {
            // Clear form and validation errors
            $('#addCategoryForm')[0].reset();
            clearValidationErrors();

            // Load item types for dropdown
            loadItemTypesDropdown();

            // Force very high z-index before showing
            $('#addCategoryModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Show modal
            $('#addCategoryModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Aggressively force z-index after modal is shown
            $('#addCategoryModal').on('shown.bs.modal', function() {
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 999998,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000,
                    'position': 'relative'
                });
                $('.modal-content').css({
                    'z-index': 1000001,
                    'position': 'relative'
                });

                // Move modal to body if it's not already there
                if (!$('#addCategoryModal').parent().is('body')) {
                    $('#addCategoryModal').appendTo('body');
                }
            });
        }

        function saveCategory() {
            const form = $('#addCategoryForm')[0];
            const formData = new FormData(form);

            // Clear previous validation errors
            clearValidationErrors();

            // Disable save button and show loading
            const saveBtn = $('button[onclick="saveCategory()"]');
            const originalHtml = saveBtn.html();
            saveBtn.prop('disabled', true);
            saveBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Saving...');

            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            });

            $.ajax({
                url: '{{ route("warehouse.item-categories.store") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        $('#addCategoryModal').modal('hide');
                        categoriesTable.draw();
                        showAlert('Category added successfully!', 'success');
                    } else {
                        showAlert(response.message || 'Error adding category', 'error');
                    }
                },
                error: function(xhr) {
                    if (xhr.status === 422) {
                        // Validation errors
                        const errors = xhr.responseJSON.errors;
                        displayValidationErrors(errors);
                    } else {
                        showAlert('Error adding category. Please try again.', 'error');
                    }
                },
                complete: function() {
                    // Re-enable save button
                    saveBtn.prop('disabled', false);
                    saveBtn.html(originalHtml);
                }
            });
        }

        function clearValidationErrors() {
            $('.form-control').removeClass('is-invalid');
            $('.invalid-feedback').text('');
        }

        function displayValidationErrors(errors) {
            $.each(errors, function(field, messages) {
                const input = $(`[name="${field}"]`);
                input.addClass('is-invalid');
                input.siblings('.invalid-feedback').text(messages[0]);
            });
        }

        function showAlert(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            `;
            $('body').append(alert);
            setTimeout(() => $('.alert').alert('close'), 5000);
        }

        function loadItemTypesDropdown() {
            $.ajax({
                url: '{{ route("warehouse.item-types.dropdown") }}',
                type: 'GET',
                success: function(response) {
                    if (response.success && response.data) {
                        const dropdown = $('#type_of_goods');
                        dropdown.empty();
                        dropdown.append('<option value="">Select Type</option>');

                        response.data.forEach(function(itemType) {
                            dropdown.append(`<option value="${itemType.type_name}">${itemType.type_name}</option>`);
                        });
                    }
                },
                error: function(xhr) {
                    console.error('Failed to load item types:', xhr);
                    showAlert('Failed to load item types for dropdown', 'error');
                }
            });
        }

        // Import functionality
        function showImportModal() {
            // Clear form and results
            $('#importForm')[0].reset();
            $('#importResults').hide();
            $('#uploadProgress').hide();
            $('.custom-file-label').text('Choose file...');

            // Force very high z-index before showing (same approach as add category)
            $('#importModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Show modal
            $('#importModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Aggressively force z-index after modal is shown (same as add category)
            $('#importModal').on('shown.bs.modal', function() {
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 999998,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000,
                    'position': 'relative'
                });
                $('.modal-content').css({
                    'z-index': 1000001,
                    'position': 'relative'
                });

                // Move modal to body if it's not already there
                if (!$('#importModal').parent().is('body')) {
                    $('#importModal').appendTo('body');
                }
            });
        }

        function downloadTemplate() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Downloading...';
            btn.disabled = true;

            // Create download link - use route helper with proper authentication context
            const downloadUrl = '{{ route("warehouse.item-categories.template") }}';

            console.log('🔗 Attempting to download template from:', downloadUrl);

            fetch(downloadUrl, {
                method: 'GET',
                credentials: 'include',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
                .then(response => {
                    console.log('📡 Response received:', response.status, response.statusText);
                    if (!response.ok) {
                        throw new Error('Template download failed');
                    }
                    return response.blob();
                })
                .then(blob => {
                    // Create download link
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = 'item_categories_import_template.xlsx';
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);

                    showAlert('Template downloaded successfully!', 'success');
                })
                .catch(error => {
                    console.error('💥 Download error:', error);
                    console.error('🔗 Failed URL:', downloadUrl);
                    showAlert('Failed to download template: ' + error.message, 'error');
                })
                .finally(() => {
                    btn.innerHTML = originalHtml;
                    btn.disabled = false;
                });
        }

        function importCategories() {
            const fileInput = $('#importFile')[0];

            if (!fileInput.files.length) {
                showAlert('Please select a file to import', 'error');
                return;
            }

            const formData = new FormData();
            formData.append('file', fileInput.files[0]);
            formData.append('_token', '{{ csrf_token() }}');

            // Show progress bar
            $('#uploadProgress').show();
            $('#importResults').hide();

            // Update import button
            const importBtn = $('#importBtn');
            const originalHtml = importBtn.html();
            importBtn.prop('disabled', true);
            importBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Importing...');

            $.ajax({
                url: '{{ route("warehouse.item-categories.import") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                xhr: function() {
                    var xhr = new window.XMLHttpRequest();
                    xhr.upload.addEventListener("progress", function(evt) {
                        if (evt.lengthComputable) {
                            var percentComplete = (evt.loaded / evt.total) * 100;
                            $('.progress-bar').css('width', percentComplete + '%');
                        }
                    }, false);
                    return xhr;
                },
                success: function(response) {
                    $('.progress-bar').css('width', '100%');

                    if (response.success) {
                        $('#importModal').modal('hide');
                        categoriesTable.draw();

                        let message = `Import completed successfully! `;
                        if (response.imported) message += `${response.imported} categories imported.`;
                        if (response.updated) message += ` ${response.updated} categories updated.`;
                        if (response.skipped) message += ` ${response.skipped} rows skipped.`;

                        showAlert(message, 'success');

                        // Reset form
                        document.getElementById('importForm').reset();
                        $('.custom-file-label').text('Choose file...');
                    } else {
                        showAlert(response.message || 'Import failed', 'error');
                    }
                },
                error: function(xhr) {
                    let errorMessage = 'Import failed. Please check your file and try again.';

                    if (xhr.status === 422) {
                        const response = xhr.responseJSON;
                        if (response.errors) {
                            errorMessage = Object.values(response.errors).flat().join(' ');
                        } else if (response.message) {
                            errorMessage = response.message;
                        }
                    } else if (xhr.status === 413) {
                        errorMessage = 'File is too large. Please reduce file size and try again.';
                    } else if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage = xhr.responseJSON.message;
                    }

                    showAlert(errorMessage, 'error');
                },
                complete: function() {
                    // Reset import button
                    importBtn.prop('disabled', false);
                    importBtn.html(originalHtml);

                    // Hide progress bar after delay
                    setTimeout(() => {
                        $('#uploadProgress').hide();
                        $('.progress-bar').css('width', '0%');
                    }, 1000);
                }
            });
        }

        // File input label update
        $(document).on('change', '#importFile', function() {
            const fileName = $(this)[0].files[0]?.name || 'Choose file...';
            $(this).next('.custom-file-label').text(fileName);
        });
    </script>
@endpush