@extends('layouts.admin-simple')

@section('title', 'Incoming Materials Report')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.index') }}">Reports</a></li>
    <li class="breadcrumb-item active">Incoming Materials</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Incoming Materials Report</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Supplier deliveries and purchase tracking</p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            <button type="button" onclick="exportReport()" class="btn btn-sm btn-success mr-2" title="Export to Excel">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>
            <button type="button" onclick="printReport()" class="btn btn-sm btn-info" title="Print Report">
                <i class="material-icons mr-1" style="font-size: 16px;">print</i>
                Print
            </button>
        </div>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Deliveries"
                :value="$incomingItems->count()"
                icon="local_shipping"
                color="primary"
                subtitle="All incoming deliveries"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Quantity"
                :value="number_format($incomingItems->sum('quantity'), 2)"
                icon="inventory_2"
                color="info"
                subtitle="Total quantity delivered"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Value"
                :value="'AED ' . number_format($incomingItems->sum('total_price'), 2)"
                icon="attach_money"
                color="success"
                subtitle="Total value received"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Active Suppliers"
                :value="$incomingItems->unique('supplier_code')->count()"
                icon="business"
                color="warning"
                subtitle="Number of suppliers"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('warehouse.reports.incoming') }}">
                <div class="row">
                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Search</label>
                        <input type="text" name="search" value="{{ request('search') }}" class="form-control" placeholder="Item code, description, LPO, or supplier">
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Supplier</label>
                        <select name="supplier_id" class="form-control">
                            <option value="">All Suppliers</option>
                            @foreach($suppliers as $supplier)
                                <option value="{{ $supplier->id }}" {{ request('supplier_id') == $supplier->id ? 'selected' : '' }}>
                                    {{ $supplier->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Division</label>
                        <select name="division_id" class="form-control">
                            <option value="">All Divisions</option>
                            @foreach($divisions as $division)
                                <option value="{{ $division->name }}" {{ request('division_id') == $division->name ? 'selected' : '' }}>
                                    {{ $division->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Category</label>
                        <select name="category_id" class="form-control">
                            <option value="">All Categories</option>
                            @foreach($categories as $category)
                                <option value="{{ $category->id }}" {{ request('category_id') == $category->id ? 'selected' : '' }}>
                                    {{ $category->category_name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Operation Type</label>
                        <select name="operation_type" class="form-control">
                            <option value="">All Types</option>
                            @foreach($operationTypes as $type)
                                <option value="{{ $type }}" {{ request('operation_type') == $type ? 'selected' : '' }}>
                                    {{ ucfirst(str_replace('_', ' ', $type)) }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Date From</label>
                        <input type="date" name="date_from" value="{{ request('date_from') }}" class="form-control">
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Date To</label>
                        <input type="date" name="date_to" value="{{ request('date_to') }}" class="form-control">
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3 d-flex align-items-end">
                        <div class="w-100">
                            <button type="submit" class="btn btn-primary btn-block">
                                <i class="material-icons mr-1" style="font-size: 16px;">search</i>
                                Apply Filters
                            </button>
                        </div>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3 d-flex align-items-end">
                        <div class="w-100">
                            <a href="{{ route('warehouse.reports.incoming') }}" class="btn btn-outline-secondary btn-block">
                                <i class="material-icons mr-1" style="font-size: 16px;">clear</i>
                                Clear Filters
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Incoming Materials Report Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">local_shipping</i>
                Incoming Materials Report ({{ $incomingItems->count() }} items)
            </h5>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-sm table-hover mb-0">
                    <thead class="bg-light">
                        <tr>
                            <th>Supplier Code</th>
                            <th>Supplier Name</th>
                            <th>LPO Number</th>
                            <th>Division</th>
                            <th>Item Code</th>
                            <th>Item Description</th>
                            <th>Unit</th>
                            <th>Quantity</th>
                            <th>Unit Price</th>
                            <th>Total Price</th>
                            <th>Production Date</th>
                            <th>Expiry Date</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($incomingItems as $item)
                            <tr>
                                <td>
                                    <span class="badge badge-outline-secondary">{{ $item->supplier_code }}</span>
                                </td>
                                <td>
                                    <div class="font-weight-medium">{{ $item->supplier_name }}</div>
                                </td>
                                <td>
                                    <span class="badge badge-outline-info">{{ $item->lpo_number }}</span>
                                </td>
                                <td>
                                    <span class="badge badge-light">{{ $item->division }}</span>
                                </td>
                                <td>
                                    <code>{{ $item->item_code }}</code>
                                </td>
                                <td>
                                    <div>{{ $item->item_description }}</div>
                                </td>
                                <td class="text-center">{{ $item->unit }}</td>
                                <td class="text-right">
                                    <strong>{{ number_format($item->quantity, 2) }}</strong>
                                </td>
                                <td class="text-right">
                                    <strong>AED {{ number_format($item->unit_price, 2) }}</strong>
                                </td>
                                <td class="text-right">
                                    <strong class="text-success">AED {{ number_format($item->total_price, 2) }}</strong>
                                </td>
                                <td class="text-center">
                                    {{ $item->production_date ? $item->production_date->format('d M, Y') : '-' }}
                                </td>
                                <td class="text-center">
                                    @if($item->expiry_date)
                                        <span class="{{ $item->expiry_date < now() ? 'text-danger' : ($item->expiry_date < now()->addDays(30) ? 'text-warning' : 'text-success') }}">
                                            {{ $item->expiry_date->format('d M, Y') }}
                                        </span>
                                    @else
                                        -
                                    @endif
                                </td>
                                <td class="text-center">
                                    <a href="{{ route('warehouse.reports.incoming.item-details', $item->id) }}" class="btn btn-sm btn-outline-primary" title="View detailed information">
                                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                                    </a>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="13" class="text-center py-4">
                                    <i class="material-icons text-muted mb-3" style="font-size: 48px;">local_shipping</i>
                                    <h5 class="text-muted">No Incoming Materials Found</h5>
                                    <p class="text-muted mb-0">No materials match your current filters.</p>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <style>
        .badge-outline-primary {
            color: #007bff;
            border: 1px solid #007bff;
            background-color: transparent;
        }

        .badge-outline-secondary {
            color: #6c757d;
            border: 1px solid #6c757d;
            background-color: transparent;
        }

        .badge-outline-info {
            color: #17a2b8;
            border: 1px solid #17a2b8;
            background-color: transparent;
        }

        .badge-outline-success {
            color: #28a745;
            border: 1px solid #28a745;
            background-color: transparent;
        }

        .badge-outline-warning {
            color: #ffc107;
            border: 1px solid #ffc107;
            background-color: transparent;
        }

        .badge-outline-danger {
            color: #dc3545;
            border: 1px solid #dc3545;
            background-color: transparent;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            font-size: 12px;
            white-space: nowrap;
        }

        .table td {
            vertical-align: middle;
            font-size: 13px;
        }

        .badge {
            font-size: 10px;
            padding: 3px 6px;
        }

        .small {
            font-size: 11px;
        }

        code {
            font-size: 12px;
            color: #6c757d;
            background-color: #f8f9fa;
            padding: 2px 6px;
            border-radius: 3px;
        }
    </style>
@endpush

@push('scripts')
    <script>
        function exportReport() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Exporting...';
            btn.disabled = true;

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                alert('Export functionality will be implemented');
            }, 1000);
        }

        function printReport() {
            window.print();
        }
    </script>
@endpush