@extends('layouts.admin-simple')

@section('title', 'Inventory Report')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.index') }}">Reports</a></li>
    <li class="breadcrumb-item active">Inventory Report</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Inventory Report</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Comprehensive inventory analysis with pricing insights</p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            <button type="button" onclick="exportReport()" class="btn btn-sm btn-success mr-2" title="Export to Excel">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>
            <button type="button" onclick="printReport()" class="btn btn-sm btn-info" title="Print Report">
                <i class="material-icons mr-1" style="font-size: 16px;">print</i>
                Print
            </button>
        </div>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Items"
                :value="$items->count()"
                icon="inventory"
                color="primary"
                subtitle="All inventory items"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Value"
                :value="$items->sum('total_value')"
                icon="account_balance_wallet"
                color="success"
                subtitle="Total inventory value (AED)"
                currency="true"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Low Stock Items"
                :value="$items->where('overall_status', 'low_stock')->count()"
                icon="warning"
                color="warning"
                subtitle="Items below reorder point"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Out of Stock"
                :value="$items->where('overall_status', 'out_of_stock')->count()"
                icon="error"
                color="danger"
                subtitle="Items completely out of stock"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('warehouse.reports.inventory') }}">
                <div class="row">
                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Search</label>
                        <input type="text" name="search" value="{{ request('search') }}" class="form-control" placeholder="Item code or description">
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Category</label>
                        <select name="category_id" class="form-control">
                            <option value="">All Categories</option>
                            @foreach($categories as $category)
                                <option value="{{ $category->id }}" {{ request('category_id') == $category->id ? 'selected' : '' }}>
                                    {{ $category->category_name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Division</label>
                        <select name="division_id" class="form-control">
                            <option value="">All Divisions</option>
                            @foreach($divisions as $division)
                                <option value="{{ $division->id }}" {{ request('division_id') == $division->id ? 'selected' : '' }}>
                                    {{ $division->name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Stock Status</label>
                        <select name="status" class="form-control">
                            <option value="">All Statuses</option>
                            @foreach($stockStatuses as $status)
                                <option value="{{ $status }}" {{ request('status') == $status ? 'selected' : '' }}>
                                    {{ ucfirst(str_replace('_', ' ', $status)) }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3">
                        <label class="form-label">Options</label>
                        <div class="form-check">
                            <input type="checkbox" name="exclude_zero_stock" value="1" class="form-check-input" id="exclude_zero_stock" {{ request('exclude_zero_stock') ? 'checked' : '' }}>
                            <label class="form-check-label" for="exclude_zero_stock">
                                Hide zero stock
                            </label>
                        </div>
                    </div>

                    <div class="col-xl-2 col-md-4 mb-3 d-flex align-items-end">
                        <div class="w-100">
                            <button type="submit" class="btn btn-primary btn-block">
                                <i class="material-icons mr-1" style="font-size: 16px;">search</i>
                                Apply Filters
                            </button>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Inventory Report Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">assessment</i>
                Inventory Report ({{ $items->count() }} items)
            </h5>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-sm table-hover mb-0">
                    <thead class="bg-light">
                        <tr>
                            <th>Item Code</th>
                            <th>Description</th>
                            <th>Category</th>
                            <th>Unit</th>
                            <th>Quantity</th>
                            <th>Avg Unit Price</th>
                            <th>Price Range</th>
                            <th>Total Value</th>
                            <th>Production Date</th>
                            <th>Expiry Date</th>
                            <th>Divisions</th>
                            <th>Stock Status</th>
                            <th class="text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($items as $item)
                            <tr>
                                <td>
                                    <div class="font-weight-medium">{{ $item->item_code }}</div>
                                </td>
                                <td>
                                    <div class="font-weight-medium">{{ $item->item_description }}</div>
                                </td>
                                <td>
                                    <span class="badge badge-outline-primary">{{ $item->category }}</span>
                                </td>
                                <td>{{ $item->unit_of_measure }}</td>
                                <td>
                                    <div class="text-right font-weight-medium">
                                        {{ number_format($item->total_quantity, 2) }}
                                    </div>
                                </td>
                                <td>
                                    <div class="text-right">
                                        <div class="font-weight-medium">AED {{ number_format($item->average_unit_price, 2) }}</div>
                                        <small class="text-muted">Current Average</small>
                                    </div>
                                </td>
                                <td>
                                    <div class="text-right">
                                        @if($item->min_price > 0 && $item->max_price > 0)
                                            <div class="small">
                                                <div><strong>Min:</strong> AED {{ number_format($item->min_price, 2) }}</div>
                                                <div><strong>Max:</strong> AED {{ number_format($item->max_price, 2) }}</div>
                                                <div class="text-muted"><strong>Avg Purchase:</strong> AED {{ number_format($item->avg_purchase_price, 2) }}</div>
                                            </div>
                                        @else
                                            <span class="text-muted">-</span>
                                        @endif
                                    </div>
                                </td>
                                <td>
                                    <div class="text-right">
                                        <div class="font-weight-medium">AED {{ number_format($item->total_value, 2) }}</div>
                                    </div>
                                </td>
                                <td>
                                    <div class="text-center">
                                        {{ $item->latest_production_date ? $item->latest_production_date->format('d M, Y') : '-' }}
                                    </div>
                                </td>
                                <td>
                                    <div class="text-center">
                                        @if($item->latest_expiry_date)
                                            @php
                                                $daysUntilExpiry = now()->diffInDays($item->latest_expiry_date, false);
                                            @endphp
                                            <div class="{{ $daysUntilExpiry < 0 ? 'text-danger' : ($daysUntilExpiry <= 30 ? 'text-warning' : '') }}">
                                                {{ $item->latest_expiry_date->format('d M, Y') }}
                                            </div>
                                            @if($daysUntilExpiry < 0)
                                                <small class="text-danger">Expired</small>
                                            @elseif($daysUntilExpiry <= 30)
                                                <small class="text-warning">{{ $daysUntilExpiry }} days left</small>
                                            @endif
                                        @else
                                            -
                                        @endif
                                    </div>
                                </td>
                                <td>
                                    <div class="small">
                                        @foreach($item->divisions as $division)
                                            <div>
                                                <span class="badge badge-light mr-1">{{ $division['name'] }}</span>
                                                <span class="text-muted">{{ number_format($division['quantity'], 2) }}</span>
                                            </div>
                                        @endforeach
                                    </div>
                                </td>
                                <td>
                                    <div class="text-center">
                                        <span class="badge badge-{{
                                            $item->overall_status === 'in_stock' ? 'success' :
                                            ($item->overall_status === 'low_stock' ? 'warning' : 'danger')
                                        }}">
                                            {{ ucfirst(str_replace('_', ' ', $item->overall_status)) }}
                                        </span>
                                        @if($item->stock_status_counts->count() > 0)
                                            <div class="small text-muted mt-1">
                                                @foreach($item->stock_status_counts as $status => $data)
                                                    <div>{{ ucfirst(str_replace('_', ' ', $status)) }}: {{ $data['count'] }}</div>
                                                @endforeach
                                            </div>
                                        @endif
                                    </div>
                                </td>
                                <td>
                                    <div class="text-center">
                                        <a href="{{ route('warehouse.reports.inventory.item-details', $item->id) }}" class="btn btn-sm btn-outline-primary" title="View detailed pricing and inventory breakdown">
                                            <i class="material-icons" style="font-size: 16px;">visibility</i>
                                        </a>
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="13" class="text-center py-4">
                                    <i class="material-icons text-muted mb-3" style="font-size: 48px;">inventory</i>
                                    <h5 class="text-muted">No Inventory Items Found</h5>
                                    <p class="text-muted mb-0">No items match your current filters.</p>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <style>
        .badge-outline-primary {
            color: #007bff;
            border: 1px solid #007bff;
            background-color: transparent;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            font-size: 12px;
            white-space: nowrap;
        }

        .table td {
            vertical-align: middle;
            font-size: 13px;
        }

        .badge {
            font-size: 10px;
            padding: 3px 6px;
        }

        .small {
            font-size: 11px;
        }
    </style>
@endpush

@push('scripts')
    <script>
        function exportReport() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Exporting...';
            btn.disabled = true;

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                alert('Export functionality will be implemented');
            }, 1000);
        }

        function printReport() {
            window.print();
        }
    </script>
@endpush