@extends('layouts.admin-simple')

@section('title', 'Supplier Delivery Operations')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item active">Supplier Delivery</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Supplier Delivery Operations</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Manage incoming deliveries from suppliers</p>
        </div>
        <div class="page-actions-right d-flex align-items-center">
            <button type="button" onclick="exportDeliveries()" class="btn btn-sm btn-info mr-2" title="Export deliveries to Excel">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>

            <a href="{{ route('warehouse.supplier-delivery.create') }}" class="btn btn-sm btn-primary" title="Add new supplier delivery">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                New Delivery
            </a>
        </div>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Deliveries"
                :value="$statistics['total_deliveries'] ?? 0"
                icon="local_shipping"
                color="primary"
                subtitle="All supplier deliveries"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Pending Delivery"
                :value="$statistics['pending_delivery'] ?? 0"
                icon="schedule"
                color="warning"
                subtitle="Awaiting delivery"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Completed Today"
                :value="$statistics['completed_today'] ?? 0"
                icon="check_circle"
                color="success"
                subtitle="Delivered today"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Value"
                :value="$statistics['total_value_aed'] ?? 0"
                icon="account_balance"
                color="info"
                subtitle="Total delivery value"
                currency="true"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-xl-2 col-md-4 mb-3">
                    <x-warehouse.select2-dropdown
                        name="filter-status"
                        id="filter-status"
                        label="Status"
                        placeholder="All Statuses"
                        :options="[
                            ['value' => '', 'text' => 'All Statuses'],
                            ['value' => 'draft', 'text' => 'Draft'],
                            ['value' => 'pending_delivery', 'text' => 'Pending Delivery'],
                            ['value' => 'delivered', 'text' => 'Delivered'],
                            ['value' => 'completed', 'text' => 'Completed'],
                            ['value' => 'cancelled', 'text' => 'Cancelled']
                        ]"
                        allowClear="true"
                    />
                </div>

                <div class="col-xl-2 col-md-4 mb-3">
                    @php
                        $supplierFilterOptions = [['value' => '', 'text' => 'All Suppliers']];
                        foreach($suppliers ?? [] as $supplier) {
                            $supplierFilterOptions[] = [
                                'value' => $supplier->id,
                                'text' => $supplier->name
                            ];
                        }
                    @endphp
                    <x-warehouse.select2-dropdown
                        name="filter-supplier"
                        id="filter-supplier"
                        label="Supplier"
                        placeholder="All Suppliers"
                        :options="$supplierFilterOptions"
                        allowClear="true"
                    />
                </div>

                <div class="col-xl-2 col-md-4 mb-3">
                    <x-warehouse.select2-dropdown
                        name="filter-division"
                        id="filter-division"
                        label="Division"
                        placeholder="All Divisions"
                        :options="[
                            ['value' => '', 'text' => 'All Divisions'],
                            ['value' => 'Construction', 'text' => 'Construction'],
                            ['value' => 'Engineering', 'text' => 'Engineering'],
                            ['value' => 'Operations', 'text' => 'Operations'],
                            ['value' => 'Maintenance', 'text' => 'Maintenance']
                        ]"
                        allowClear="true"
                    />
                </div>

                <div class="col-xl-2 col-md-4 mb-3">
                    <x-warehouse.select2-dropdown
                        name="filter-lpo-status"
                        id="filter-lpo-status"
                        label="LPO Status"
                        placeholder="All LPO Status"
                        :options="[
                            ['value' => '', 'text' => 'All LPO Status'],
                            ['value' => 'approved', 'text' => 'Approved'],
                            ['value' => 'pending', 'text' => 'Pending'],
                            ['value' => 'cancelled', 'text' => 'Cancelled']
                        ]"
                        allowClear="true"
                    />
                </div>

                <div class="col-xl-2 col-md-4 mb-3">
                    <label class="form-label">Date Range</label>
                    <div class="d-flex">
                        <input type="date" id="filter-date-from" class="form-control mr-2">
                        <input type="date" id="filter-date-to" class="form-control">
                    </div>
                </div>

                <div class="col-xl-2 col-md-4 mb-3">
                    <label class="form-label d-flex align-items-center">
                        Amount Range
                        <span class="dirham-symbol text-muted ml-1" style="width: 12px; height: 12px; display: inline-flex; align-items: center;">
                            <svg viewBox="0 0 1000 870" width="12" height="12" style="fill: currentColor;">
                                <path d="m88.3 1c0.4 0.6 2.6 3.3 4.7 5.9 15.3 18.2 26.8 47.8 33 85.1 4.1 24.5 4.3 32.2 4.3 125.6v87h-41.8c-38.2 0-42.6-0.2-50.1-1.7-11.8-2.5-24-9.2-32.2-17.8-6.5-6.9-6.3-7.3-5.9 13.6 0.5 17.3 0.7 19.2 3.2 28.6 4 14.9 9.5 26 17.8 35.9 11.3 13.6 22.8 21.2 39.2 26.3 3.5 1 10.9 1.4 37.1 1.6l32.7 0.5v43.3 43.4l-46.1-0.3-46.3-0.3-8-3.2c-9.5-3.8-13.8-6.6-23.1-14.9l-6.8-6.1 0.4 19.1c0.5 17.7 0.6 19.7 3.1 28.7 8.7 31.8 29.7 54.5 57.4 61.9 6.9 1.9 9.6 2 38.5 2.4l30.9 0.4v89.6c0 54.1-0.3 94-0.8 100.8-0.5 6.2-2.1 17.8-3.5 25.9-6.5 37.3-18.2 65.4-35 83.6l-3.4 3.7h169.1c101.1 0 176.7-0.4 187.8-0.9 19.5-1 63-5.3 72.8-7.4 3.1-0.6 8.9-1.5 12.7-2.1 8.1-1.2 21.5-4 40.8-8.9 27.2-6.8 52-15.3 76.3-26.1 7.6-3.4 29.4-14.5 35.2-18 3.1-1.8 6.8-4 8.2-4.7 3.9-2.1 10.4-6.3 19.9-13.1 4.7-3.4 9.4-6.7 10.4-7.4 4.2-2.8 18.7-14.9 25.3-21 25.1-23.1 46.1-48.8 62.4-76.3 2.3-4 5.3-9 6.6-11.1 3.3-5.6 16.9-33.6 18.2-37.8 0.6-1.9 1.4-3.9 1.8-4.3 2.6-3.4 17.6-50.6 19.4-60.9 0.6-3.3 0.9-3.8 3.4-4.3 1.6-0.3 24.9-0.3 51.8-0.1 53.8 0.4 53.8 0.4 65.7 5.9 6.7 3.1 8.7 4.5 16.1 11.2 9.7 8.7 8.8 10.1 8.2-11.7-0.4-12.8-0.9-20.7-1.8-23.9-3.4-12.3-4.2-14.9-7.2-21.1-9.8-21.4-26.2-36.7-47.2-44l-8.2-3-33.4-0.4-33.3-0.5 0.4-11.7c0.4-15.4 0.4-45.9-0.1-61.6l-0.4-12.6 44.6-0.2c38.2-0.2 45.3 0 49.5 1.1 12.6 3.5 21.1 8.3 31.5 17.8l5.8 5.4v-14.8c0-17.6-0.9-25.4-4.5-37-7.1-23.5-21.1-41-41.1-51.8-13-7-13.8-7.2-58.5-7.5-26.2-0.2-39.9-0.6-40.6-1.2-0.6-0.6-1.1-1.6-1.1-2.4 0-0.8-1.5-7.1-3.5-13.9-23.4-82.7-67.1-148.4-131-197.1-8.7-6.7-30-20.8-38.6-25.6-3.3-1.9-6.9-3.9-7.8-4.5-4.2-2.3-28.3-14.1-34.3-16.6-3.6-1.6-8.3-3.6-10.4-4.4-35.3-15.3-94.5-29.8-139.7-34.3-7.4-0.7-17.2-1.8-21.7-2.2-20.4-2.3-48.7-2.6-209.4-2.6-135.8 0-169.9 0.3-169.4 1zm330.7 43.3c33.8 2 54.6 4.6 78.9 10.5 74.2 17.6 126.4 54.8 164.3 117 3.5 5.8 18.3 36 20.5 42.1 10.5 28.3 15.6 45.1 20.1 67.3 1.1 5.4 2.6 12.6 3.3 16 0.7 3.3 1 6.4 0.7 6.7-0.5 0.4-100.9 0.6-223.3 0.5l-222.5-0.2-0.3-128.5c-0.1-70.6 0-129.3 0.3-130.4l0.4-1.9h71.1c39 0 78 0.4 86.5 0.9zm297.5 350.3c0.7 4.3 0.7 77.3 0 80.9l-0.6 2.7-227.5-0.2-227.4-0.3-0.2-42.4c-0.2-23.3 0-42.7 0.2-43.1 0.3-0.5 97.2-0.8 227.7-0.8h227.2zm-10.2 171.7c0.5 1.5-1.9 13.8-6.8 33.8-5.6 22.5-13.2 45.2-20.9 62-3.8 8.6-13.3 27.2-15.6 30.7-1.1 1.6-4.3 6.7-7.1 11.2-18 28.2-43.7 53.9-73 72.9-10.7 6.8-32.7 18.4-38.6 20.2-1.2 0.3-2.5 0.9-3 1.3-0.7 0.6-9.8 4-20.4 7.8-19.5 6.9-56.6 14.4-86.4 17.5-19.3 1.9-22.4 2-96.7 2h-76.9v-129.7-129.8l220.9-0.4c121.5-0.2 221.6-0.5 222.4-0.7 0.9-0.1 1.8 0.5 2.1 1.2z"/>
                            </svg>
                        </span>
                    </label>
                    <div class="d-flex">
                        <input type="number" id="filter-amount-min" placeholder="Min" class="form-control mr-2">
                        <input type="number" id="filter-amount-max" placeholder="Max" class="form-control">
                    </div>
                </div>
            </div>

            <div class="d-flex justify-content-between align-items-center mt-3 pt-3 border-top">
                <button type="button" onclick="applyFilters()" class="btn btn-primary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">filter_list</i>
                    Apply Filters
                </button>

                <button type="button" onclick="clearFilters()" class="btn btn-outline-secondary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">clear</i>
                    Clear Filters
                </button>
            </div>
        </div>
    </div>

    <!-- Supplier Deliveries Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">list</i>
                Supplier Deliveries
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25" selected>25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="deliveries-table" class="table table-hover">
                    <thead>
                        <tr>
                            <th>Operation Details</th>
                            <th>Supplier</th>
                            <th>LPO Information</th>
                            <th>Division</th>
                            <th>Delivery Date</th>
                            <th>Status</th>
                            <th>Items Count</th>
                            <th>Total Amount</th>
                            <th>Received By</th>
                            <th class="text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Data will be populated dynamically -->
                        @if(isset($deliveries) && $deliveries->count() > 0)
                            @foreach($deliveries as $delivery)
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <div class="mr-3">
                                                <div class="bg-primary text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                                    <i class="material-icons" style="font-size: 18px;">local_shipping</i>
                                                </div>
                                            </div>
                                            <div>
                                                <div class="font-weight-medium">{{ $delivery->operation_number }}</div>
                                                <small class="text-muted">{{ $delivery->delivery_note_number ?? 'N/A' }}</small>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <div>
                                            <div class="font-weight-medium">{{ $delivery->supplier->name ?? 'N/A' }}</div>
                                            <small class="text-muted">{{ $delivery->supplier->supplier_code ?? 'N/A' }}</small>
                                        </div>
                                    </td>
                                    <td>
                                        <div>
                                            <div class="font-weight-medium">{{ $delivery->lpo_number ?? 'N/A' }}</div>
                                            <small class="text-muted">{{ ucfirst($delivery->lpo_status ?? 'pending') }}</small>
                                        </div>
                                    </td>
                                    <td>{{ $delivery->division ?? 'N/A' }}</td>
                                    <td>{{ $delivery->operation_date ? $delivery->operation_date->format('d/m/Y') : 'N/A' }}</td>
                                    <td>
                                        <span class="badge badge-{{
                                            $delivery->status === 'completed' ? 'success' :
                                            ($delivery->status === 'delivered' ? 'info' :
                                            ($delivery->status === 'pending_delivery' ? 'warning' : 'secondary'))
                                        }}">
                                            {{ ucfirst(str_replace('_', ' ', $delivery->status)) }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-outline-primary">{{ $delivery->items->count() }} items</span>
                                    </td>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            <span class="dirham-symbol text-dark mr-1" style="width: 16px; height: 16px; display: inline-flex; align-items: center;">
                                                <svg viewBox="0 0 1000 870" width="16" height="16" style="fill: currentColor;">
                                                    <path d="m88.3 1c0.4 0.6 2.6 3.3 4.7 5.9 15.3 18.2 26.8 47.8 33 85.1 4.1 24.5 4.3 32.2 4.3 125.6v87h-41.8c-38.2 0-42.6-0.2-50.1-1.7-11.8-2.5-24-9.2-32.2-17.8-6.5-6.9-6.3-7.3-5.9 13.6 0.5 17.3 0.7 19.2 3.2 28.6 4 14.9 9.5 26 17.8 35.9 11.3 13.6 22.8 21.2 39.2 26.3 3.5 1 10.9 1.4 37.1 1.6l32.7 0.5v43.3 43.4l-46.1-0.3-46.3-0.3-8-3.2c-9.5-3.8-13.8-6.6-23.1-14.9l-6.8-6.1 0.4 19.1c0.5 17.7 0.6 19.7 3.1 28.7 8.7 31.8 29.7 54.5 57.4 61.9 6.9 1.9 9.6 2 38.5 2.4l30.9 0.4v89.6c0 54.1-0.3 94-0.8 100.8-0.5 6.2-2.1 17.8-3.5 25.9-6.5 37.3-18.2 65.4-35 83.6l-3.4 3.7h169.1c101.1 0 176.7-0.4 187.8-0.9 19.5-1 63-5.3 72.8-7.4 3.1-0.6 8.9-1.5 12.7-2.1 8.1-1.2 21.5-4 40.8-8.9 27.2-6.8 52-15.3 76.3-26.1 7.6-3.4 29.4-14.5 35.2-18 3.1-1.8 6.8-4 8.2-4.7 3.9-2.1 10.4-6.3 19.9-13.1 4.7-3.4 9.4-6.7 10.4-7.4 4.2-2.8 18.7-14.9 25.3-21 25.1-23.1 46.1-48.8 62.4-76.3 2.3-4 5.3-9 6.6-11.1 3.3-5.6 16.9-33.6 18.2-37.8 0.6-1.9 1.4-3.9 1.8-4.3 2.6-3.4 17.6-50.6 19.4-60.9 0.6-3.3 0.9-3.8 3.4-4.3 1.6-0.3 24.9-0.3 51.8-0.1 53.8 0.4 53.8 0.4 65.7 5.9 6.7 3.1 8.7 4.5 16.1 11.2 9.7 8.7 8.8 10.1 8.2-11.7-0.4-12.8-0.9-20.7-1.8-23.9-3.4-12.3-4.2-14.9-7.2-21.1-9.8-21.4-26.2-36.7-47.2-44l-8.2-3-33.4-0.4-33.3-0.5 0.4-11.7c0.4-15.4 0.4-45.9-0.1-61.6l-0.4-12.6 44.6-0.2c38.2-0.2 45.3 0 49.5 1.1 12.6 3.5 21.1 8.3 31.5 17.8l5.8 5.4v-14.8c0-17.6-0.9-25.4-4.5-37-7.1-23.5-21.1-41-41.1-51.8-13-7-13.8-7.2-58.5-7.5-26.2-0.2-39.9-0.6-40.6-1.2-0.6-0.6-1.1-1.6-1.1-2.4 0-0.8-1.5-7.1-3.5-13.9-23.4-82.7-67.1-148.4-131-197.1-8.7-6.7-30-20.8-38.6-25.6-3.3-1.9-6.9-3.9-7.8-4.5-4.2-2.3-28.3-14.1-34.3-16.6-3.6-1.6-8.3-3.6-10.4-4.4-35.3-15.3-94.5-29.8-139.7-34.3-7.4-0.7-17.2-1.8-21.7-2.2-20.4-2.3-48.7-2.6-209.4-2.6-135.8 0-169.9 0.3-169.4 1zm330.7 43.3c33.8 2 54.6 4.6 78.9 10.5 74.2 17.6 126.4 54.8 164.3 117 3.5 5.8 18.3 36 20.5 42.1 10.5 28.3 15.6 45.1 20.1 67.3 1.1 5.4 2.6 12.6 3.3 16 0.7 3.3 1 6.4 0.7 6.7-0.5 0.4-100.9 0.6-223.3 0.5l-222.5-0.2-0.3-128.5c-0.1-70.6 0-129.3 0.3-130.4l0.4-1.9h71.1c39 0 78 0.4 86.5 0.9zm297.5 350.3c0.7 4.3 0.7 77.3 0 80.9l-0.6 2.7-227.5-0.2-227.4-0.3-0.2-42.4c-0.2-23.3 0-42.7 0.2-43.1 0.3-0.5 97.2-0.8 227.7-0.8h227.2zm-10.2 171.7c0.5 1.5-1.9 13.8-6.8 33.8-5.6 22.5-13.2 45.2-20.9 62-3.8 8.6-13.3 27.2-15.6 30.7-1.1 1.6-4.3 6.7-7.1 11.2-18 28.2-43.7 53.9-73 72.9-10.7 6.8-32.7 18.4-38.6 20.2-1.2 0.3-2.5 0.9-3 1.3-0.7 0.6-9.8 4-20.4 7.8-19.5 6.9-56.6 14.4-86.4 17.5-19.3 1.9-22.4 2-96.7 2h-76.9v-129.7-129.8l220.9-0.4c121.5-0.2 221.6-0.5 222.4-0.7 0.9-0.1 1.8 0.5 2.1 1.2z"/>
                                                </svg>
                                            </span>
                                            <div>
                                                <div class="font-weight-medium text-dark">{{ number_format($delivery->total_amount_aed ?? 0, 2) }}</div>
                                                @if($delivery->currency !== 'AED')
                                                    <small class="text-muted">{{ $delivery->currency }} {{ number_format($delivery->total_amount ?? 0, 2) }}</small>
                                                @endif
                                            </div>
                                        </div>
                                    </td>
                                    <td>{{ $delivery->received_by_name ?? 'N/A' }}</td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('warehouse.incoming-operations.show', $delivery->id) }}" class="btn btn-sm btn-outline-primary" title="View">
                                                <i class="material-icons" style="font-size: 16px;">visibility</i>
                                            </a>
                                            <a href="{{ route('warehouse.incoming-operations.edit', $delivery->id) }}" class="btn btn-sm btn-outline-success" title="Edit">
                                                <i class="material-icons" style="font-size: 16px;">edit</i>
                                            </a>
                                            <button onclick="printDeliveryNote({{ $delivery->id }})" class="btn btn-sm btn-outline-info" title="Print">
                                                <i class="material-icons" style="font-size: 16px;">print</i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        @else
                            <tr>
                                <td colspan="10" class="text-center py-4">
                                    <i class="material-icons text-muted mb-3" style="font-size: 48px;">local_shipping</i>
                                    <h5 class="text-muted">No Supplier Deliveries Found</h5>
                                    <p class="text-muted mb-4">No supplier deliveries match your current filters.</p>
                                    <a href="{{ route('warehouse.supplier-delivery.create') }}" class="btn btn-primary">
                                        <i class="material-icons mr-1">add</i>
                                        Create First Delivery
                                    </a>
                                </td>
                            </tr>
                        @endif
                    </tbody>
                </table>
            </div>

            <!-- Pagination -->
            @if(isset($deliveries) && $deliveries->hasPages())
                <div class="d-flex justify-content-center mt-4">
                    {{ $deliveries->links() }}
                </div>
            @endif
        </div>
    </div>

    <!-- Quick Statistics -->
    <div class="row mt-4">
        <div class="col-lg-6">
            <div class="card">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="material-icons mr-2">trending_up</i>
                        Recent Activity
                    </h6>
                </div>
                <div class="card-body">
                    <div class="list-group list-group-flush">
                        <div class="list-group-item d-flex align-items-center px-0">
                            <div class="mr-3">
                                <div class="bg-success text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 32px; height: 32px;">
                                    <i class="material-icons" style="font-size: 16px;">check</i>
                                </div>
                            </div>
                            <div class="flex-grow-1">
                                <h6 class="mb-0">Today's Completions</h6>
                                <small class="text-muted">{{ $statistics['completed_today'] ?? 0 }} deliveries completed</small>
                            </div>
                        </div>
                        <div class="list-group-item d-flex align-items-center px-0">
                            <div class="mr-3">
                                <div class="bg-warning text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 32px; height: 32px;">
                                    <i class="material-icons" style="font-size: 16px;">schedule</i>
                                </div>
                            </div>
                            <div class="flex-grow-1">
                                <h6 class="mb-0">Pending Deliveries</h6>
                                <small class="text-muted">{{ $statistics['pending_delivery'] ?? 0 }} awaiting delivery</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-lg-6">
            <div class="card">
                <div class="card-header">
                    <h6 class="card-title mb-0">
                        <i class="material-icons mr-2">business</i>
                        Top Suppliers
                    </h6>
                </div>
                <div class="card-body">
                    <div class="list-group list-group-flush">
                        @if(isset($topSuppliers))
                            @foreach($topSuppliers->take(3) as $supplier)
                                <div class="list-group-item d-flex align-items-center px-0">
                                    <div class="mr-3">
                                        <div class="bg-info text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 32px; height: 32px;">
                                            <span style="font-size: 12px; font-weight: bold;">{{ substr($supplier->name, 0, 1) }}</span>
                                        </div>
                                    </div>
                                    <div class="flex-grow-1">
                                        <h6 class="mb-0">{{ $supplier->name }}</h6>
                                        <small class="text-muted">{{ $supplier->deliveries_count }} deliveries</small>
                                    </div>
                                </div>
                            @endforeach
                        @else
                            <div class="text-center py-3">
                                <small class="text-muted">No supplier data available</small>
                            </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <style>
        /* Custom styles specific to supplier delivery page */

        .badge-outline-primary {
            color: #007bff;
            border: 1px solid #007bff;
            background-color: transparent;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }
    </style>
@endpush

@push('scripts')
    <script>
        function applyFilters() {
            // Implement filtering logic
            console.log('Applying filters...');
            // You can implement AJAX filtering here
        }

        function clearFilters() {
            $('#filter-status, #filter-supplier, #filter-division, #filter-lpo-status, #filter-date-from, #filter-date-to, #filter-amount-min, #filter-amount-max').val('');
            applyFilters();
        }

        function exportDeliveries() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Exporting...';
            btn.disabled = true;

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                alert('Export functionality will be implemented');
            }, 1000);
        }

        function printDeliveryNote(deliveryId) {
            window.open(`{{ url('warehouse/incoming-operations') }}/${deliveryId}/print`, '_blank');
        }

        // Auto-refresh every 5 minutes
        setInterval(() => {
            window.location.reload();
        }, 300000);
    </script>
@endpush