@extends('layouts.admin-simple')

@section('title', 'Suppliers')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item active">Suppliers</li>
@endsection

@section('page-title')
    <div class="d-flex justify-content-between align-items-center w-100">
        <div class="page-title-left">
            <h3 class="mb-1">Suppliers</h3>
            <p class="text-muted mb-0" style="font-size: 13px;">Manage supplier information and performance</p>
        </div>
        @can('warehouse.suppliers.create')
        <div class="page-actions-right d-flex align-items-center">
            <button type="button" onclick="showImportModal()" class="btn btn-sm btn-success mr-2" title="Import suppliers from Excel file">
                <i class="material-icons mr-1" style="font-size: 16px;">upload_file</i>
                Import
            </button>

            <button type="button" onclick="exportSuppliers()" class="btn btn-sm btn-info mr-2" title="Export suppliers to Excel file">
                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                Export
            </button>

            <button type="button" onclick="showAddSupplierModal()" class="btn btn-sm btn-primary" title="Add new supplier to the system">
                <i class="material-icons mr-1" style="font-size: 16px;">add</i>
                Add Supplier
            </button>
        </div>
        @endcan
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Total Suppliers"
                :value="$statistics['total_suppliers'] ?? 0"
                icon="business"
                color="primary"
                subtitle="All registered suppliers"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Active Suppliers"
                :value="$statistics['active_suppliers'] ?? 0"
                icon="check_circle"
                color="success"
                subtitle="Currently active suppliers"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Average Rating"
                :value="number_format($statistics['average_rating'] ?? 0, 1)"
                icon="star"
                color="warning"
                subtitle="Overall supplier rating"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <x-warehouse.stats-card
                title="Top Rated Suppliers"
                :value="$statistics['top_rated_suppliers'] ?? 0"
                icon="trending_up"
                color="info"
                subtitle="5-star rated suppliers"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <div class="row">
                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Status</label>
                    <select id="filter-status" class="form-control">
                        <option value="">All Statuses</option>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                        <option value="suspended">Suspended</option>
                    </select>
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Country</label>
                    <select id="filter-country" class="form-control">
                        <option value="">All Countries</option>
                        @foreach($filterOptions['countries'] ?? [] as $country)
                            <option value="{{ $country }}">{{ $country }}</option>
                        @endforeach
                    </select>
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">City</label>
                    <select id="filter-city" class="form-control">
                        <option value="">All Cities</option>
                        @foreach($filterOptions['cities'] ?? [] as $city)
                            <option value="{{ $city }}">{{ $city }}</option>
                        @endforeach
                    </select>
                </div>

                <div class="col-xl-3 col-md-6 mb-3">
                    <label class="form-label">Rating Range</label>
                    <div class="d-flex">
                        <input type="number" id="filter-rating-min" placeholder="Min" min="0" max="5" step="0.1" class="form-control mr-2">
                        <input type="number" id="filter-rating-max" placeholder="Max" min="0" max="5" step="0.1" class="form-control">
                    </div>
                </div>
            </div>

            <div class="d-flex justify-content-between align-items-center mt-3 pt-3 border-top">
                <button type="button" onclick="applyFilters()" class="btn btn-primary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">filter_list</i>
                    Apply Filters
                </button>

                <button type="button" onclick="clearFilters()" class="btn btn-outline-secondary">
                    <i class="material-icons mr-1" style="font-size: 18px; vertical-align: middle;">clear</i>
                    Clear Filters
                </button>
            </div>
        </div>
    </div>

    <!-- Suppliers Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0 d-flex align-items-center">
                <i class="material-icons mr-2">list</i>
                Suppliers List
            </h5>
            <div class="d-flex align-items-center">
                <label class="text-muted mr-2" style="font-size: 14px;">Show:</label>
                <select id="entries-per-page" class="form-control form-control-sm" style="width: auto;">
                    <option value="10">10</option>
                    <option value="25">25</option>
                    <option value="50">50</option>
                    <option value="100">100</option>
                </select>
                <span class="text-muted ml-2" style="font-size: 14px;">entries</span>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table id="suppliers-table" class="table table-hover">
                    <thead>
                        <tr>
                            <th>Supplier</th>
                            <th>Contact</th>
                            <th>Location</th>
                            <th>Status</th>
                            <th>Rating</th>
                            <th>Performance</th>
                            <th>Payment Terms</th>
                            <th>Created</th>
                            <th class="text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Data will be populated by DataTables -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Import Modal -->
    <div class="modal fade" id="importModal" tabindex="-1" role="dialog" aria-labelledby="importModalLabel" aria-hidden="true">
        <div class="modal-dialog" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="importModalLabel">
                        <i class="material-icons mr-2">upload_file</i>
                        Import Suppliers
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <div class="d-flex justify-content-between align-items-center mb-3">
                            <h6 class="mb-0">Step 1: Download Template</h6>
                            <button type="button" onclick="downloadTemplate()" class="btn btn-sm btn-outline-info">
                                <i class="material-icons mr-1" style="font-size: 16px;">file_download</i>
                                Download Template
                            </button>
                        </div>
                        <small class="text-muted">Download the Excel template with the correct column format before uploading your supplier data.</small>
                    </div>

                    <hr>

                    <form id="importForm" enctype="multipart/form-data">
                        @csrf
                        <h6 class="mb-3">Step 2: Upload Your File</h6>
                        <div class="form-group">
                            <label for="importFile">Select Excel File <span class="text-danger">*</span></label>
                            <div class="custom-file">
                                <input type="file" name="file" id="importFile" accept=".xlsx,.xls,.csv" required class="custom-file-input">
                                <label class="custom-file-label" for="importFile">Choose file...</label>
                            </div>
                            <small class="form-text text-muted">
                                Supported formats: Excel (.xlsx, .xls) or CSV. Maximum file size: 5MB
                            </small>
                        </div>

                        <div class="alert alert-info">
                            <h6><i class="material-icons mr-1" style="font-size: 16px;">info</i>Expected Columns:</h6>
                            <div class="row">
                                <div class="col-md-6">
                                    <small>
                                        • <strong>name*</strong> - Supplier name<br>
                                        • <strong>contact_person</strong> - Contact person name<br>
                                        • <strong>email</strong> - Email address<br>
                                        • <strong>phone</strong> - Phone number<br>
                                        • <strong>address</strong> - Street address
                                    </small>
                                </div>
                                <div class="col-md-6">
                                    <small>
                                        • <strong>city</strong> - City name<br>
                                        • <strong>state</strong> - State/Province<br>
                                        • <strong>country</strong> - Country name<br>
                                        • <strong>postal_code</strong> - Postal code<br>
                                        • <strong>payment_terms</strong> - Payment terms
                                    </small>
                                </div>
                            </div>
                            <small class="text-danger mt-2 d-block">* Required fields</small>
                        </div>

                        <div id="import-progress" class="d-none">
                            <div class="progress mb-3">
                                <div class="progress-bar progress-bar-striped progress-bar-animated" role="progressbar" style="width: 0%"></div>
                            </div>
                            <div class="text-center">
                                <small class="text-muted">Processing your import...</small>
                            </div>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="button" onclick="performImport()" class="btn btn-primary">
                        <i class="material-icons mr-1" style="font-size: 16px;">upload</i>
                        Import
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Supplier Modal -->
    <div class="modal fade" id="addSupplierModal" tabindex="-1" role="dialog" aria-labelledby="addSupplierModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addSupplierModalLabel">
                        <i class="material-icons mr-2">business</i>
                        Add New Supplier
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <form id="addSupplierForm">
                        @csrf
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="supplier_name">Supplier Name <span class="text-danger">*</span></label>
                                    <input type="text" name="name" id="supplier_name" class="form-control" required>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="supplier_code">Supplier Code</label>
                                    <input type="text" name="supplier_code" id="supplier_code" class="form-control">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="contact_person">Contact Person</label>
                                    <input type="text" name="contact_person" id="contact_person" class="form-control">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="email">Email</label>
                                    <input type="email" name="email" id="email" class="form-control">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="phone">Phone</label>
                                    <input type="text" name="phone" id="phone" class="form-control">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="status">Status</label>
                                    <select name="status" id="status" class="form-control">
                                        <option value="active">Active</option>
                                        <option value="inactive">Inactive</option>
                                        <option value="suspended">Suspended</option>
                                    </select>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="address">Address</label>
                            <textarea name="address" id="address" class="form-control" rows="2"></textarea>
                            <div class="invalid-feedback"></div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="city">City</label>
                                    <input type="text" name="city" id="city" class="form-control">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="state">State/Province</label>
                                    <input type="text" name="state" id="state" class="form-control">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="country">Country</label>
                                    <input type="text" name="country" id="country" class="form-control">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="postal_code">Postal Code</label>
                                    <input type="text" name="postal_code" id="postal_code" class="form-control">
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="payment_terms">Payment Terms</label>
                                    <select name="payment_terms" id="payment_terms" class="form-control">
                                        <option value="">Select Payment Terms</option>
                                        <option value="Net 15">Net 15</option>
                                        <option value="Net 30">Net 30</option>
                                        <option value="Net 45">Net 45</option>
                                        <option value="Net 60">Net 60</option>
                                        <option value="COD">Cash on Delivery</option>
                                        <option value="Advance">Advance Payment</option>
                                    </select>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="rating">Rating</label>
                                    <select name="rating" id="rating" class="form-control">
                                        <option value="">Select Rating</option>
                                        <option value="1">1 Star</option>
                                        <option value="2">2 Stars</option>
                                        <option value="3">3 Stars</option>
                                        <option value="4">4 Stars</option>
                                        <option value="5">5 Stars</option>
                                    </select>
                                    <div class="invalid-feedback"></div>
                                </div>
                            </div>
                        </div>

                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="button" onclick="saveSupplier()" class="btn btn-primary">
                        <i class="material-icons mr-1" style="font-size: 16px;">save</i>
                        Save Supplier
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap4.min.css">
    <style>
        .btn-sm {
            padding: 6px 12px;
            font-size: 12px;
            line-height: 1.5;
            border-radius: 4px;
        }

        .btn-sm .material-icons {
            font-size: 16px !important;
        }

        .table th {
            border-top: none;
            font-weight: 600;
            color: #495057;
            background-color: #f8f9fa;
        }

        .table-responsive {
            border-radius: 0.375rem;
        }

        .status-badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.75rem;
            font-size: 0.75rem;
            font-weight: 500;
            line-height: 1;
            border-radius: 0.375rem;
        }

        .status-active {
            color: #155724;
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
        }

        .status-inactive {
            color: #721c24;
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
        }

        .status-suspended {
            color: #856404;
            background-color: #fff3cd;
            border: 1px solid #ffeaa7;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        /* Modal z-index fix - simplified approach */
        .modal,
        #addSupplierModal,
        #importModal {
            z-index: 999999 !important;
        }

        .modal-backdrop {
            z-index: 999998 !important;
        }

        .modal-dialog {
            z-index: 1000000 !important;
            position: relative !important;
        }

        .modal-content {
            z-index: 1000001 !important;
            position: relative !important;
        }

        /* Ensure body doesn't scroll when modal is open */
        body.modal-open {
            overflow: hidden !important;
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap4.min.js"></script>

    <script>
        let suppliersTable;

        $(document).ready(function() {
            initializeDataTable();

            // Entries per page change handler
            $('#entries-per-page').change(function() {
                suppliersTable.page.len($(this).val()).draw();
            });

            // File input change handler to show filename
            $('#importFile').on('change', function() {
                const fileName = this.files[0] ? this.files[0].name : 'Choose file...';
                $(this).siblings('.custom-file-label').text(fileName);
            });
        });

        function initializeDataTable() {
            suppliersTable = $('#suppliers-table').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: '{{ route("warehouse.suppliers.index") }}',
                    data: function(d) {
                        d.status = $('#filter-status').val();
                        d.country = $('#filter-country').val();
                        d.city = $('#filter-city').val();
                        d.rating_min = $('#filter-rating-min').val();
                        d.rating_max = $('#filter-rating-max').val();
                    }
                },
                columns: [
                    {
                        data: null,
                        name: 'supplier_code',
                        render: function(data) {
                            return `
                                <div class="d-flex align-items-center">
                                    <div class="mr-3">
                                        <div class="bg-primary text-white d-flex align-items-center justify-content-center rounded-circle" style="width: 40px; height: 40px;">
                                            <span class="font-weight-bold" style="font-size: 14px;">${data.name.charAt(0).toUpperCase()}</span>
                                        </div>
                                    </div>
                                    <div>
                                        <div class="font-weight-medium">${data.name}</div>
                                        <small class="text-muted">${data.supplier_code}</small>
                                    </div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'contact_person',
                        render: function(data) {
                            return `
                                <div>
                                    <div class="font-weight-medium mb-1">${data.contact_person || '-'}</div>
                                    <div class="small text-muted mb-1">
                                        <i class="material-icons" style="font-size: 12px; vertical-align: middle;">email</i>
                                        ${data.email || '-'}
                                    </div>
                                    <div class="small text-muted">
                                        <i class="material-icons" style="font-size: 12px; vertical-align: middle;">phone</i>
                                        ${data.phone || '-'}
                                    </div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'city',
                        render: function(data) {
                            let location = [];
                            if (data.city) location.push(data.city);
                            if (data.country) location.push(data.country);
                            return `
                                <div>
                                    <i class="material-icons text-muted mr-1" style="font-size: 14px; vertical-align: middle;">location_on</i>
                                    <span>${location.join(', ') || '-'}</span>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'status',
                        name: 'status',
                        render: function(data) {
                            const statusClasses = {
                                'active': 'status-active',
                                'inactive': 'status-inactive',
                                'suspended': 'status-suspended'
                            };
                            return `<span class="status-badge ${statusClasses[data] || 'status-inactive'}">${data.charAt(0).toUpperCase() + data.slice(1)}</span>`;
                        }
                    },
                    {
                        data: 'rating',
                        name: 'rating',
                        render: function(data) {
                            const rating = parseFloat(data);
                            let stars = '';

                            for (let i = 1; i <= 5; i++) {
                                if (i <= rating) {
                                    stars += '<i class="material-icons text-warning" style="font-size: 16px;">star</i>';
                                } else {
                                    stars += '<i class="material-icons text-muted" style="font-size: 16px;">star_border</i>';
                                }
                            }

                            return `
                                <div class="text-center">
                                    <div class="mb-1">${stars}</div>
                                    <small class="text-muted">${data}</small>
                                </div>
                            `;
                        }
                    },
                    {
                        data: 'performance_score',
                        name: 'performance_score',
                        render: function(data) {
                            const score = parseFloat(data);
                            let badgeClass = 'badge-danger';
                            if (score >= 80) {
                                badgeClass = 'badge-success';
                            } else if (score >= 60) {
                                badgeClass = 'badge-warning';
                            }

                            return `<div class="text-center"><span class="badge ${badgeClass}">${data}%</span></div>`;
                        }
                    },
                    {
                        data: 'payment_terms',
                        name: 'payment_terms',
                        render: function(data) {
                            return data || '-';
                        }
                    },
                    {
                        data: null,
                        name: 'created_at',
                        render: function(data) {
                            return `
                                <div>
                                    <div class="small">${data.created_at}</div>
                                    <div class="small text-muted">by ${data.created_by}</div>
                                </div>
                            `;
                        }
                    },
                    {
                        data: null,
                        name: 'actions',
                        orderable: false,
                        searchable: false,
                        render: function(data) {
                            return `
                                <div class="btn-group" role="group">
                                    <a href="{{ url('warehouse/suppliers') }}/${data.id}" class="btn btn-sm btn-outline-primary" title="View">
                                        <i class="material-icons" style="font-size: 16px;">visibility</i>
                                    </a>
                                    @can('warehouse.suppliers.edit')
                                    <a href="{{ url('warehouse/suppliers') }}/${data.id}/edit" class="btn btn-sm btn-outline-success" title="Edit">
                                        <i class="material-icons" style="font-size: 16px;">edit</i>
                                    </a>
                                    @endcan
                                    @can('warehouse.suppliers.delete')
                                    <button onclick="deleteSupplier(${data.id})" class="btn btn-sm btn-outline-danger" title="Delete">
                                        <i class="material-icons" style="font-size: 16px;">delete</i>
                                    </button>
                                    @endcan
                                </div>
                            `;
                        }
                    }
                ],
                order: [[6, 'desc']],
                pageLength: 25,
                lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                language: {
                    processing: '<div class="text-center"><div class="spinner-border text-primary" role="status"><span class="sr-only">Loading...</span></div></div>',
                    emptyTable: 'No suppliers found',
                    zeroRecords: 'No matching suppliers found'
                },
                drawCallback: function(settings) {
                    $('#entries-per-page').val(settings._iDisplayLength);
                }
            });
        }

        function applyFilters() {
            suppliersTable.draw();
        }

        function clearFilters() {
            $('#filter-status, #filter-country, #filter-city, #filter-rating-min, #filter-rating-max').val('');
            suppliersTable.draw();
        }

        function deleteSupplier(id) {
            if (confirm('Are you sure you want to delete this supplier?')) {
                $.ajaxSetup({
                    headers: {
                        'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                    }
                });

                $.ajax({
                    url: `{{ url('warehouse/suppliers') }}/${id}`,
                    type: 'DELETE',
                    success: function(response) {
                        if (response.success) {
                            suppliersTable.draw();
                            showAlert('Supplier deleted successfully', 'success');
                        } else {
                            showAlert(response.message, 'error');
                        }
                    },
                    error: function(xhr) {
                        showAlert('Error deleting supplier', 'error');
                    }
                });
            }
        }

        function showImportModal() {
            // Reset form
            document.getElementById('importForm').reset();
            $('.custom-file-label').text('Choose file...');
            $('#import-progress').addClass('d-none');
            $('.progress-bar').css('width', '0%');

            // Force very high z-index before showing (same approach as add supplier)
            $('#importModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Show modal
            $('#importModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Aggressively force z-index after modal is shown (same as add supplier)
            $('#importModal').on('shown.bs.modal', function() {
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 999998,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000,
                    'position': 'relative'
                });
                $('.modal-content').css({
                    'z-index': 1000001,
                    'position': 'relative'
                });

                // Move modal to body if it's not already there
                if (!$('#importModal').parent().is('body')) {
                    $('#importModal').appendTo('body');
                }
            });
        }

        function performImport() {
            const fileInput = document.getElementById('importFile');
            const file = fileInput.files[0];

            if (!file) {
                showAlert('Please select a file to import', 'error');
                return;
            }

            // Validate file size (5MB limit)
            const maxSize = 5 * 1024 * 1024; // 5MB in bytes
            if (file.size > maxSize) {
                showAlert('File size exceeds 5MB limit', 'error');
                return;
            }

            // Validate file type
            const allowedTypes = [
                'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet', // .xlsx
                'application/vnd.ms-excel', // .xls
                'text/csv' // .csv
            ];

            if (!allowedTypes.includes(file.type)) {
                showAlert('Invalid file type. Please upload Excel (.xlsx, .xls) or CSV file', 'error');
                return;
            }

            const formData = new FormData(document.getElementById('importForm'));

            // Show progress
            $('#import-progress').removeClass('d-none');
            $('.progress-bar').css('width', '20%');

            // Disable import button
            const importBtn = $('button[onclick="performImport()"]');
            const originalHtml = importBtn.html();
            importBtn.prop('disabled', true);
            importBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Importing...');

            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            });

            $.ajax({
                url: '{{ route("warehouse.suppliers.import") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                xhr: function() {
                    var xhr = new window.XMLHttpRequest();
                    xhr.upload.addEventListener("progress", function(evt) {
                        if (evt.lengthComputable) {
                            var percentComplete = (evt.loaded / evt.total) * 100;
                            $('.progress-bar').css('width', percentComplete + '%');
                        }
                    }, false);
                    return xhr;
                },
                success: function(response) {
                    $('.progress-bar').css('width', '100%');

                    if (response.success) {
                        $('#importModal').modal('hide');
                        suppliersTable.draw();

                        let message = `Import completed successfully! `;
                        if (response.imported) message += `${response.imported} suppliers imported.`;
                        if (response.updated) message += ` ${response.updated} suppliers updated.`;
                        if (response.skipped) message += ` ${response.skipped} rows skipped.`;

                        showAlert(message, 'success');

                        // Reset form
                        document.getElementById('importForm').reset();
                        $('.custom-file-label').text('Choose file...');
                    } else {
                        showAlert(response.message || 'Import failed', 'error');
                    }
                },
                error: function(xhr) {
                    let errorMessage = 'Import failed. Please check your file and try again.';

                    if (xhr.status === 422) {
                        const response = xhr.responseJSON;
                        if (response.errors) {
                            errorMessage = Object.values(response.errors).flat().join(' ');
                        } else if (response.message) {
                            errorMessage = response.message;
                        }
                    } else if (xhr.status === 413) {
                        errorMessage = 'File is too large. Please reduce file size and try again.';
                    } else if (xhr.responseJSON && xhr.responseJSON.message) {
                        errorMessage = xhr.responseJSON.message;
                    }

                    showAlert(errorMessage, 'error');
                },
                complete: function() {
                    // Hide progress and re-enable button
                    $('#import-progress').addClass('d-none');
                    $('.progress-bar').css('width', '0%');
                    importBtn.prop('disabled', false);
                    importBtn.html(originalHtml);
                }
            });
        }

        function downloadTemplate() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Downloading...';
            btn.disabled = true;

            // Create download link - use route helper with proper authentication context
            const downloadUrl = '{{ route("warehouse.suppliers.template") }}';

            console.log('🔗 Attempting to download template from:', downloadUrl);

            fetch(downloadUrl, {
                method: 'GET',
                credentials: 'include',
                headers: {
                    'X-Requested-With': 'XMLHttpRequest',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                }
            })
                .then(response => {
                    console.log('📡 Response received:', response.status, response.statusText);
                    if (!response.ok) {
                        throw new Error('Template download failed');
                    }
                    return response.blob();
                })
                .then(blob => {
                    // Create download link
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = 'suppliers_import_template.xlsx';
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);

                    showAlert('Template downloaded successfully!', 'success');
                })
                .catch(error => {
                    console.error('💥 Download error:', error);
                    console.error('🔗 Failed URL:', downloadUrl);
                    showAlert('Failed to download template: ' + error.message, 'error');
                })
                .finally(() => {
                    btn.innerHTML = originalHtml;
                    btn.disabled = false;
                });
        }

        function exportSuppliers() {
            const btn = event.target.closest('button');
            const originalHtml = btn.innerHTML;
            btn.innerHTML = '<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Exporting...';
            btn.disabled = true;

            const filters = {
                status: $('#filter-status').val(),
                country: $('#filter-country').val(),
                city: $('#filter-city').val(),
                rating_min: $('#filter-rating-min').val(),
                rating_max: $('#filter-rating-max').val()
            };

            const params = new URLSearchParams(filters);

            setTimeout(() => {
                btn.innerHTML = originalHtml;
                btn.disabled = false;
                showAlert('Export functionality is not yet implemented', 'error');
            }, 800);
        }

        function showAddSupplierModal() {
            // Clear form and validation errors
            $('#addSupplierForm')[0].reset();
            clearValidationErrors();

            // Force very high z-index before showing (same approach as import modal)
            $('#addSupplierModal').css({
                'z-index': 999999,
                'position': 'fixed',
                'display': 'block'
            });

            // Show modal
            $('#addSupplierModal').modal({
                backdrop: true,
                keyboard: true,
                show: true
            });

            // Aggressively force z-index after modal is shown (same as import modal)
            $('#addSupplierModal').on('shown.bs.modal', function() {
                $(this).css({
                    'z-index': 999999,
                    'position': 'fixed'
                });
                $('.modal-backdrop').css({
                    'z-index': 999998,
                    'position': 'fixed'
                });
                $('.modal-dialog').css({
                    'z-index': 1000000,
                    'position': 'relative'
                });
                $('.modal-content').css({
                    'z-index': 1000001,
                    'position': 'relative'
                });

                // Move modal to body if it's not already there
                if (!$('#addSupplierModal').parent().is('body')) {
                    $('#addSupplierModal').appendTo('body');
                }
            });
        }

        function saveSupplier() {
            const form = $('#addSupplierForm')[0];
            const formData = new FormData(form);

            // Clear previous validation errors
            clearValidationErrors();

            // Disable save button and show loading
            const saveBtn = $('button[onclick="saveSupplier()"]');
            const originalHtml = saveBtn.html();
            saveBtn.prop('disabled', true);
            saveBtn.html('<i class="material-icons mr-1" style="font-size: 16px; animation: spin 1s linear infinite;">sync</i>Saving...');

            $.ajaxSetup({
                headers: {
                    'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
                }
            });

            $.ajax({
                url: '{{ route("warehouse.suppliers.store") }}',
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    if (response.success) {
                        $('#addSupplierModal').modal('hide');
                        suppliersTable.draw();
                        showAlert('Supplier added successfully!', 'success');
                    } else {
                        showAlert(response.message || 'Error adding supplier', 'error');
                    }
                },
                error: function(xhr) {
                    if (xhr.status === 422) {
                        // Validation errors
                        const errors = xhr.responseJSON.errors;
                        displayValidationErrors(errors);
                    } else {
                        showAlert('Error adding supplier. Please try again.', 'error');
                    }
                },
                complete: function() {
                    // Re-enable save button
                    saveBtn.prop('disabled', false);
                    saveBtn.html(originalHtml);
                }
            });
        }

        function clearValidationErrors() {
            $('.form-control').removeClass('is-invalid');
            $('.invalid-feedback').text('');
        }

        function displayValidationErrors(errors) {
            $.each(errors, function(field, messages) {
                const input = $(`[name="${field}"]`);
                input.addClass('is-invalid');
                input.siblings('.invalid-feedback').text(messages[0]);
            });
        }

        function showAlert(message, type) {
            const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
            const alert = `
                <div class="alert ${alertClass} alert-dismissible fade show position-fixed" style="top: 20px; right: 20px; z-index: 9999;">
                    ${message}
                    <button type="button" class="close" data-dismiss="alert">&times;</button>
                </div>
            `;
            $('body').append(alert);
            setTimeout(() => $('.alert').alert('close'), 5000);
        }
    </script>
@endpush