<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\Inventory;
use App\Models\Warehouse\Project;

class AddSampleInventory extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'db:add-sample-inventory {--items=10 : Number of items to add inventory for}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Add sample inventory data for testing';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🏭 Adding sample inventory data...');

        $itemCount = $this->option('items');

        // Get random items
        $items = Item::limit($itemCount)->get();

        if ($items->isEmpty()) {
            $this->error('No items found in database. Please add items first.');
            return 1;
        }

        $this->info('📦 Adding inventory for ' . $items->count() . ' items...');

        $addedCount = 0;

        foreach ($items as $item) {
            // Add 2-3 inventory records per item (different batches)
            $batches = rand(1, 3);

            for ($i = 0; $i < $batches; $i++) {
                $quantity = rand(100, 1000); // Random quantity between 100-1000
                $unitPrice = rand(10, 500); // Random price between 10-500

                Inventory::create([
                    'item_id' => $item->id,
                    'division_id' => null, // Can be null
                    'quantity_available' => $quantity,
                    'quantity_allocated' => 0,
                    'quantity_on_hand' => $quantity,
                    'unit_price' => $unitPrice,
                    'total_value' => $quantity * $unitPrice,
                    'batch_number' => 'BATCH-' . date('Ymd') . '-' . str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT),
                    'expiry_date' => rand(0, 1) ? date('Y-m-d', strtotime('+' . rand(30, 365) . ' days')) : null,
                    'location' => 'Warehouse-' . chr(65 + rand(0, 3)) . '-' . rand(1, 10), // Like Warehouse-A-5
                    'supplier_id' => null, // Can add if needed
                    'purchase_order_number' => 'PO-' . date('Ymd') . '-' . str_pad(rand(1, 999), 3, '0', STR_PAD_LEFT),
                    'received_date' => date('Y-m-d', strtotime('-' . rand(1, 30) . ' days')),
                    'notes' => 'Sample inventory for testing',
                ]);

                $addedCount++;
            }

            $this->line("  ✅ Added {$batches} batch(es) for: {$item->item_code} - {$item->item_description}");
        }

        $this->newLine();
        $this->info("✅ Successfully added {$addedCount} inventory records!");

        // Show summary
        $totalQuantity = Inventory::sum('quantity_available');
        $totalValue = Inventory::sum('total_value');
        $uniqueItems = Inventory::distinct('item_id')->count();

        $this->newLine();
        $this->info('📊 Inventory Summary:');
        $this->line("  • Total Items with Stock: {$uniqueItems}");
        $this->line("  • Total Quantity Available: " . number_format($totalQuantity, 2));
        $this->line("  • Total Inventory Value: AED " . number_format($totalValue, 2));

        return 0;
    }
}