<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class CleanTransactionData extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'db:clean-transactions {--force : Skip confirmation prompt}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clean all transaction data from database while preserving master data';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        if (!$this->option('force')) {
            if (!$this->confirm('⚠️  This will DELETE ALL transaction data. Are you sure?')) {
                $this->info('Operation cancelled.');
                return;
            }
        }

        $this->info('🧹 Starting Database Transaction Data Cleanup...');
        $this->newLine();

        try {
            $this->info('📋 Tables to be cleaned:');

            // Define tables to clean (transaction data only)
            $transactionTables = [
                // Core transaction tables
                'outgoing_transactions',
                'outgoing_items',
                'incoming_operations',
                'incoming_operation_items',
                'return_transactions',
                'return_items',
                'stock_movements',

                // Material transfer tables
                'material_transfer_requests',
                'material_transfer_request_items',
                'material_transfer_issues',
                'material_transfer_issue_items',

                // Direct delivery tables
                'direct_deliveries',
                'direct_delivery_items',

                // Project inventory table
                'project_inventory',

                // Delivery notes
                'material_delivery_notes',

                // Sync history
                'sync_history',
            ];

            // Check which tables exist and clean them
            foreach ($transactionTables as $table) {
                if (Schema::hasTable($table)) {
                    $count = DB::table($table)->count();
                    if ($count > 0) {
                        $this->line("  📊 {$table}: {$count} records");

                        // Disable foreign key checks temporarily
                        DB::statement('SET FOREIGN_KEY_CHECKS=0');

                        // Truncate the table
                        DB::table($table)->truncate();

                        // Re-enable foreign key checks
                        DB::statement('SET FOREIGN_KEY_CHECKS=1');

                        $this->line("     ✅ Cleaned");
                    } else {
                        $this->line("  📊 {$table}: Already empty");
                    }
                } else {
                    $this->line("  ⚠️  {$table}: Table does not exist");
                }
            }

            $this->newLine();
            $this->info('🔄 Resetting inventory quantities...');

            // Reset inventory quantities to zero (keeping items and categories)
            if (Schema::hasTable('inventory')) {
                $inventoryCount = DB::table('inventory')->count();
                $this->line("  📊 inventory: {$inventoryCount} records");

                if ($inventoryCount > 0) {
                    DB::statement('SET FOREIGN_KEY_CHECKS=0');
                    DB::table('inventory')->truncate();
                    DB::statement('SET FOREIGN_KEY_CHECKS=1');
                    $this->line("     ✅ Cleaned");
                }
            }

            $this->newLine();
            $this->info('🔄 Cleaning additional transaction-related data...');

            // Clean any other transaction-related tables that might exist
            $additionalTables = [
                'activity_logs',
                'notifications',
            ];

            foreach ($additionalTables as $table) {
                if (Schema::hasTable($table)) {
                    $count = DB::table($table)->count();
                    if ($count > 0) {
                        $this->line("  📊 {$table}: {$count} records");
                        DB::statement('SET FOREIGN_KEY_CHECKS=0');
                        DB::table($table)->truncate();
                        DB::statement('SET FOREIGN_KEY_CHECKS=1');
                        $this->line("     ✅ Cleaned");
                    } else {
                        $this->line("  📊 {$table}: Already empty");
                    }
                }
            }

            // All operations completed successfully

            $this->newLine();
            $this->info('✅ Database cleanup completed successfully!');
            $this->newLine();
            $this->info('📋 Summary:');
            $this->line('  ✅ All transaction data removed');
            $this->line('  ✅ All project inventory cleared');
            $this->line('  ✅ All stock movements cleared');
            $this->line('  ✅ All delivery records cleared');
            $this->line('  ✅ Master data preserved (items, categories, projects, suppliers, users)');

            $this->newLine();
            $this->info('🎯 You can now test the new material issue workflow with a clean database!');

        } catch (\Exception $e) {
            // Handle error

            $this->newLine();
            $this->error('❌ Error occurred during cleanup: ' . $e->getMessage());
            $this->error('🔄 Error occurred during cleanup.');

            return 1;
        }

        $this->newLine();
        $this->info('🏁 Cleanup script completed.');

        return 0;
    }
}