<?php

namespace App\Console\Commands;

use App\Services\ExternalProjectImportService;
use Illuminate\Console\Command;
use Exception;

class ImportProjectsCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'projects:import
                            {--table=projects : External table name}
                            {--batch-size=100 : Number of records to process at once}
                            {--skip-existing : Skip projects that already exist}
                            {--test-connection : Test external database connection only}
                            {--show-tables : Show available tables in external database}
                            {--describe-table= : Show structure of specific table}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Import projects from external database';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $importService = new ExternalProjectImportService();

        try {
            // Test connection option
            if ($this->option('test-connection')) {
                return $this->testConnection($importService);
            }

            // Show tables option
            if ($this->option('show-tables')) {
                return $this->showTables($importService);
            }

            // Describe table option
            if ($this->option('describe-table')) {
                return $this->describeTable($importService, $this->option('describe-table'));
            }

            // Main import process
            $this->info('Starting project import from external database...');

            $options = [
                'table_name' => $this->option('table'),
                'batch_size' => (int) $this->option('batch-size'),
                'skip_existing' => $this->option('skip-existing'),
            ];

            $results = $importService->importProjects($options);

            $this->displayResults($results);

        } catch (Exception $e) {
            $this->error('Import failed: ' . $e->getMessage());
            return Command::FAILURE;
        }

        return Command::SUCCESS;
    }

    /**
     * Test external database connection
     */
    private function testConnection(ExternalProjectImportService $importService)
    {
        $this->info('Testing external database connection...');

        $info = $importService->getExternalDatabaseInfo();

        if ($info['connection_status'] === 'connected') {
            $this->info('✓ External database connection successful');
            $this->info('Available tables: ' . implode(', ', $info['tables']));
        } else {
            $this->error('✗ External database connection failed: ' . $info['error']);
            return Command::FAILURE;
        }

        return Command::SUCCESS;
    }

    /**
     * Show available tables
     */
    private function showTables(ExternalProjectImportService $importService)
    {
        $this->info('Fetching tables from external database...');

        $info = $importService->getExternalDatabaseInfo();

        if ($info['connection_status'] === 'connected') {
            $this->table(['Available Tables'], array_map(function($table) {
                return [$table];
            }, $info['tables']));
        } else {
            $this->error('Cannot connect to external database: ' . $info['error']);
            return Command::FAILURE;
        }

        return Command::SUCCESS;
    }

    /**
     * Describe table structure
     */
    private function describeTable(ExternalProjectImportService $importService, $tableName)
    {
        $this->info("Describing table structure: {$tableName}");

        try {
            $columns = $importService->getTableStructure($tableName);

            $headers = ['Field', 'Type', 'Null', 'Key', 'Default', 'Extra'];
            $rows = [];

            foreach ($columns as $column) {
                $rows[] = [
                    $column->Field,
                    $column->Type,
                    $column->Null,
                    $column->Key,
                    $column->Default,
                    $column->Extra
                ];
            }

            $this->table($headers, $rows);

        } catch (Exception $e) {
            $this->error('Failed to describe table: ' . $e->getMessage());
            return Command::FAILURE;
        }

        return Command::SUCCESS;
    }

    /**
     * Display import results
     */
    private function displayResults($results)
    {
        $this->info('Import completed!');
        $this->newLine();

        $this->info("📊 Import Summary:");
        $this->line("Total processed: {$results['total_processed']}");
        $this->line("Successfully imported: {$results['imported']}");
        $this->line("Skipped (already exist): {$results['skipped']}");
        $this->line("Errors: {$results['errors']}");

        if (!empty($results['error_details'])) {
            $this->newLine();
            $this->error('Error Details:');
            foreach ($results['error_details'] as $error) {
                $this->line("  • {$error}");
            }
        }

        if ($results['imported'] > 0) {
            $this->newLine();
            $this->info('✓ Import completed successfully!');
        }
    }
}
