<?php

namespace App\Http\Controllers\Warehouse;

use App\Http\Controllers\Controller;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\ItemCategory;
use App\Models\Warehouse\Inventory;
use App\Models\Warehouse\IncomingOperation;
use App\Models\Warehouse\IncomingOperationItem;
use App\Models\Warehouse\OutgoingItem;
use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\Project;
use App\Models\Warehouse\ProjectDivision;
use App\Models\Warehouse\ProjectInventory;
use App\Models\Warehouse\ReturnItem;
use App\Models\Warehouse\ReturnTransaction;
use App\Models\Warehouse\Supplier;
use App\Models\Department;
use App\Models\DirectDelivery;
use App\Models\Warehouse\MaterialTransferRequest;
use App\Models\Warehouse\MaterialTransferRequestItem;
use App\Models\Warehouse\OutgoingFulfillment;
use App\Models\FocInventory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    public function index()
    {
        return view('warehouse.reports.index');
    }

    public function inventory(Request $request)
    {
        $query = Item::with(['category', 'inventories.division'])
            ->select('items.*')
            ->leftJoin('inventory', 'items.id', '=', 'inventory.item_id')
            ->groupBy('items.id', 'items.item_code', 'items.item_description',
                     'items.category_id', 'items.item_type_id', 'items.unit_id',
                     'items.unit_of_measure_old', 'items.unit_of_measure',
                     'items.min_stock_level', 'items.max_stock_level',
                     'items.reorder_point', 'items.status', 'items.notes',
                     'items.created_at', 'items.updated_at');

        // Apply filters
        if ($request->filled('category_id')) {
            $query->where('items.category_id', $request->category_id);
        }

        if ($request->filled('division_id')) {
            $query->where('inventory.division_id', $request->division_id);
        }

        if ($request->filled('status')) {
            $query->where('inventory.status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('items.item_code', 'like', "%{$search}%")
                  ->orWhere('items.item_description', 'like', "%{$search}%");
            });
        }

        // Paginate the query first
        $paginatedItems = $query->paginate(25);
        
        // Map the items with calculated values
        $items = $paginatedItems->getCollection()->map(function ($item) {
            // Get all inventory records for this item
            $inventories = $item->inventories;

            // Calculate aggregated data
            $totalQuantity = $inventories->sum('quantity_available');
            $totalValue = $inventories->sum('total_value');

            // Calculate average price (weighted by quantity)
            $averagePrice = $totalQuantity > 0 ? $totalValue / $totalQuantity : 0;

            // Get pricing range from incoming operations
            $pricingData = IncomingOperationItem::where('item_id', $item->id)
                ->where('quantity_delivered', '>', 0)
                ->selectRaw('MIN(unit_price_aed) as min_price, MAX(unit_price_aed) as max_price, AVG(unit_price_aed) as avg_price')
                ->first();

            // Get stock status counts
            $stockStatusCounts = $inventories->groupBy('status')->map(function ($group) {
                return [
                    'count' => $group->count(),
                    'quantity' => $group->sum('quantity_available')
                ];
            });

            // Get latest production and expiry dates
            $latestProduction = $inventories->whereNotNull('production_date')->max('production_date');
            $latestExpiry = $inventories->whereNotNull('expiry_date')->max('expiry_date');

            // Get divisions where this item is stored
            $divisions = $inventories->groupBy('division_id')->map(function ($group) {
                $division = $group->first()->division;
                return [
                    'name' => $division ? $division->division_name : 'Unknown',
                    'quantity' => $group->sum('quantity_available')
                ];
            });

            return (object) [
                'id' => $item->id,
                'item_code' => $item->item_code,
                'item_description' => $item->item_description,
                'category' => $item->category ? $item->category->category_name : 'No Category',
                'category_id' => $item->category_id,
                'unit_of_measure' => $item->unit_of_measure,
                'total_quantity' => $totalQuantity,
                'average_unit_price' => $averagePrice,
                'total_value' => $totalValue,
                'min_price' => $pricingData->min_price ?? 0,
                'max_price' => $pricingData->max_price ?? 0,
                'avg_purchase_price' => $pricingData->avg_price ?? 0,
                'latest_production_date' => $latestProduction,
                'latest_expiry_date' => $latestExpiry,
                'divisions' => $divisions,
                'stock_status_counts' => $stockStatusCounts,
                'overall_status' => $this->determineOverallStatus($stockStatusCounts, $totalQuantity)
            ];
        });

        // Set the mapped collection back to the paginator
        $paginatedItems->setCollection($items);

        // Filter out items with no inventory if requested
        if ($request->has('exclude_zero_stock') && $request->exclude_zero_stock) {
            $filteredItems = $items->filter(function ($item) {
                return $item->total_quantity > 0;
            });
            $paginatedItems->setCollection($filteredItems);
        }

        // Get filter options
        $categories = ItemCategory::all();
        $divisions = ProjectDivision::where('status', 'active')->orderBy('division_name')->get();
        $stockStatuses = ['in_stock', 'low_stock', 'out_of_stock'];

        return view('warehouse.reports.inventory', [
            'items' => $paginatedItems->appends($request->query()),
            'categories' => $categories,
            'divisions' => $divisions,
            'stockStatuses' => $stockStatuses
        ]);
    }

    public function inventoryExport(Request $request)
    {
        $query = Item::with(['category', 'inventories.division'])
            ->select('items.*')
            ->leftJoin('inventory', 'items.id', '=', 'inventory.item_id')
            ->groupBy('items.id', 'items.item_code', 'items.item_description',
                     'items.category_id', 'items.item_type_id', 'items.unit_id',
                     'items.unit_of_measure_old', 'items.unit_of_measure',
                     'items.min_stock_level', 'items.max_stock_level',
                     'items.reorder_point', 'items.status', 'items.notes',
                     'items.created_at', 'items.updated_at');

        // Apply the same filters as the main inventory report
        if ($request->filled('category_id')) {
            $query->where('items.category_id', $request->category_id);
        }

        if ($request->filled('division_id')) {
            $query->where('inventory.division_id', $request->division_id);
        }

        if ($request->filled('status')) {
            $query->where('inventory.status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('items.item_code', 'like', "%{$search}%")
                  ->orWhere('items.item_description', 'like', "%{$search}%");
            });
        }

        // Get all items (without pagination for export)
        $items = $query->get()->map(function ($item) {
            // Get all inventory records for this item
            $inventories = $item->inventories;

            // Calculate aggregated data
            $totalQuantity = $inventories->sum('quantity_available');
            $totalValue = $inventories->sum('total_value');

            // Calculate average price (weighted by quantity)
            $averagePrice = $totalQuantity > 0 ? $totalValue / $totalQuantity : 0;

            // Get pricing range from incoming operations
            $pricingData = \App\Models\Warehouse\IncomingOperationItem::where('item_id', $item->id)
                ->where('quantity_delivered', '>', 0)
                ->selectRaw('MIN(unit_price_aed) as min_price, MAX(unit_price_aed) as max_price')
                ->first();

            // Get production and expiry date ranges
            $dateData = $inventories
                ->where('quantity_available', '>', 0)
                ->groupBy(function($inv) { return $inv->item_id; })
                ->map(function($group) {
                    return (object)[
                        'latest_production_date' => $group->whereNotNull('production_date')->max('production_date'),
                        'latest_expiry_date' => $group->whereNotNull('expiry_date')->min('expiry_date')
                    ];
                })->first();

            // Determine stock status
            $overallStatus = 'in_stock';
            if ($totalQuantity <= 0) {
                $overallStatus = 'out_of_stock';
            } elseif ($item->reorder_point && $totalQuantity <= $item->reorder_point) {
                $overallStatus = 'low_stock';
            }

            return (object) [
                'id' => $item->id,
                'item_code' => $item->item_code,
                'item_description' => $item->item_description,
                'category' => $item->category ? $item->category->category_name : 'Uncategorized',
                'unit_of_measure' => $item->unit_of_measure,
                'total_quantity' => $totalQuantity,
                'average_unit_price' => $averagePrice,
                'min_price' => $pricingData ? $pricingData->min_price : 0,
                'max_price' => $pricingData ? $pricingData->max_price : 0,
                'total_value' => $totalValue,
                'latest_production_date' => $dateData ? $dateData->latest_production_date : null,
                'latest_expiry_date' => $dateData ? $dateData->latest_expiry_date : null,
                'divisions' => $inventories->unique('division_id')->pluck('division.name')->filter()->implode(', ') ?: '-',
                'overall_status' => $overallStatus,
                'reorder_point' => $item->reorder_point,
                'min_stock_level' => $item->min_stock_level,
                'max_stock_level' => $item->max_stock_level,
            ];
        });

        // Check if there are items to export
        if ($items->isEmpty()) {
            // Return a simple message if no data to export
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();
            $sheet->setTitle('Inventory Report');
            $sheet->setCellValue('A1', 'No inventory items found matching your filters.');
            $sheet->getStyle('A1')->getFont()->setBold(true);

            $filename = 'inventory-report-empty-' . now()->format('Y-m-d-H-i-s') . '.xlsx';
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);

            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Cache-Control: max-age=0');

            $writer->save('php://output');
            exit;
        }

        // Create Excel file using PhpSpreadsheet
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Inventory Report');

        // Set headers
        $headers = [
            'A' => 'Item Code',
            'B' => 'Description',
            'C' => 'Category',
            'D' => 'Unit',
            'E' => 'Quantity',
            'F' => 'Avg Unit Price (AED)',
            'G' => 'Min Price (AED)',
            'H' => 'Max Price (AED)',
            'I' => 'Total Value (AED)',
            'J' => 'Production Date',
            'K' => 'Expiry Date',
            'L' => 'Divisions',
            'M' => 'Status',
            'N' => 'Reorder Point',
            'O' => 'Min Stock Level',
            'P' => 'Max Stock Level'
        ];

        // Apply header styling
        foreach ($headers as $col => $header) {
            $sheet->setCellValue($col . '1', $header);
            $sheet->getStyle($col . '1')->getFont()->setBold(true);
            $sheet->getStyle($col . '1')->getFill()
                ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FFE2E2E2');
        }

        // Add data rows
        $row = 2;
        foreach ($items as $item) {
            $sheet->setCellValue('A' . $row, $item->item_code);
            $sheet->setCellValue('B' . $row, $item->item_description);
            $sheet->setCellValue('C' . $row, $item->category);
            $sheet->setCellValue('D' . $row, $item->unit_of_measure);
            $sheet->setCellValue('E' . $row, number_format($item->total_quantity, 2));
            $sheet->setCellValue('F' . $row, number_format($item->average_unit_price, 2));
            $sheet->setCellValue('G' . $row, number_format($item->min_price, 2));
            $sheet->setCellValue('H' . $row, number_format($item->max_price, 2));
            $sheet->setCellValue('I' . $row, number_format($item->total_value, 2));
            $sheet->setCellValue('J' . $row, $item->latest_production_date ? \Carbon\Carbon::parse($item->latest_production_date)->format('d M Y') : '-');
            $sheet->setCellValue('K' . $row, $item->latest_expiry_date ? \Carbon\Carbon::parse($item->latest_expiry_date)->format('d M Y') : '-');
            $sheet->setCellValue('L' . $row, $item->divisions);
            $sheet->setCellValue('M' . $row, ucfirst(str_replace('_', ' ', $item->overall_status)));
            $sheet->setCellValue('N' . $row, $item->reorder_point ?: '-');
            $sheet->setCellValue('O' . $row, $item->min_stock_level ?: '-');
            $sheet->setCellValue('P' . $row, $item->max_stock_level ?: '-');
            $row++;
        }

        // Auto-size columns
        foreach (range('A', 'P') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        // Create filename with timestamp and filters
        $filename = 'inventory-report-' . now()->format('Y-m-d-H-i-s');
        if ($request->filled('search')) {
            $filename .= '-search-' . \Str::slug($request->search);
        }
        if ($request->filled('category_id')) {
            $category = \App\Models\Warehouse\ItemCategory::find($request->category_id);
            if ($category) {
                $filename .= '-' . \Str::slug($category->category_name);
            }
        }
        $filename .= '.xlsx';

        // Create Excel writer
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);

        // Set headers for download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        // Write to output
        $writer->save('php://output');
        exit;
    }

    public function inventoryItemDetails($itemId)
    {
        $item = Item::with(['category', 'inventories.division'])->findOrFail($itemId);

        // Get detailed pricing history from incoming operations
        $pricingHistory = IncomingOperationItem::with(['operation.supplier'])
            ->where('item_id', $itemId)
            ->where('quantity_delivered', '>', 0)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($incomingItem) {
                return (object) [
                    'date' => $incomingItem->created_at,
                    'supplier' => $incomingItem->operation->supplier->name ?? 'Unknown',
                    'quantity' => $incomingItem->quantity_delivered,
                    'unit_price' => $incomingItem->unit_price_aed,
                    'total_price' => $incomingItem->total_price_aed,
                    'currency' => $incomingItem->currency,
                    'original_unit_price' => $incomingItem->unit_price,
                    'batch_number' => $incomingItem->batch_number,
                    'operation_number' => $incomingItem->operation->operation_number ?? 'N/A'
                ];
            });

        // Get detailed inventory breakdown by division and batch
        $inventoryDetails = $item->inventories()
            ->with('division')
            ->where('quantity_available', '>', 0)
            ->orderBy('expiry_date', 'asc')
            ->get()
            ->map(function ($inventory) {
                return (object) [
                    'division' => $inventory->division->division_name ?? 'Unknown',
                    'location' => $inventory->location,
                    'batch_number' => $inventory->batch_number,
                    'quantity' => $inventory->quantity_available,
                    'unit_price' => $inventory->unit_price,
                    'total_value' => $inventory->total_value,
                    'production_date' => $inventory->production_date,
                    'expiry_date' => $inventory->expiry_date,
                    'status' => $inventory->status,
                    'days_until_expiry' => $inventory->days_until_expiry
                ];
            });

        return view('warehouse.reports.inventory-details', compact('item', 'pricingHistory', 'inventoryDetails'));
    }

    public function materials(Request $request)
    {
        $query = Item::with(['category', 'inventories.division'])
            ->select('items.*')
            ->leftJoin('inventory', 'items.id', '=', 'inventory.item_id')
            ->groupBy('items.id', 'items.item_code', 'items.item_description',
                     'items.category_id', 'items.item_type_id', 'items.unit_id',
                     'items.unit_of_measure_old', 'items.unit_of_measure',
                     'items.min_stock_level', 'items.max_stock_level',
                     'items.reorder_point', 'items.status', 'items.notes',
                     'items.created_at', 'items.updated_at');

        // Apply filters
        if ($request->filled('category_id')) {
            $query->where('items.category_id', $request->category_id);
        }

        if ($request->filled('division_id')) {
            $query->where('inventory.division_id', $request->division_id);
        }

        if ($request->filled('status')) {
            $query->where('inventory.status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('items.item_code', 'like', "%{$search}%")
                  ->orWhere('items.item_description', 'like', "%{$search}%");
            });
        }

        // Paginate the query first
        $paginatedItems = $query->paginate(25);

        // Map the items with calculated values
        $items = $paginatedItems->getCollection()->map(function ($item) {
            // Get all inventory records for this item
            $inventories = $item->inventories;

            // Calculate aggregated data
            $totalQuantity = $inventories->sum('quantity_available');
            $totalValue = $inventories->sum('total_value');

            // Calculate average price (weighted by quantity)
            $averagePrice = $totalQuantity > 0 ? $totalValue / $totalQuantity : 0;

            // Get pricing range from incoming operations
            $pricingData = IncomingOperationItem::where('item_id', $item->id)
                ->where('quantity_delivered', '>', 0)
                ->selectRaw('MIN(unit_price_aed) as min_price, MAX(unit_price_aed) as max_price, AVG(unit_price_aed) as avg_price')
                ->first();

            // Get stock status counts
            $stockStatusCounts = $inventories->groupBy('status')->map(function ($group) {
                return [
                    'count' => $group->count(),
                    'quantity' => $group->sum('quantity_available')
                ];
            });

            // Get latest production and expiry dates
            $latestProduction = $inventories->whereNotNull('production_date')->max('production_date');
            $latestExpiry = $inventories->whereNotNull('expiry_date')->max('expiry_date');

            // Get divisions where this item is stored
            $divisions = $inventories->groupBy('division_id')->map(function ($group) {
                $division = $group->first()->division;
                return [
                    'name' => $division ? $division->division_name : 'Unknown',
                    'quantity' => $group->sum('quantity_available')
                ];
            });

            return (object) [
                'id' => $item->id,
                'item_code' => $item->item_code,
                'item_description' => $item->item_description,
                'category' => $item->category ? $item->category->category_name : 'No Category',
                'category_id' => $item->category_id,
                'unit_of_measure' => $item->unit_of_measure,
                'total_quantity' => $totalQuantity,
                'average_unit_price' => $averagePrice,
                'total_value' => $totalValue,
                'min_price' => $pricingData->min_price ?? 0,
                'max_price' => $pricingData->max_price ?? 0,
                'avg_purchase_price' => $pricingData->avg_price ?? 0,
                'latest_production_date' => $latestProduction,
                'latest_expiry_date' => $latestExpiry,
                'divisions' => $divisions,
                'stock_status_counts' => $stockStatusCounts,
                'overall_status' => $this->determineOverallStatus($stockStatusCounts, $totalQuantity)
            ];
        });

        // Set the mapped collection back to the paginator
        $paginatedItems->setCollection($items);

        // Filter out items with no inventory if requested
        if ($request->has('exclude_zero_stock') && $request->exclude_zero_stock) {
            $filteredItems = $items->filter(function ($item) {
                return $item->total_quantity > 0;
            });
            $paginatedItems->setCollection($filteredItems);
        }

        // Get filter options
        $categories = ItemCategory::all();
        $divisions = ProjectDivision::where('status', 'active')->orderBy('division_name')->get();
        $stockStatuses = ['in_stock', 'low_stock', 'out_of_stock'];

        return view('warehouse.reports.materials', [
            'items' => $paginatedItems->appends($request->query()),
            'categories' => $categories,
            'divisions' => $divisions,
            'stockStatuses' => $stockStatuses
        ]);
    }

    public function materialsItemDetails($itemId)
    {
        $item = Item::with(['category', 'inventories.division'])->findOrFail($itemId);

        // Get detailed pricing history from incoming operations
        $pricingHistory = IncomingOperationItem::with(['operation.supplier'])
            ->where('item_id', $itemId)
            ->where('quantity_delivered', '>', 0)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($incomingItem) {
                return (object) [
                    'date' => $incomingItem->created_at,
                    'supplier' => $incomingItem->operation->supplier->name ?? 'Unknown',
                    'quantity' => $incomingItem->quantity_delivered,
                    'unit_price' => $incomingItem->unit_price_aed,
                    'total_price' => $incomingItem->total_price_aed,
                    'currency' => $incomingItem->currency,
                    'original_unit_price' => $incomingItem->unit_price,
                    'batch_number' => $incomingItem->batch_number,
                    'operation_number' => $incomingItem->operation->operation_number ?? 'N/A'
                ];
            });

        // Get regular warehouse inventory breakdown by division and batch
        $warehouseInventory = $item->inventories()
            ->with('division')
            ->where('quantity_available', '>', 0)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($inventory) {
                return [
                    'type' => 'warehouse',
                    'location' => $inventory->division ? $inventory->division->division_name : 'Unknown',
                    'quantity_available' => $inventory->quantity_available,
                    'unit_price' => $inventory->unit_price,
                    'total_value' => $inventory->total_value,
                    'batch_number' => $inventory->batch_number,
                    'production_date' => $inventory->production_date,
                    'expiry_date' => $inventory->expiry_date,
                    'status' => $inventory->status,
                    'days_until_expiry' => $inventory->days_until_expiry,
                    'delivery_reference' => $inventory->delivery_reference,
                    'delivery_type' => $inventory->delivery_type
                ];
            });

        // Get project inventory (materials allocated to projects)
        $projectInventory = ProjectInventory::with(['project', 'item'])
            ->where('item_id', $itemId)
            ->where('quantity_available', '>', 0)
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($inventory) {
                return [
                    'type' => 'project',
                    'location' => $inventory->project ? $inventory->project->project_name : 'Unknown Project',
                    'project_id' => $inventory->project_id,
                    'quantity_available' => $inventory->quantity_available,
                    'allocated_quantity' => $inventory->allocated_quantity,
                    'available_for_transfer' => $inventory->available_for_transfer,
                    'unit_price' => $inventory->unit_price,
                    'total_value' => $inventory->total_value,
                    'batch_number' => '-',
                    'production_date' => null,
                    'expiry_date' => null,
                    'status' => $inventory->quantity_available > 0 ? 'in_stock' : 'out_of_stock',
                    'days_until_expiry' => null,
                    'delivery_reference' => null,
                    'delivery_type' => 'project_allocation'
                ];
            });

        // Get FOC inventory (Free of Charge materials from damaged/returned items)
        $focInventory = FocInventory::with(['item', 'sourceProject'])
            ->where('item_id', $itemId)
            ->where('quantity_available', '>', 0)
            ->where('status', 'available')
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($inventory) {
                return [
                    'type' => 'foc',
                    'location' => $inventory->sourceProject ? $inventory->sourceProject->project_name : 'FOC Warehouse',
                    'project_id' => $inventory->source_project_id,
                    'quantity_available' => $inventory->quantity_available,
                    'quantity_reserved' => $inventory->quantity_reserved,
                    'net_available' => $inventory->getNetAvailableQuantity(),
                    'unit_price' => 0, // FOC items have no cost
                    'total_value' => 0,
                    'estimated_value' => $inventory->estimated_value,
                    'recoverable_value' => $inventory->recoverable_value,
                    'batch_number' => $inventory->batch_number,
                    'production_date' => $inventory->production_date,
                    'expiry_date' => $inventory->expiry_date,
                    'status' => $inventory->status,
                    'condition' => $inventory->condition,
                    'condition_notes' => $inventory->condition_notes,
                    'damage_severity' => $inventory->damage_severity,
                    'source_type' => $inventory->source_type,
                    'source_reference' => $inventory->source_reference,
                    'days_until_expiry' => $inventory->expiry_date ? now()->diffInDays($inventory->expiry_date, false) : null,
                    'delivery_reference' => $inventory->source_reference,
                    'delivery_type' => 'foc_transfer'
                ];
            });

        // Combine all inventories for summary statistics
        $allInventories = collect()
            ->merge($warehouseInventory)
            ->merge($projectInventory)
            ->merge($focInventory);

        // Calculate summary statistics
        $totalQuantity = $allInventories->sum('quantity_available');
        $totalValue = $warehouseInventory->sum('total_value') + $projectInventory->sum('total_value');
        $estimatedFocValue = $focInventory->sum('estimated_value');

        // Group project inventory by project for easier display
        $projectSummary = $projectInventory->groupBy('project_id')->map(function ($items, $projectId) {
            $project = $items->first();
            return [
                'project_name' => $project['location'],
                'project_id' => $projectId,
                'total_quantity' => $items->sum('quantity_available'),
                'allocated_quantity' => $items->sum('allocated_quantity'),
                'available_for_transfer' => $items->sum('available_for_transfer'),
                'total_value' => $items->sum('total_value'),
                'batches_count' => $items->count()
            ];
        });

        // Group FOC inventory by source project
        $focSummary = $focInventory->groupBy('project_id')->map(function ($items, $projectId) {
            $firstItem = $items->first();
            return [
                'project_name' => $firstItem['location'],
                'project_id' => $projectId,
                'total_quantity' => $items->sum('quantity_available'),
                'reserved_quantity' => $items->sum('quantity_reserved'),
                'net_available' => $items->sum('net_available'),
                'estimated_value' => $items->sum('estimated_value'),
                'recoverable_value' => $items->sum('recoverable_value'),
                'batches_count' => $items->count()
            ];
        });

        return view('warehouse.reports.materials-details', compact(
            'item',
            'pricingHistory',
            'warehouseInventory',
            'projectInventory',
            'focInventory',
            'allInventories',
            'totalQuantity',
            'totalValue',
            'estimatedFocValue',
            'projectSummary',
            'focSummary'
        ));
    }

    private function determineOverallStatus($statusCounts, $totalQuantity)
    {
        if ($totalQuantity <= 0) {
            return 'out_of_stock';
        }

        $lowStockQty = $statusCounts->get('low_stock')['quantity'] ?? 0;
        $inStockQty = $statusCounts->get('in_stock')['quantity'] ?? 0;

        if ($lowStockQty > 0 && $inStockQty === 0) {
            return 'low_stock';
        }

        return 'in_stock';
    }

    public function stockMovements()
    {
        return view('warehouse.reports.stock-movements');
    }

    public function transactions(Request $request)
    {
        // Collect all transactions from different sources
        $transactions = collect();

        // Get Incoming Operations (exclude site returns as they're separate)
        $incomingQuery = IncomingOperation::with(['supplier', 'project', 'division', 'receiver'])
            ->where('operation_type', '!=', 'site_return');

        // Get Outgoing Transactions
        $outgoingQuery = OutgoingTransaction::with(['project', 'division', 'requester']);

        // Get Direct Deliveries
        $directDeliveryQuery = DirectDelivery::with(['supplier', 'project', 'projectDivision']);

        // Get Material Transfer Requests
        $transferQuery = MaterialTransferRequest::with(['transferrerProject', 'receiverProject', 'requestedBy']);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $incomingQuery->where(function($q) use ($search) {
                $q->where('operation_number', 'like', "%{$search}%")
                  ->orWhere('lpo_number', 'like', "%{$search}%")
                  ->orWhere('delivery_note_number', 'like', "%{$search}%");
            });

            $outgoingQuery->where(function($q) use ($search) {
                $q->where('transaction_number', 'like', "%{$search}%")
                  ->orWhere('reference_number', 'like', "%{$search}%");
            });

            $directDeliveryQuery->where(function($q) use ($search) {
                $q->where('reference_number', 'like', "%{$search}%");
            });

            $transferQuery->where(function($q) use ($search) {
                $q->where('transfer_request_number', 'like', "%{$search}%");
            });
        }

        if ($request->filled('date_from')) {
            $dateFrom = $request->date_from;
            $incomingQuery->whereDate('operation_date', '>=', $dateFrom);
            $outgoingQuery->whereDate('transaction_date', '>=', $dateFrom);
            $directDeliveryQuery->whereDate('delivery_date', '>=', $dateFrom);
            $transferQuery->whereDate('request_date', '>=', $dateFrom);
        }

        if ($request->filled('date_to')) {
            $dateTo = $request->date_to;
            $incomingQuery->whereDate('operation_date', '<=', $dateTo);
            $outgoingQuery->whereDate('transaction_date', '<=', $dateTo);
            $directDeliveryQuery->whereDate('delivery_date', '<=', $dateTo);
            $transferQuery->whereDate('request_date', '<=', $dateTo);
        }

        // Filter by division if specified
        if ($request->filled('division_id')) {
            $divisionId = $request->division_id;

            // IncomingOperation has direct division_id
            $incomingQuery->where('division_id', $divisionId);

            // OutgoingTransaction - filter by direct division_id or through project
            $outgoingQuery->where(function ($q) use ($divisionId) {
                $q->where('division_id', $divisionId)
                  ->orWhereHas('project', function ($projectQ) use ($divisionId) {
                      $projectQ->where('project_division_id', $divisionId);
                  });
            });

            // DirectDelivery - filter by project_division_id or through project
            $directDeliveryQuery->where(function ($q) use ($divisionId) {
                $q->where('project_division_id', $divisionId)
                  ->orWhereHas('project', function ($projectQ) use ($divisionId) {
                      $projectQ->where('project_division_id', $divisionId);
                  });
            });

            // MaterialTransferRequest - filter through transferrer or receiver project
            $transferQuery->where(function ($q) use ($divisionId) {
                $q->whereHas('transferrerProject', function ($projectQ) use ($divisionId) {
                    $projectQ->where('project_division_id', $divisionId);
                })->orWhereHas('receiverProject', function ($projectQ) use ($divisionId) {
                    $projectQ->where('project_division_id', $divisionId);
                });
            });
        }

        // Get the data
        $incoming = $incomingQuery->get();
        $outgoing = $outgoingQuery->get();
        $directDeliveries = $directDeliveryQuery->get();
        $transfers = $transferQuery->get();

        // Format and merge all transactions
        foreach ($incoming as $item) {
            $transactions->push([
                'id' => $item->operation_number,
                'date' => $item->operation_date,
                'type' => ucfirst($item->operation_type),
                'type_badge' => 'success',
                'reference' => $item->lpo_number ?: $item->delivery_note_number,
                'entity' => $item->supplier?->name ?: 'N/A',
                'project' => $item->project?->project_name ?: 'N/A',
                'status' => ucfirst($item->status),
                'status_badge' => $this->getStatusBadge($item->status),
                'created_by' => $item->receiver?->name ?: 'System',
                'model' => 'incoming',
                'model_id' => $item->id
            ]);
        }

        foreach ($outgoing as $item) {
            $transactions->push([
                'id' => $item->transaction_number,
                'date' => $item->transaction_date,
                'type' => 'Outgoing',
                'type_badge' => 'info',
                'reference' => $item->reference_number,
                'entity' => $item->project?->project_name ?: 'N/A',
                'project' => $item->project?->project_name ?: 'N/A',
                'status' => ucfirst($item->status),
                'status_badge' => $this->getStatusBadge($item->status),
                'created_by' => $item->requester?->name ?: 'System',
                'model' => 'outgoing',
                'model_id' => $item->id
            ]);
        }

        foreach ($directDeliveries as $item) {
            $transactions->push([
                'id' => $item->reference_number ?: $item->delivery_number,
                'date' => $item->delivery_date,
                'type' => 'Direct Delivery',
                'type_badge' => 'warning',
                'reference' => $item->reference_number ?: $item->delivery_number,
                'entity' => $item->supplier?->name ?: 'Direct Delivery',
                'project' => $item->project?->project_name ?: 'N/A',
                'status' => ucfirst($item->status),
                'status_badge' => $this->getStatusBadge($item->status),
                'created_by' => 'System',
                'model' => 'direct_delivery',
                'model_id' => $item->id
            ]);
        }

        foreach ($transfers as $item) {
            $transactions->push([
                'id' => $item->transfer_request_number,
                'date' => $item->request_date,
                'type' => 'Transfer',
                'type_badge' => 'primary',
                'reference' => $item->transfer_request_number,
                'entity' => $item->transferrerProject?->project_name ?: 'N/A',
                'project' => $item->receiverProject?->project_name ?: 'N/A',
                'status' => ucfirst($item->status),
                'status_badge' => $this->getStatusBadge($item->status),
                'created_by' => $item->requestedBy?->name ?: 'System',
                'model' => 'transfer',
                'model_id' => $item->id
            ]);
        }

        // Filter by type if specified
        if ($request->filled('type')) {
            $typeFilter = $request->type;
            $transactions = $transactions->filter(function($item) use ($typeFilter) {
                switch($typeFilter) {
                    case 'supplier_delivery':
                        return in_array(strtolower($item['type']), ['supply_delivery', 'supplier_delivery']);
                    case 'outgoing':
                        return strtolower($item['type']) === 'outgoing';
                    case 'direct_delivery':
                        return strtolower($item['type']) === 'direct delivery';
                    case 'transfer':
                        return strtolower($item['type']) === 'transfer';
                    default:
                        return true;
                }
            });
        }

        // Filter by status if specified
        if ($request->filled('status')) {
            $statusFilter = strtolower($request->status);
            $transactions = $transactions->filter(function($item) use ($statusFilter) {
                return strtolower($item['status']) === $statusFilter;
            });
        }

        // Sort by date descending
        $transactions = $transactions->sortByDesc(function ($item) {
            return $item['date'];
        })->values();

        // Calculate statistics (before pagination)
        $stats = [
            'total' => $transactions->count(),
            'incoming' => $incoming->count(),
            'outgoing' => $outgoing->count(),
            'direct_deliveries' => $directDeliveries->count(),
            'transfers' => $transfers->count()
        ];

        // Paginate the transactions
        $perPage = 25;
        $currentPage = request()->input('page', 1);
        $paginatedItems = $transactions->slice(($currentPage - 1) * $perPage, $perPage)->values();

        $transactions = new \Illuminate\Pagination\LengthAwarePaginator(
            $paginatedItems,
            $transactions->count(),
            $perPage,
            $currentPage,
            ['path' => request()->url()]
        );

        // Append query parameters to pagination links
        $transactions->appends($request->all());

        // Get divisions for filter dropdown
        $divisions = ProjectDivision::where('status', 'active')->orderBy('division_name')->get();

        return view('warehouse.reports.transactions', compact('transactions', 'stats', 'divisions'));
    }

    private function getStatusBadge($status)
    {
        switch(strtolower($status)) {
            case 'completed':
            case 'approved':
                return 'success';
            case 'pending':
            case 'draft':
                return 'warning';
            case 'cancelled':
            case 'rejected':
                return 'danger';
            case 'in_progress':
            case 'partial':
                return 'info';
            default:
                return 'secondary';
        }
    }

    public function aging()
    {
        return view('warehouse.reports.aging');
    }

    public function lowStock()
    {
        return view('warehouse.reports.low-stock');
    }

    public function supplierPerformance()
    {
        return view('warehouse.reports.supplier-performance');
    }

    public function incoming(Request $request)
    {
        $allIncomingItems = collect();

        // Get data from IncomingOperationItems (standard incoming operations)
        $operationQuery = IncomingOperationItem::with([
            'operation.supplier',
            'operation.division',
            'item.category'
        ]);

        // Apply filters for operations
        if ($request->filled('supplier_id')) {
            $operationQuery->whereHas('operation', function ($q) use ($request) {
                $q->where('supplier_id', $request->supplier_id);
            });
        }

        if ($request->filled('division_id')) {
            $operationQuery->whereHas('operation', function ($q) use ($request) {
                $q->where('division_id', $request->division_id);
            });
        }

        if ($request->filled('category_id')) {
            $operationQuery->whereHas('item', function ($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        if ($request->filled('operation_type')) {
            $operationQuery->whereHas('operation', function ($q) use ($request) {
                $q->where('operation_type', $request->operation_type);
            });
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $operationQuery->where(function ($q) use ($search) {
                $q->whereHas('item', function ($itemQuery) use ($search) {
                    $itemQuery->where('item_code', 'like', "%{$search}%")
                             ->orWhere('item_description', 'like', "%{$search}%");
                })->orWhereHas('operation', function ($opQuery) use ($search) {
                    $opQuery->where('lpo_number', 'like', "%{$search}%")
                           ->orWhere('operation_number', 'like', "%{$search}%");
                })->orWhereHas('operation.supplier', function ($supQuery) use ($search) {
                    $supQuery->where('name', 'like', "%{$search}%")
                            ->orWhere('supplier_code', 'like', "%{$search}%");
                });
            });
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $operationQuery->whereHas('operation', function ($q) use ($request) {
                $q->whereBetween('operation_date', [$request->date_from, $request->date_to]);
            });
        }

        // Get standard incoming operation items
        $operationItems = $operationQuery->orderBy('created_at', 'desc')->get()->map(function ($incomingItem) {
            // Handle division name - check division_id first, then legacy division column
            $divisionName = null;
            $operation = $incomingItem->operation;

            if ($operation) {
                // First check if division_id exists and get related ProjectDivision
                if ($operation->division_id) {
                    // Use getRelation to get the eager-loaded relationship (not the column)
                    $divisionRelation = $operation->relationLoaded('division') ? $operation->getRelation('division') : null;
                    if ($divisionRelation) {
                        $divisionName = $divisionRelation->division_name;
                    } else {
                        // Fallback: load division if not eager loaded
                        $div = \App\Models\Warehouse\ProjectDivision::find($operation->division_id);
                        $divisionName = $div?->division_name;
                    }
                } elseif ($operation->getAttributes()['division'] ?? null) {
                    // Legacy format: division column might contain name or ID as string
                    $legacyDivision = $operation->getAttributes()['division'];
                    if (is_numeric($legacyDivision)) {
                        // It's an ID, look up the name
                        $div = \App\Models\Warehouse\ProjectDivision::find($legacyDivision);
                        $divisionName = $div?->division_name;
                    } else {
                        // It's a name
                        $divisionName = $legacyDivision;
                    }
                }
            }

            return (object) [
                'id' => $incomingItem->id,
                'operation_id' => $incomingItem->operation?->id,
                'supplier_code' => $incomingItem->operation?->supplier?->supplier_code ?? 'N/A',
                'supplier_name' => $incomingItem->operation?->supplier?->name ?? 'Unknown',
                'lpo_number' => $incomingItem->operation?->lpo_number ?? 'N/A',
                'division_name' => $divisionName,
                'item_code' => $incomingItem->item->item_code,
                'item_description' => $incomingItem->item->item_description,
                'unit' => $incomingItem->item->unit_of_measure,
                'quantity_ordered' => $incomingItem->quantity_ordered,
                'quantity' => $incomingItem->quantity_delivered,
                'unit_price' => $incomingItem->unit_price_aed,
                'total_price' => $incomingItem->total_price_aed,
                'production_date' => $incomingItem->production_date,
                'expiry_date' => $incomingItem->expiry_date,
                'operation_date' => $incomingItem->operation?->operation_date ?? null,
                'operation_type' => $incomingItem->operation?->operation_type ?? 'N/A',
                'status' => $incomingItem->operation?->status ?? 'N/A',
                'quality_status' => $incomingItem->quality_status,
                'batch_number' => $incomingItem->batch_number
            ];
        });

        // Get data from IncomingItems (direct delivery transactions)
        $transactionQuery = \App\Models\Warehouse\IncomingItem::with([
            'transaction.supplier',
            'item.category'
        ]);

        // Apply filters for transactions
        if ($request->filled('supplier_id')) {
            $transactionQuery->whereHas('transaction', function ($q) use ($request) {
                $q->where('supplier_id', $request->supplier_id);
            });
        }

        if ($request->filled('category_id')) {
            $transactionQuery->whereHas('item', function ($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        if ($request->filled('operation_type') && $request->operation_type !== 'direct_delivery') {
            // If filtering by operation type and it's not direct_delivery, exclude transaction items
            $transactionQuery->whereRaw('1 = 0'); // This will return no results
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $transactionQuery->where(function ($q) use ($search) {
                $q->whereHas('item', function ($itemQuery) use ($search) {
                    $itemQuery->where('item_code', 'like', "%{$search}%")
                             ->orWhere('item_description', 'like', "%{$search}%");
                })->orWhereHas('transaction', function ($transQuery) use ($search) {
                    $transQuery->where('lpo_number', 'like', "%{$search}%")
                              ->orWhere('delivery_note_number', 'like', "%{$search}%");
                })->orWhereHas('transaction.supplier', function ($supQuery) use ($search) {
                    $supQuery->where('name', 'like', "%{$search}%")
                            ->orWhere('supplier_code', 'like', "%{$search}%");
                });
            });
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $transactionQuery->whereHas('transaction', function ($q) use ($request) {
                $q->whereBetween('transaction_date', [$request->date_from, $request->date_to]);
            });
        }

        // Get direct delivery transaction items
        $transactionItems = $transactionQuery->orderBy('created_at', 'desc')->get()->map(function ($incomingItem) {
            return (object) [
                'id' => 'T_' . $incomingItem->id, // Prefix to differentiate
                'operation_id' => null, // Direct deliveries don't have an operation
                'transaction_id' => $incomingItem->transaction?->id,
                'supplier_code' => $incomingItem->transaction?->supplier?->supplier_code ?? 'N/A',
                'supplier_name' => $incomingItem->transaction?->supplier?->name ?? 'Unknown',
                'lpo_number' => $incomingItem->transaction?->lpo_number ?? 'N/A',
                'division_name' => 'Direct Delivery', // Custom division label for direct deliveries
                'item_code' => $incomingItem->item->item_code,
                'item_description' => $incomingItem->item->item_description,
                'unit' => $incomingItem->item->unit_of_measure,
                'quantity_ordered' => $incomingItem->quantity_ordered ?? null,
                'quantity' => $incomingItem->quantity_received,
                'unit_price' => $incomingItem->unit_price,
                'total_price' => $incomingItem->total_price,
                'production_date' => $incomingItem->production_date,
                'expiry_date' => $incomingItem->expiry_date,
                'operation_date' => $incomingItem->transaction?->transaction_date ?? null,
                'operation_type' => $incomingItem->transaction?->transaction_type ?? 'direct_delivery',
                'status' => $incomingItem->transaction?->status ?? 'N/A',
                'quality_status' => $incomingItem->stock_status,
                'batch_number' => $incomingItem->batch_number
            ];
        });

        // Combine both collections and sort by operation_date
        $allIncomingItems = collect($operationItems)->merge(collect($transactionItems))
                                       ->sortByDesc(function ($item) {
                                           return $item->operation_date;
                                       })
                                       ->values();

        // Implement manual pagination for the combined collection
        $perPage = 25;
        $currentPage = request()->get('page', 1);
        $total = $allIncomingItems->count();
        $offset = ($currentPage - 1) * $perPage;
        $items = $allIncomingItems->slice($offset, $perPage);

        // Create a LengthAwarePaginator instance
        $incomingItems = new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $total,
            $perPage,
            $currentPage,
            [
                'path' => request()->url(),
                'pageName' => 'page',
            ]
        );

        // Append query parameters to pagination links
        $incomingItems->appends(request()->query());

        // Get filter options
        $suppliers = Supplier::all();
        $divisions = Department::all();
        $categories = ItemCategory::all();
        $operationTypes = ['purchase', 'supplier_delivery', 'site_return', 'direct_delivery'];

        return view('warehouse.reports.incoming', compact(
            'incomingItems', 'suppliers', 'divisions', 'categories', 'operationTypes'
        ));
    }

    public function outgoing(Request $request)
    {
        $query = OutgoingTransaction::with([
            'project.projectDivision',
            'division',
            'requester',
            'items.item.category'
        ])->withCount('items')
        // Only show actual material requests, not dispatch references
        // Exclude dispatch reference IDs which have suffixes:
        // -INV (Inventory Dispatch), -DD (Direct Delivery), -TR (Project Transfer), -REF (General Reference)
        // Example: MR-20251209-229-INV is a dispatch reference, MR-20251209-229 is the actual request
        ->whereNotNull('material_request_number')
        ->where(function($q) {
            $q->where('material_request_number', 'NOT LIKE', '%-INV')
              ->where('material_request_number', 'NOT LIKE', '%-DD')
              ->where('material_request_number', 'NOT LIKE', '%-TR')
              ->where('material_request_number', 'NOT LIKE', '%-REF');
        })
        // Also ensure we have a requested_by field (indicating it's an actual request, not a dispatch operation)
        ->whereNotNull('requested_by');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('material_request_number', 'like', "%{$search}%")
                  ->orWhereHas('project', function ($projectQ) use ($search) {
                      $projectQ->where('project_name', 'like', "%{$search}%")
                               ->orWhere('project_code', 'like', "%{$search}%");
                  })
                  ->orWhereHas('requester', function ($userQ) use ($search) {
                      $userQ->where('name', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('division_id')) {
            $divisionId = $request->division_id;
            $query->where(function ($q) use ($divisionId) {
                // Filter by direct division_id on transaction
                $q->where('division_id', $divisionId)
                  // Or filter through project's division
                  ->orWhereHas('project', function ($projectQ) use ($divisionId) {
                      $projectQ->where('project_division_id', $divisionId);
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $query->whereBetween('transaction_date', [$request->date_from, $request->date_to]);
        }

        // Get the outgoing requests with calculated values using the same logic as material issue page
        $allOutgoingRequests = $query->orderBy('created_at', 'desc')->orderBy('transaction_date', 'desc')->get();

        // Add calculated properties to each transaction
        foreach ($allOutgoingRequests as $transaction) {
            // Get fulfillments for this transaction grouped by item_id (exact same logic as MaterialIssueController lines 108-110)
            $fulfillments = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
                ->get()
                ->groupBy('item_id');

            // Calculate remaining quantities for each item (exact same logic as MaterialIssueController lines 113-120)
            $totalRequested = 0;
            $totalReleased = 0;

            foreach ($transaction->items as $item) {
                $totalRequested += $item->quantity_requested;

                $issuedQty = 0;
                if (isset($fulfillments[$item->item_id])) {
                    $issuedQty = $fulfillments[$item->item_id]->sum('quantity_fulfilled');
                }
                $totalReleased += $issuedQty;

                // Set the same properties as MaterialIssueController for consistency
                $item->quantity_fulfilled = $issuedQty;
                $item->quantity_remaining = $item->quantity_requested - $issuedQty;
            }

            // Calculate balance
            $totalBalance = $totalRequested - $totalReleased;

            // Determine overall fulfillment status
            $fulfillmentStatus = 'pending';
            if ($totalReleased > 0) {
                $fulfillmentStatus = $totalBalance <= 0 ? 'completed' : 'partial';
            }

            // Count dispatch methods used based on fulfillments only (clean approach matching material issue page)
            $dispatchMethods = [];
            $fulfillmentTypes = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $transaction->id)
                ->distinct('fulfillment_type')
                ->pluck('fulfillment_type')
                ->toArray();

            foreach ($fulfillmentTypes as $type) {
                switch ($type) {
                    case 'inventory':
                    case 'inventory_dispatch':
                        $dispatchMethods[] = 'Inventory';
                        break;
                    case 'direct_delivery':
                        $dispatchMethods[] = 'Direct Delivery';
                        break;
                    case 'project_transfer':
                    case 'transfer':
                    case 'material_transfer':
                        $dispatchMethods[] = 'Project Transfer';
                        break;
                }
            }

            // Remove duplicates and sort
            $dispatchMethods = array_unique($dispatchMethods);

            // Add calculated properties directly to the Eloquent model
            $transaction->total_requested = $totalRequested;
            $transaction->total_released = $totalReleased;
            $transaction->total_balance = $totalBalance;
            $transaction->fulfillment_status = $fulfillmentStatus;
            $transaction->dispatch_methods = $dispatchMethods;
            $transaction->dispatch_methods_text = empty($dispatchMethods) ? 'Pending' : implode(', ', $dispatchMethods);
        }

        // Filter by fulfillment status if requested
        if ($request->filled('fulfillment_status')) {
            $allOutgoingRequests = $allOutgoingRequests->filter(function ($request_item) use ($request) {
                return $request_item->fulfillment_status === $request->fulfillment_status;
            });
        }

        // Implement manual pagination for the collection
        $perPage = 25;
        $currentPage = request()->get('page', 1);
        $total = $allOutgoingRequests->count();
        $offset = ($currentPage - 1) * $perPage;
        $items = $allOutgoingRequests->slice($offset, $perPage);

        // Create a LengthAwarePaginator instance
        $outgoingRequests = new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $total,
            $perPage,
            $currentPage,
            [
                'path' => request()->url(),
                'pageName' => 'page',
            ]
        );

        // Append query parameters to pagination links
        $outgoingRequests->appends(request()->query());

        // Get filter options
        $projects = Project::all();
        $divisions = ProjectDivision::where('status', 'active')->orderBy('division_name')->get();
        $categories = ItemCategory::all();
        $statuses = ['pending', 'approved', 'partially_fulfilled', 'completed', 'cancelled'];
        $fulfillmentStatuses = ['pending', 'partial', 'completed'];

        return view('warehouse.reports.outgoing', compact(
            'outgoingRequests', 'projects', 'divisions', 'categories', 'statuses', 'fulfillmentStatuses'
        ));
    }

    public function incomingItemDetails($itemId)
    {
        $incomingItem = IncomingOperationItem::with([
            'operation.supplier',
            'operation.division',
            'operation.receiver',
            'operation.inspector',
            'operation.approver',
            'item.category'
        ])->findOrFail($itemId);

        // Get detailed operation information
        $operationDetails = (object) [
            'id' => $incomingItem->operation->id,
            'operation_number' => $incomingItem->operation->operation_number,
            'operation_type' => $incomingItem->operation->operation_type,
            'operation_date' => $incomingItem->operation->operation_date,
            'status' => $incomingItem->operation->status,
            'lpo_number' => $incomingItem->operation->lpo_number,
            'lpo_status' => $incomingItem->operation->lpo_status,
            'delivery_note_number' => $incomingItem->operation->delivery_note_number,
            'supplier_name' => $incomingItem->operation->supplier->name ?? 'Unknown',
            'supplier_code' => $incomingItem->operation->supplier->supplier_code ?? 'N/A',
            'division' => $incomingItem->operation->division ?? 'Unknown',
            'received_by' => $incomingItem->operation->receiver->name ?? $incomingItem->operation->received_by_name ?? 'Unknown',
            'inspector' => $incomingItem->operation->inspector->name ?? 'Not assigned',
            'inspection_datetime' => $incomingItem->operation->inspection_datetime,
            'inspection_result' => $incomingItem->operation->inspection_result,
            'inspection_notes' => $incomingItem->operation->inspection_notes,
            'approved_by' => $incomingItem->operation->approver->name ?? 'Not approved',
            'approved_at' => $incomingItem->operation->approved_at,
            'total_amount' => $incomingItem->operation->total_amount_aed,
            'currency' => $incomingItem->operation->currency,
            'notes' => $incomingItem->operation->notes,
            'created_at' => $incomingItem->operation->created_at,
            'updated_at' => $incomingItem->operation->updated_at
        ];

        // Get detailed item information
        $itemDetails = (object) [
            'id' => $incomingItem->item->id,
            'item_code' => $incomingItem->item->item_code,
            'item_description' => $incomingItem->item->item_description,
            'category_name' => $incomingItem->item->category->category_name ?? 'Unknown',
            'unit_of_measure' => $incomingItem->item->unit_of_measure,
            'quantity_ordered' => $incomingItem->quantity_ordered,
            'quantity_delivered' => $incomingItem->quantity_delivered,
            'quantity_accepted' => $incomingItem->quantity_accepted,
            'quantity_rejected' => $incomingItem->quantity_rejected,
            'unit_price' => $incomingItem->unit_price,
            'unit_price_aed' => $incomingItem->unit_price_aed,
            'total_price_aed' => $incomingItem->total_price_aed,
            'currency' => $incomingItem->currency,
            'batch_number' => $incomingItem->batch_number,
            'serial_number' => $incomingItem->serial_number,
            'production_date' => $incomingItem->production_date,
            'expiry_date' => $incomingItem->expiry_date,
            'quality_status' => $incomingItem->quality_status,
            'quality_notes' => $incomingItem->quality_notes,
            'rejection_reason' => $incomingItem->rejection_reason,
            'storage_location' => $incomingItem->storage_location,
            'manufacturer' => $incomingItem->manufacturer,
            'country_of_origin' => $incomingItem->country_of_origin,
            'warranty_period_months' => $incomingItem->warranty_period_months,
            'warranty_expiry' => $incomingItem->warranty_expiry,
            'hazardous_material' => $incomingItem->hazardous_material,
            'days_until_expiry' => $incomingItem->days_until_expiry,
            'acceptance_rate' => $incomingItem->acceptance_rate,
            'rejection_rate' => $incomingItem->rejection_rate,
            'created_at' => $incomingItem->created_at,
            'updated_at' => $incomingItem->updated_at
        ];

        return view('warehouse.reports.incoming-details', compact('operationDetails', 'itemDetails'));
    }

    public function outgoingItemDetails($itemId)
    {
        $outgoingItem = OutgoingItem::with([
            'transaction.project',
            'transaction.division',
            'transaction.requester',
            'item.category'
        ])->findOrFail($itemId);

        // Get detailed transaction information
        $transactionDetails = (object) [
            'id' => $outgoingItem->transaction->id,
            'material_request_number' => $outgoingItem->transaction->material_request_number,
            'transaction_date' => $outgoingItem->transaction->transaction_date,
            'status' => $outgoingItem->transaction->status,
            'project_name' => $outgoingItem->transaction->project->project_name ?? 'Unknown',
            'project_code' => $outgoingItem->transaction->project->project_code ?? $outgoingItem->transaction->project_id,
            'division_name' => $outgoingItem->transaction->division->division_name ?? 'Unknown',
            'requester_name' => $outgoingItem->transaction->requester->name ?? 'Unknown',
            'receiver_name' => $outgoingItem->transaction->receiver_name,
            'delivery_location' => $outgoingItem->transaction->delivery_location,
            'purpose' => $outgoingItem->transaction->purpose,
            'remarks' => $outgoingItem->transaction->remarks,
            'created_at' => $outgoingItem->transaction->created_at,
            'updated_at' => $outgoingItem->transaction->updated_at
        ];

        // Get item information
        $itemDetails = (object) [
            'id' => $outgoingItem->item->id,
            'item_code' => $outgoingItem->item->item_code,
            'item_description' => $outgoingItem->item->item_description,
            'category_name' => $outgoingItem->item->category->category_name ?? 'Unknown',
            'unit_of_measure' => $outgoingItem->item->unit_of_measure,
            'quantity_requested' => $outgoingItem->quantity_requested,
            'quantity_released' => $outgoingItem->quantity_released,
            'quantity_balance' => $outgoingItem->quantity_balance,
            'unit_price' => $outgoingItem->unit_price,
            'total_price' => $outgoingItem->total_price,
            'fulfillment_status' => $this->determineFulfillmentStatus($outgoingItem),
            'release_date' => $outgoingItem->release_date,
            'batch_number' => $outgoingItem->batch_number,
            'notes' => $outgoingItem->notes,
            'created_at' => $outgoingItem->created_at,
            'updated_at' => $outgoingItem->updated_at
        ];

        return view('warehouse.reports.outgoing-details', compact('transactionDetails', 'itemDetails'));
    }

    public function outgoingRequestDetails($requestId)
    {
        $request = OutgoingTransaction::with([
            'project.projectDivision',
            'division',
            'requester',
            'items.item.category'
        ])->findOrFail($requestId);

        // Get fulfillments for this transaction (same logic as MaterialIssueController and main report)
        $fulfillments = \App\Models\Warehouse\OutgoingFulfillment::where('outgoing_transaction_id', $request->id)
            ->get()
            ->groupBy('item_id');

        // Calculate remaining quantities for each item and get dispatch details
        $totalRequested = 0;
        $totalReleased = 0;

        foreach ($request->items as $item) {
            $issuedQty = 0;
            $dispatchMethods = [];

            if (isset($fulfillments[$item->item_id])) {
                // Get fulfillment breakdown by type
                $itemFulfillments = $fulfillments[$item->item_id];
                $issuedQty = $itemFulfillments->sum('quantity_fulfilled');

                // Group by fulfillment type for display
                $fulfillmentsByType = $itemFulfillments->groupBy('fulfillment_type');

                // Inventory fulfillments
                $inventoryQty = $fulfillmentsByType->get('inventory', collect())->sum('quantity_fulfilled') +
                               $fulfillmentsByType->get('inventory_dispatch', collect())->sum('quantity_fulfilled');
                if ($inventoryQty > 0) {
                    $dispatchMethods[] = [
                        'type' => 'Inventory Dispatch',
                        'quantity' => $inventoryQty,
                        'date' => ($itemFulfillments->where('fulfillment_type', 'inventory')->first()->created_at ??
                                 $itemFulfillments->where('fulfillment_type', 'inventory_dispatch')->first()->created_at ?? now())
                                 ->format('d-M-Y g:i A'),
                        'icon' => 'inventory',
                        'color' => 'success'
                    ];
                }

                // Direct delivery fulfillments
                $deliveryQty = $fulfillmentsByType->get('direct_delivery', collect())->sum('quantity_fulfilled');
                if ($deliveryQty > 0) {
                    $dispatchMethods[] = [
                        'type' => 'Direct Delivery',
                        'quantity' => $deliveryQty,
                        'date' => ($itemFulfillments->where('fulfillment_type', 'direct_delivery')->first()->created_at ?? now())
                                 ->format('d-M-Y g:i A'),
                        'icon' => 'local_shipping',
                        'color' => 'info'
                    ];
                }

                // Project transfer fulfillments
                $transferQty = $fulfillmentsByType->get('project_transfer', collect())->sum('quantity_fulfilled');
                if ($transferQty > 0) {
                    $dispatchMethods[] = [
                        'type' => 'Project Transfer',
                        'quantity' => $transferQty,
                        'date' => ($itemFulfillments->where('fulfillment_type', 'project_transfer')->first()->created_at ?? now())
                                 ->format('d-M-Y g:i A'),
                        'icon' => 'swap_horiz',
                        'color' => 'primary'
                    ];
                }
            }

            // Set calculated properties
            $item->quantity_fulfilled = $issuedQty;
            $item->quantity_remaining = max(0, $item->quantity_requested - $issuedQty);
            $item->dispatch_methods = $dispatchMethods;
            $item->fulfillment_status = $item->quantity_remaining == 0 ? 'Fully Dispatched' :
                                      ($issuedQty == 0 ? 'Pending' : 'Partially Dispatched');

            $totalRequested += $item->quantity_requested;
            $totalReleased += $issuedQty;
        }

        // Request summary using calculated values
        $requestSummary = (object) [
            'material_request_number' => $request->material_request_number ?: 'MR-' . str_pad($request->id, 6, '0', STR_PAD_LEFT),
            'transaction_date' => $request->transaction_date,
            'project_name' => $request->project->project_name ?? 'N/A',
            'division_name' => $request->division->division_name ??
                              ($request->project->projectDivision->division_name ?? 'Main'),
            'requester_name' => $request->requester->name ?? 'N/A',
            'receiver_name' => $request->receiver_name,
            'delivery_location' => $request->delivery_location,
            'purpose' => $request->purpose,
            'total_items' => $request->items->count(),
            'total_requested' => $totalRequested,
            'total_released' => $totalReleased,
            'total_balance' => $totalRequested - $totalReleased,
            'overall_status' => ($totalRequested - $totalReleased) == 0 ? 'Fully Dispatched' :
                              ($totalReleased == 0 ? 'Pending' : 'Partially Dispatched')
        ];

        return view('warehouse.reports.outgoing-request-details', compact('requestSummary', 'request'));
    }

    public function returns(Request $request)
    {
        // Site returns are stored as IncomingOperationItems with operation_type = 'site_return'
        // They link to original outgoing items via outgoing_item_id
        $query = IncomingOperationItem::with([
            'operation.receiver',
            'item.category'
        ])->whereHas('operation', function ($q) {
            $q->where('operation_type', 'site_return');
        })->where(function($q) {
            $q->whereNotNull('outgoing_item_id')
              ->orWhereNotNull('direct_delivery_item_id')
              ->orWhereNotNull('material_transfer_item_id');
        });

        // Apply filters
        if ($request->filled('project_id')) {
            $query->whereHas('operation', function ($q) use ($request) {
                $q->where('project_id', $request->project_id);
            });
        }

        if ($request->filled('division_id')) {
            $query->whereHas('operation', function ($q) use ($request) {
                $q->where('division_id', $request->division_id);
            });
        }

        if ($request->filled('category_id')) {
            $query->whereHas('item', function ($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        if ($request->filled('quality_status')) {
            $query->where('quality_status', $request->quality_status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->whereHas('item', function ($itemQuery) use ($search) {
                    $itemQuery->where('item_code', 'like', "%{$search}%")
                             ->orWhere('item_description', 'like', "%{$search}%");
                })->orWhereHas('operation', function ($opQuery) use ($search) {
                    $opQuery->where('operation_number', 'like', "%{$search}%");
                })->orWhereHas('operation.project', function ($projQuery) use ($search) {
                    $projQuery->where('project_name', 'like', "%{$search}%");
                });
            });
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $query->whereHas('operation', function ($q) use ($request) {
                $q->whereBetween('operation_date', [$request->date_from, $request->date_to]);
            });
        }

        // Get the return items with calculated values and pagination
        $allReturnItems = $query->orderBy('created_at', 'desc')->get()->map(function ($returnItem) {
            // Get the source reference info based on the return type
            $sourceInfo = null;
            $projectInfo = null;

            if ($returnItem->outgoing_item_id) {
                // Material Request return
                $outgoingItem = OutgoingItem::with(['transaction.project', 'transaction.division'])
                    ->find($returnItem->outgoing_item_id);
                $sourceInfo = $outgoingItem;
                $projectInfo = $outgoingItem?->transaction?->project;
            } elseif ($returnItem->direct_delivery_item_id) {
                // Direct Delivery return
                $directDeliveryItem = \App\Models\DirectDeliveryItem::with(['directDelivery.project'])
                    ->find($returnItem->direct_delivery_item_id);
                $sourceInfo = $directDeliveryItem;
                $projectInfo = $directDeliveryItem?->directDelivery?->project;
            } elseif ($returnItem->material_transfer_item_id) {
                // Material Transfer return
                $materialTransferItem = \App\Models\Warehouse\MaterialTransferRequestItem::with(['transferRequest.receiverProject'])
                    ->find($returnItem->material_transfer_item_id);
                $sourceInfo = $materialTransferItem;
                $projectInfo = $materialTransferItem?->transferRequest?->receiverProject;
            }

            // Get production/expiry dates from the original incoming operation for this item
            $originalIncoming = IncomingOperationItem::where('item_id', $returnItem->item_id)
                ->where(function($q) {
                    $q->whereNotNull('production_date')->orWhereNotNull('expiry_date');
                })
                ->orderBy('created_at', 'desc')
                ->first();

            return (object) [
                'id' => $returnItem->id,
                'reference_number' => $this->getReferenceNumber($returnItem, $sourceInfo),
                'reference_type' => $this->getReferenceType($returnItem),
                'project_name' => $projectInfo?->project_name ?? $returnItem->operation->project?->project_name ?? 'Unknown',
                'project_number' => $projectInfo?->project_number ?? $projectInfo?->project_code ?? ($projectInfo?->id ?? 'N/A'),
                'division' => $this->getDivisionName($returnItem, $sourceInfo) ?? 'Unknown',
                'type_of_goods' => $returnItem->item->category->category_name ?? 'Unknown',
                'item_code' => $returnItem->item->item_code,
                'item_description' => $returnItem->item->item_description,
                'unit' => $returnItem->item->unit_of_measure,
                'quantity' => $returnItem->quantity_delivered, // For returns, this is quantity returned
                'unit_price' => $returnItem->unit_price_aed ?: ($originalIncoming->unit_price_aed ?? 0),
                'total_price' => $returnItem->total_price_aed ?: (($originalIncoming->unit_price_aed ?? 0) * $returnItem->quantity_delivered),
                'production_date' => $originalIncoming->production_date ?? null,
                'expiry_date' => $originalIncoming->expiry_date ?? null,
                'quality_status' => $returnItem->quality_status,
                'batch_number' => $returnItem->batch_number,
                'operation_number' => $returnItem->operation->operation_number,
                'operation_date' => $returnItem->operation->operation_date,
                'return_reason' => $returnItem->operation->notes ?? 'Site return',
                'received_by' => $returnItem->operation->receiver->name ?? $returnItem->operation->received_by_name ?? 'Unknown'
            ];
        });

        // Implement manual pagination for the collection
        $perPage = 25;
        $currentPage = request()->get('page', 1);
        $total = $allReturnItems->count();
        $offset = ($currentPage - 1) * $perPage;
        $items = $allReturnItems->slice($offset, $perPage);

        // Create a LengthAwarePaginator instance
        $returnItems = new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $total,
            $perPage,
            $currentPage,
            [
                'path' => request()->url(),
                'pageName' => 'page',
            ]
        );

        // Append query parameters to pagination links
        $returnItems->appends(request()->query());

        // Calculate division breakdown for returns
        $divisionBreakdown = $allReturnItems->filter(function($item) {
            return !empty($item->division) && $item->division !== 'Unknown';
        })->groupBy('division')->map(function ($items, $division) {
            return [
                'division' => $division,
                'total_value' => $items->sum('total_price'),
                'total_quantity' => $items->sum('quantity'),
                'item_count' => $items->count(),
            ];
        })->sortByDesc('total_value')->values();

        // Get filter options
        $projects = Project::all();
        $divisions = \App\Models\Warehouse\ProjectDivision::where('status', 'active')->orderBy('division_name')->get();
        $categories = ItemCategory::all();
        $qualityStatuses = ['pending', 'passed', 'failed', 'conditional'];

        return view('warehouse.reports.returns', compact(
            'returnItems', 'projects', 'divisions', 'categories', 'qualityStatuses', 'divisionBreakdown'
        ));
    }

    public function returnItemDetails($itemId)
    {
        // Get the site return item (stored as IncomingOperationItem)
        $returnItem = IncomingOperationItem::with([
            'operation.receiver',
            'operation.inspector',
            'operation.approver',
            'item.category'
        ])->whereHas('operation', function ($q) {
            $q->where('operation_type', 'site_return');
        })->findOrFail($itemId);

        // Get the source reference info based on the return type
        $sourceInfo = null;
        $projectInfo = null;
        $divisionInfo = null;

        if ($returnItem->outgoing_item_id) {
            // Material Request return
            $outgoingItem = OutgoingItem::with(['transaction.project', 'transaction.division'])
                ->find($returnItem->outgoing_item_id);
            $sourceInfo = $outgoingItem;
            $projectInfo = $outgoingItem?->transaction?->project;
            $divisionInfo = $outgoingItem?->transaction?->division;
        } elseif ($returnItem->direct_delivery_item_id) {
            // Direct Delivery return
            $directDeliveryItem = \App\Models\DirectDeliveryItem::with(['directDelivery.project'])
                ->find($returnItem->direct_delivery_item_id);
            $sourceInfo = $directDeliveryItem;
            $projectInfo = $directDeliveryItem?->directDelivery?->project;
            $divisionInfo = $directDeliveryItem?->directDelivery?->project?->projectDivision;
        } elseif ($returnItem->material_transfer_item_id) {
            // Material Transfer return
            $materialTransferItem = \App\Models\Warehouse\MaterialTransferRequestItem::with(['transferRequest.receiverProject'])
                ->find($returnItem->material_transfer_item_id);
            $sourceInfo = $materialTransferItem;
            $projectInfo = $materialTransferItem?->transferRequest?->receiverProject;
            $divisionInfo = $materialTransferItem?->transferRequest?->receiverProject?->projectDivision;
        }

        // Get production/expiry dates from the original incoming operation for this item
        $originalIncoming = IncomingOperationItem::where('item_id', $returnItem->item_id)
            ->where(function($q) {
                $q->whereNotNull('production_date')->orWhereNotNull('expiry_date');
            })
            ->orderBy('created_at', 'desc')
            ->first();

        // Get detailed operation information
        $operationDetails = (object) [
            'id' => $returnItem->operation->id,
            'operation_number' => $returnItem->operation->operation_number,
            'operation_type' => $returnItem->operation->operation_type,
            'operation_date' => $returnItem->operation->operation_date,
            'status' => $returnItem->operation->status,
            'project_name' => $projectInfo?->project_name ?? $returnItem->operation->project?->project_name ?? 'Unknown',
            'project_code' => $projectInfo?->project_code ?? $projectInfo?->project_number ?? ($projectInfo?->id ?? 'N/A'),
            'division_name' => $divisionInfo?->division_name ?? 'Unknown',
            'return_reason' => $returnItem->operation->notes ?? 'Site return',
            'received_by' => $returnItem->operation->receiver->name ?? $returnItem->operation->received_by_name ?? 'Unknown',
            'inspector' => $returnItem->operation->inspector->name ?? 'Not assigned',
            'inspection_datetime' => $returnItem->operation->inspection_datetime,
            'inspection_result' => $returnItem->operation->inspection_result,
            'inspection_notes' => $returnItem->operation->inspection_notes,
            'approved_by' => $returnItem->operation->approver->name ?? 'Not approved',
            'approved_at' => $returnItem->operation->approved_at,
            'notes' => $returnItem->operation->notes,
            'created_at' => $returnItem->operation->created_at,
            'updated_at' => $returnItem->operation->updated_at
        ];

        // Get detailed item information
        $itemDetails = (object) [
            'id' => $returnItem->item->id,
            'item_code' => $returnItem->item->item_code,
            'item_description' => $returnItem->item->item_description,
            'category_name' => $returnItem->item->category->category_name ?? 'Unknown',
            'unit_of_measure' => $returnItem->item->unit_of_measure,
            'quantity_returned' => $returnItem->quantity_delivered, // For returns, this is quantity returned
            'unit_price' => $returnItem->unit_price_aed ?: ($originalIncoming->unit_price_aed ?? 0),
            'total_value' => $returnItem->total_price_aed ?: (($originalIncoming->unit_price_aed ?? 0) * $returnItem->quantity_delivered),
            'production_date' => $originalIncoming->production_date ?? null,
            'expiry_date' => $originalIncoming->expiry_date ?? null,
            'quality_status' => $returnItem->quality_status,
            'quality_notes' => $returnItem->quality_notes,
            'rejection_reason' => $returnItem->rejection_reason,
            'batch_number' => $returnItem->batch_number,
            'serial_number' => $returnItem->serial_number,
            'storage_location' => $returnItem->storage_location,
            'days_until_expiry' => $originalIncoming ? $originalIncoming->days_until_expiry : null,
            'acceptance_rate' => $returnItem->acceptance_rate,
            'created_at' => $returnItem->created_at,
            'updated_at' => $returnItem->updated_at
        ];

        return view('warehouse.reports.returns-details', compact('operationDetails', 'itemDetails'));
    }

    /**
     * Project to Project Transfer Report
     */
    public function projectTransfers(Request $request)
    {
        // Get Material Transfer Requests with related data
        $query = MaterialTransferRequest::with([
            'transferrerProject',
            'receiverProject',
            'requestedBy',
            'transferrerApprovedBy',
            'receiverApprovedBy',
            'items.item.category'
        ]);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('request_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('request_date', '<=', $request->date_to);
        }

        if ($request->filled('source_project')) {
            $query->where('transferrer_project_id', $request->source_project);
        }

        if ($request->filled('destination_project')) {
            $query->where('receiver_project_id', $request->destination_project);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('transfer_request_number', 'like', "%{$search}%")
                  ->orWhere('transfer_reason', 'like', "%{$search}%");
            });
        }

        // Order by date descending and paginate
        $perPage = 20;
        $transfers = $query->orderBy('request_date', 'desc')->paginate($perPage);

        // Append query parameters to pagination links
        $transfers->appends($request->all());

        // Get all projects for filter dropdown
        $projects = Project::orderBy('project_name')->get();

        // Calculate statistics
        $stats = [
            'total_transfers' => MaterialTransferRequest::count(),
            'pending' => MaterialTransferRequest::where('status', 'pending')->count(),
            'approved' => MaterialTransferRequest::where('status', 'approved')->count(),
            'completed' => MaterialTransferRequest::where('status', 'completed')->count(),
            'cancelled' => MaterialTransferRequest::where('status', 'cancelled')->count(),
        ];

        return view('warehouse.reports.project-transfers', compact('transfers', 'projects', 'stats'));
    }

    /**
     * Supplier Direct Delivery Report
     */
    public function directDeliveries(Request $request)
    {
        // Get Direct Deliveries with related data
        $query = DirectDelivery::with([
            'supplier',
            'project',
            'projectDivision',
            'items.item.category'
        ]);

        // Apply filters
        if ($request->filled('date_from')) {
            $query->whereDate('delivery_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('delivery_date', '<=', $request->date_to);
        }

        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('delivery_status')) {
            $query->where('status', $request->delivery_status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('reference_number', 'like', "%{$search}%")
                  ->orWhere('delivery_number', 'like', "%{$search}%")
                  ->orWhere('purchase_order_number', 'like', "%{$search}%")
                  ->orWhere('lpo_number', 'like', "%{$search}%");
            });
        }

        // Order by date descending and paginate
        $perPage = 20;
        $deliveries = $query->orderBy('delivery_date', 'desc')->paginate($perPage);

        // Append query parameters to pagination links
        $deliveries->appends($request->all());

        // Get suppliers and projects for filter dropdowns
        $suppliers = Supplier::orderBy('name')->get();
        $projects = Project::orderBy('project_name')->get();

        // Calculate statistics
        $stats = [
            'total_deliveries' => DirectDelivery::count(),
            'pending' => DirectDelivery::where('status', 'pending')->count(),
            'delivered' => DirectDelivery::where('status', 'delivered')->count(),
            'partial' => DirectDelivery::where('status', 'partial')->count(),
            'returned' => DirectDelivery::where('status', 'returned')->count(),
            'total_value' => DirectDelivery::sum('total_amount_aed')
        ];

        return view('warehouse.reports.direct-deliveries', compact('deliveries', 'suppliers', 'projects', 'stats'));
    }

    public function stockOnProject(Request $request)
    {
        $query = Project::select('projects.*')
            ->addSelect(DB::raw('(
                SELECT COALESCE(SUM(project_inventory.quantity_available), 0)
                FROM project_inventory
                WHERE project_inventory.project_id = projects.id
            ) as total_stock'))
            ->addSelect(DB::raw('(
                SELECT COUNT(DISTINCT project_inventory.item_id)
                FROM project_inventory
                WHERE project_inventory.project_id = projects.id
                AND project_inventory.quantity_available > 0
            ) as unique_items'))
            ->addSelect(DB::raw('(
                SELECT COALESCE(SUM(project_inventory.quantity_available * COALESCE(project_inventory.unit_price, 0)), 0)
                FROM project_inventory
                WHERE project_inventory.project_id = projects.id
            ) as total_value'))
            ->whereExists(function ($query) {
                $query->select(DB::raw(1))
                    ->from('project_inventory')
                    ->whereColumn('project_inventory.project_id', 'projects.id')
                    ->where('project_inventory.quantity_available', '>', 0);
            });

        // Apply filters
        if ($request->filled('project_status')) {
            $query->where('status', $request->project_status);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('project_name', 'like', "%{$search}%")
                  ->orWhere('project_number', 'like', "%{$search}%");
            });
        }

        if ($request->filled('date_from') && $request->filled('date_to')) {
            $query->whereBetween('created_at', [$request->date_from, $request->date_to]);
        }

        // Add project filter if specified
        if ($request->filled('project_id')) {
            $query->where('projects.id', $request->project_id);
        }

        // Add division filter if specified
        if ($request->filled('division_id')) {
            $query->where('projects.project_division_id', $request->division_id);
        }

        // Get all projects for statistics
        $allProjectsCollection = $query->orderBy('project_name')->get();

        // Implement manual pagination for the collection
        $perPage = 25;
        $currentPage = request()->get('page', 1);
        $total = $allProjectsCollection->count();
        $offset = ($currentPage - 1) * $perPage;
        $items = $allProjectsCollection->slice($offset, $perPage);

        // Create a LengthAwarePaginator instance
        $projects = new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $total,
            $perPage,
            $currentPage,
            [
                'path' => request()->url(),
                'pageName' => 'page',
            ]
        );

        // Append query parameters to pagination links
        $projects->appends(request()->query());

        // Get filter options
        $allProjects = Project::orderBy('project_name')->get();
        $projectStatuses = ['active', 'completed', 'on_hold', 'cancelled'];
        $divisions = ProjectDivision::where('status', 'active')->orderBy('division_name')->get();

        // Calculate statistics from all projects (not just paginated ones)
        $stats = [
            'total_projects' => $allProjectsCollection->count(),
            'total_stock' => $allProjectsCollection->sum('total_stock'),
            'total_unique_items' => $allProjectsCollection->sum('unique_items'),
            'total_value' => $allProjectsCollection->sum('total_value')
        ];

        return view('warehouse.reports.stock-on-project', compact('projects', 'allProjects', 'projectStatuses', 'divisions', 'request', 'stats'));
    }

    public function stockOnProjectDetails(Request $request, $projectId)
    {
        $project = Project::findOrFail($projectId);

        $query = Item::with('category')
            ->select('items.*')
            ->addSelect(DB::raw("(
                SELECT project_inventory.quantity_available
                FROM project_inventory
                WHERE project_inventory.project_id = {$projectId}
                AND project_inventory.item_id = items.id
            ) as project_stock"))
            ->addSelect(DB::raw("(
                SELECT COALESCE(project_inventory.unit_price, 0)
                FROM project_inventory
                WHERE project_inventory.project_id = {$projectId}
                AND project_inventory.item_id = items.id
            ) as effective_unit_price"))
            ->addSelect(DB::raw("(
                SELECT COALESCE(SUM(outgoing_fulfillments.quantity_fulfilled), 0)
                FROM outgoing_fulfillments
                INNER JOIN outgoing_transactions ON outgoing_fulfillments.outgoing_transaction_id = outgoing_transactions.id
                WHERE outgoing_transactions.project_id = {$projectId}
                AND outgoing_fulfillments.item_id = items.id
            ) as total_delivered"))
            ->whereExists(function ($query) use ($projectId) {
                $query->select(DB::raw(1))
                    ->from('project_inventory')
                    ->whereColumn('project_inventory.item_id', 'items.id')
                    ->where('project_inventory.project_id', $projectId)
                    ->where('project_inventory.quantity_available', '>', 0);
            });

        // Apply filters
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('item_description', 'like', "%{$search}%")
                  ->orWhere('item_code', 'like', "%{$search}%");
            });
        }

        if ($request->filled('stock_status')) {
            if ($request->stock_status === 'low_stock') {
                $query->whereRaw("(
                    SELECT project_inventory.quantity_available
                    FROM project_inventory
                    WHERE project_inventory.project_id = {$projectId}
                    AND project_inventory.item_id = items.id
                ) <= items.min_stock_level");
            } elseif ($request->stock_status === 'out_of_stock') {
                $query->whereRaw("(
                    SELECT COALESCE(project_inventory.quantity_available, 0)
                    FROM project_inventory
                    WHERE project_inventory.project_id = {$projectId}
                    AND project_inventory.item_id = items.id
                ) = 0");
            }
        }

        $items = $query->orderBy('item_description')->get();

        // Calculate totals
        $totalItems = $items->count();
        $totalStock = $items->sum('project_stock');
        $totalValue = $items->sum(function ($item) {
            return ($item->project_stock ?? 0) * ($item->effective_unit_price ?? 0);
        });

        // Get filter options
        $categories = ItemCategory::orderBy('category_name')->get();

        return view('warehouse.reports.stock-on-project-details', compact(
            'project', 'items', 'totalItems', 'totalStock', 'totalValue', 'categories', 'request'
        ));
    }

    private function determineFulfillmentStatus($outgoingItem)
    {
        if ($outgoingItem->quantity_released <= 0) {
            return 'pending';
        } elseif ($outgoingItem->quantity_balance > 0) {
            return 'partial';
        } else {
            return 'completed';
        }
    }

    /**
     * Show reference allocations page for a specific item in a project
     */
    public function itemReferenceAllocations($projectId, $itemId)
    {
        $project = Project::findOrFail($projectId);
        $item = Item::findOrFail($itemId);

        // Get Material Requests
        $materialRequests = OutgoingTransaction::with(['requester', 'division'])
            ->where('project_id', $projectId)
            ->whereHas('items', function($query) use ($itemId) {
                $query->where('item_id', $itemId)
                      ->where('quantity_released', '>', 0);
            })
            ->get()
            ->map(function($request) use ($itemId) {
                $outgoingItem = $request->items->where('item_id', $itemId)->first();

                // Calculate returned quantity for this material request
                $returnedQty = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($q) {
                    $q->where('operation_type', 'site_return')
                      ->where('status', 'completed');
                })->where('outgoing_item_id', $outgoingItem->id)
                  ->sum('quantity_delivered');

                // Calculate transferred quantity via Material Transfers
                $transferredQty = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($q) {
                    $q->whereIn('status', ['approved', 'partially_approved']);
                })->where('outgoing_item_id', $outgoingItem->id)
                  ->sum('quantity_approved');

                // Make reference number unique if it's a common value
                $referenceNumber = $request->material_request_number ?: 'MR-' . str_pad($request->id, 6, '0', STR_PAD_LEFT);
                if (in_array(strtoupper($referenceNumber), ['INV', 'INVENTORY', 'N/A'])) {
                    $referenceNumber = "MR#{$request->id}: {$referenceNumber}-{$request->id}";
                } else {
                    $referenceNumber = "MR#{$request->id}: {$referenceNumber}";
                }

                return [
                    'type' => 'Material Request',
                    'reference_number' => $referenceNumber,
                    'date' => $request->transaction_date->format('Y-m-d'),
                    'requested_by' => $request->requester ? $request->requester->name : 'N/A',
                    'division' => $request->division ? $request->division->division_name : 'N/A',
                    'status' => $outgoingItem->quantity_released == 0 ? 'Pending' :
                                ($outgoingItem->quantity_balance == 0 ? 'Fully Dispatched' : 'Partially Dispatched'),
                    'quantity_released' => $outgoingItem->quantity_released,
                    'quantity_returned' => $returnedQty,
                    'quantity_transferred' => $transferredQty,
                    'net_quantity' => $outgoingItem->quantity_released - $returnedQty - $transferredQty,
                    'unit_cost' => $outgoingItem->unit_cost ?? 0,
                    'total_value' => ($outgoingItem->quantity_released - $returnedQty - $transferredQty) * ($outgoingItem->unit_cost ?? 0)
                ];
            });

        // Get Direct Deliveries (including those from multi-source material issue)
        $directDeliveryIds = \App\Models\Warehouse\OutgoingFulfillment::join('outgoing_transactions', 'outgoing_fulfillments.outgoing_transaction_id', '=', 'outgoing_transactions.id')
            ->where('outgoing_fulfillments.fulfillment_type', 'direct_delivery')
            ->where('outgoing_fulfillments.item_id', $itemId)
            ->where('outgoing_transactions.project_id', $projectId)
            ->pluck('outgoing_fulfillments.reference_id');

        $directDeliveries = \App\Models\DirectDelivery::with(['supplier'])
            ->whereIn('id', $directDeliveryIds)
            ->whereHas('items', function($query) use ($itemId) {
                $query->where('item_id', $itemId);
            })
            ->get()
            ->map(function($delivery) use ($itemId, $projectId) {
                $deliveryItem = $delivery->items->where('item_id', $itemId)->first();

                // Get the actual issued quantity from fulfillment record
                $fulfilledQty = \App\Models\Warehouse\OutgoingFulfillment::join('outgoing_transactions', 'outgoing_fulfillments.outgoing_transaction_id', '=', 'outgoing_transactions.id')
                    ->where('outgoing_fulfillments.fulfillment_type', 'direct_delivery')
                    ->where('outgoing_fulfillments.reference_id', $delivery->id)
                    ->where('outgoing_fulfillments.item_id', $itemId)
                    ->where('outgoing_transactions.project_id', $projectId)
                    ->sum('outgoing_fulfillments.quantity_fulfilled');

                $deliveredQty = $fulfilledQty > 0 ? $fulfilledQty : ($deliveryItem->received_quantity > 0 ? $deliveryItem->received_quantity : $deliveryItem->delivered_quantity);

                // Calculate returned quantity for this direct delivery
                $returnedQty = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($q) {
                    $q->where('operation_type', 'site_return')
                      ->where('status', 'completed');
                })->where('direct_delivery_item_id', $deliveryItem->id)
                  ->sum('quantity_delivered');

                // Calculate transferred quantity via Material Transfers
                $transferredQty = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($q) {
                    $q->whereIn('status', ['approved', 'partially_approved']);
                })->where('direct_delivery_item_id', $deliveryItem->id)
                  ->sum('quantity_approved');

                // Make reference number unique if it's a common value
                $referenceNumber = $delivery->delivery_number ?? $delivery->reference_number ?? 'DD-' . str_pad($delivery->id, 6, '0', STR_PAD_LEFT);
                if (in_array(strtoupper($referenceNumber), ['DD', 'DELIVERY', 'N/A']) || strlen($referenceNumber) <= 3) {
                    $referenceNumber = "DD#{$delivery->id}: {$referenceNumber}-{$delivery->id}";
                } else {
                    $referenceNumber = "DD#{$delivery->id}: {$referenceNumber}";
                }

                return [
                    'type' => 'Direct Delivery',
                    'reference_number' => $referenceNumber,
                    'date' => $delivery->delivery_date->format('Y-m-d'),
                    'requested_by' => $delivery->supplier ? $delivery->supplier->name : 'Direct Supply',
                    'division' => 'N/A',
                    'status' => $deliveredQty > 0 ? 'Delivered' : 'Pending',
                    'quantity_released' => $deliveredQty,
                    'quantity_returned' => $returnedQty,
                    'quantity_transferred' => $transferredQty,
                    'net_quantity' => $deliveredQty - $returnedQty - $transferredQty,
                    'unit_cost' => $deliveryItem->unit_price ?? 0,
                    'total_value' => ($deliveredQty - $returnedQty - $transferredQty) * ($deliveryItem->unit_price ?? 0)
                ];
            });

        // Get Material Transfers (received by this project)
        $materialTransfers = \App\Models\Warehouse\MaterialTransferRequest::with(['transferrerProject', 'requestedBy'])
            ->where('receiver_project_id', $projectId)
            ->whereIn('status', ['approved', 'partially_approved'])
            ->whereHas('items', function($query) use ($itemId) {
                $query->where('item_id', $itemId)
                      ->where('quantity_approved', '>', 0);
            })
            ->get()
            ->map(function($transfer) use ($itemId) {
                $transferItem = $transfer->items->where('item_id', $itemId)->first();

                // Calculate returned quantity for this material transfer
                $returnedQty = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($q) {
                    $q->where('operation_type', 'site_return')
                      ->where('status', 'completed');
                })->where('material_transfer_item_id', $transferItem->id)
                  ->sum('quantity_delivered');

                // Calculate subsequent transfers from this material transfer
                $subsequentTransfers = \App\Models\Warehouse\MaterialTransferRequestItem::whereHas('transferRequest', function($q) {
                    $q->whereIn('status', ['approved', 'partially_approved']);
                })->where('material_transfer_item_id', $transferItem->id)
                  ->sum('quantity_approved');

                // Make reference number unique if it's a common value
                $referenceNumber = $transfer->transfer_request_number ?? 'MT-' . str_pad($transfer->id, 6, '0', STR_PAD_LEFT);
                if (in_array(strtoupper($referenceNumber), ['MT', 'TRANSFER', 'N/A']) || strlen($referenceNumber) <= 3) {
                    $referenceNumber = "MT#{$transfer->id}: {$referenceNumber}-{$transfer->id}";
                } else {
                    $referenceNumber = "MT#{$transfer->id}: {$referenceNumber}";
                }

                return [
                    'type' => 'Material Transfer',
                    'reference_number' => $referenceNumber,
                    'date' => $transfer->request_date->format('Y-m-d'),
                    'requested_by' => $transfer->requestedBy ? $transfer->requestedBy->name : 'N/A',
                    'division' => 'From: ' . ($transfer->transferrerProject ? $transfer->transferrerProject->project_name : 'N/A'),
                    'status' => $transferItem->quantity_approved > 0 ? 'Transferred' : 'Pending',
                    'quantity_released' => $transferItem->quantity_approved,
                    'quantity_returned' => $returnedQty,
                    'quantity_transferred' => $subsequentTransfers,
                    'net_quantity' => $transferItem->quantity_approved - $returnedQty - $subsequentTransfers,
                    'unit_cost' => $transferItem->estimated_unit_price ?? 0,
                    'total_value' => ($transferItem->quantity_approved - $returnedQty - $subsequentTransfers) * ($transferItem->estimated_unit_price ?? 0)
                ];
            });

        // Get Supplier Deliveries made directly to project site
        $supplierDeliveries = \App\Models\Warehouse\IncomingOperation::with(['supplier', 'items'])
            ->where('operation_type', 'supplier_delivery')
            ->where('delivery_type', 'project_site')
            ->where('project_id', $projectId)
            ->where('status', 'completed')
            ->whereHas('items', function($query) use ($itemId) {
                $query->where('item_id', $itemId)
                      ->where('quantity_delivered', '>', 0);
            })
            ->get()
            ->map(function($delivery) use ($itemId, $projectId) {
                $deliveryItem = $delivery->items->where('item_id', $itemId)->first();

                // Calculate returned quantity for this supplier delivery
                $returnedQty = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($q) {
                    $q->where('operation_type', 'site_return')
                      ->where('status', 'completed');
                })->where('supplier_delivery_item_id', $deliveryItem->id)
                  ->sum('quantity_delivered');

                // Calculate site transfers from this supplier delivery
                $transferredQty = DB::table('material_site_transfer_items as msti')
                    ->join('material_site_transfers as mst', 'msti.site_transfer_id', '=', 'mst.id')
                    ->where('mst.from_project_id', $projectId)
                    ->where('msti.item_id', $itemId)
                    ->where('mst.status', 'completed')
                    ->sum('msti.quantity_transferred');

                return [
                    'type' => 'Supplier Delivery (Direct)',
                    'reference_number' => 'SD#' . $delivery->id . ': ' . ($delivery->operation_number ?? 'N/A'),
                    'date' => $delivery->operation_date->format('Y-m-d'),
                    'requested_by' => $delivery->supplier->name ?? 'N/A',
                    'division' => 'Direct to Project',
                    'status' => 'Delivered',
                    'quantity_released' => $deliveryItem->quantity_delivered,
                    'quantity_returned' => $returnedQty,
                    'quantity_transferred' => $transferredQty,
                    'net_quantity' => $deliveryItem->quantity_delivered - $returnedQty - $transferredQty,
                    'unit_cost' => $deliveryItem->unit_price ?? 0,
                    'total_value' => ($deliveryItem->quantity_delivered - $returnedQty - $transferredQty) * ($deliveryItem->unit_price ?? 0)
                ];
            });

        // Get Site Transfers received by this project
        $siteTransfersReceived = \App\Models\Warehouse\MaterialSiteTransfer::with(['fromProject'])
            ->where('to_project_id', $projectId)
            ->where('status', 'completed')
            ->whereHas('items', function($query) use ($itemId) {
                $query->where('item_id', $itemId)
                      ->where('quantity_transferred', '>', 0);
            })
            ->get()
            ->map(function($transfer) use ($itemId, $projectId) {
                $transferItem = $transfer->items->where('item_id', $itemId)->first();

                // Calculate returned quantity for this site transfer
                $returnedQty = \App\Models\Warehouse\IncomingOperationItem::whereHas('operation', function($q) {
                    $q->where('operation_type', 'site_return')
                      ->where('status', 'completed');
                })->where('material_transfer_item_id', $transferItem->id)
                  ->sum('quantity_delivered');

                // Calculate subsequent transfers from this site transfer
                $subsequentTransfers = DB::table('material_site_transfer_items as msti')
                    ->join('material_site_transfers as mst', 'msti.site_transfer_id', '=', 'mst.id')
                    ->where('mst.from_project_id', $projectId)
                    ->where('msti.item_id', $itemId)
                    ->where('mst.status', 'completed')
                    ->sum('msti.quantity_transferred');

                return [
                    'type' => 'Site Transfer (Received)',
                    'reference_number' => 'ST#' . $transfer->id . ': From ' . ($transfer->fromProject->project_name ?? 'Unknown Project'),
                    'date' => $transfer->transfer_date->format('Y-m-d'),
                    'requested_by' => 'Inter-Project Transfer',
                    'division' => 'Site Transfer',
                    'status' => 'Received',
                    'quantity_released' => $transferItem->quantity_transferred,
                    'quantity_returned' => $returnedQty,
                    'quantity_transferred' => $subsequentTransfers,
                    'net_quantity' => $transferItem->quantity_transferred - $returnedQty - $subsequentTransfers,
                    'unit_cost' => $transferItem->unit_price ?? 0,
                    'total_value' => ($transferItem->quantity_transferred - $returnedQty - $subsequentTransfers) * ($transferItem->unit_price ?? 0)
                ];
            });

        // Combine all references
        $references = collect()
            ->merge($materialRequests)
            ->merge($directDeliveries)
            ->merge($materialTransfers)
            ->merge($supplierDeliveries)
            ->merge($siteTransfersReceived)
            ->sortByDesc(function ($item) {
                return $item['date'];
            })
            ->values();

        // Calculate totals
        $totals = [
            'total_released' => $references->sum('quantity_released'),
            'total_returned' => $references->sum('quantity_returned'),
            'total_transferred' => $references->sum('quantity_transferred'),
            'total_net' => $references->sum('net_quantity'),
            'total_value' => $references->sum('total_value')
        ];

        return view('warehouse.reports.item-reference-allocations', compact(
            'project', 'item', 'references', 'totals'
        ));
    }

    /**
     * Get division name based on return source type
     */
    private function getDivisionName($returnItem, $sourceInfo)
    {
        if ($returnItem->outgoing_item_id && $sourceInfo) {
            // Material Request return
            return $sourceInfo->transaction?->division?->division_name;
        } elseif ($returnItem->direct_delivery_item_id && $sourceInfo) {
            // Direct Delivery return
            return $sourceInfo->directDelivery?->project?->projectDivision?->division_name ?? 'Direct Delivery';
        } elseif ($returnItem->material_transfer_item_id && $sourceInfo) {
            // Material Transfer return
            return $sourceInfo->transferRequest?->receiverProject?->projectDivision?->division_name ?? 'Material Transfer';
        }

        // Fallback to operation project division if available
        return $returnItem->operation->project?->projectDivision?->division_name;
    }

    /**
     * Get reference number based on return source type
     */
    private function getReferenceNumber($returnItem, $sourceInfo)
    {
        if ($returnItem->outgoing_item_id && $sourceInfo) {
            // Material Request return
            return $sourceInfo->transaction?->material_request_number ?: 'MR-' . str_pad($sourceInfo->transaction?->id, 6, '0', STR_PAD_LEFT);
        } elseif ($returnItem->direct_delivery_item_id && $sourceInfo) {
            // Direct Delivery return
            return $sourceInfo->directDelivery?->delivery_number ?? 'DD-' . $sourceInfo->directDelivery?->id;
        } elseif ($returnItem->material_transfer_item_id && $sourceInfo) {
            // Material Transfer return
            return $sourceInfo->transferRequest?->transfer_request_number ?? 'MT-' . $sourceInfo->transferRequest?->id;
        }

        // Fallback to operation number
        return $returnItem->operation->operation_number ?? 'SR-' . $returnItem->operation->id;
    }

    /**
     * Get reference type based on return source
     */
    private function getReferenceType($returnItem)
    {
        if ($returnItem->outgoing_item_id) {
            return 'Material Request';
        } elseif ($returnItem->direct_delivery_item_id) {
            return 'Direct Delivery';
        } elseif ($returnItem->material_transfer_item_id) {
            return 'Material Transfer';
        }

        return 'Site Return';
    }

    /**
     * Display damaged items report
     */
    public function damagedItems(Request $request)
    {
        // Check if we should show detailed view for a specific item
        if ($request->filled('item_id') && $request->filled('details')) {
            return $this->damagedItemsDetails($request);
        }

        // Default: show item-wise summary
        return $this->damagedItemsSummary($request);
    }

    private function damagedItemsSummary(Request $request)
    {
        // Get item-wise summary of damaged items
        $query = \App\Models\DamagedItem::with(['item', 'item.category', 'item.unit'])
            ->select('item_id')
            ->selectRaw('COUNT(*) as record_count')
            ->selectRaw('SUM(quantity) as total_quantity')
            ->selectRaw('SUM(total_cost) as total_cost')
            ->selectRaw('SUM(recoverable_amount) as total_recoverable')
            ->selectRaw('SUM(total_cost - recoverable_amount) as loss_amount')
            ->selectRaw('MAX(created_at) as latest_incident')
            ->selectRaw('MIN(created_at) as first_incident')
            ->selectRaw('COUNT(CASE WHEN status = "reported" THEN 1 END) as pending_count')
            ->selectRaw('COUNT(CASE WHEN status = "approved" THEN 1 END) as approved_count')
            ->selectRaw('COUNT(CASE WHEN status = "settled" THEN 1 END) as settled_count')
            ->selectRaw('SUM(CASE WHEN status = "approved" THEN quantity ELSE 0 END) as approved_quantity')
            ->groupBy('item_id');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('item', function ($itemQuery) use ($search) {
                $itemQuery->where('item_code', 'like', "%{$search}%")
                         ->orWhere('item_description', 'like', "%{$search}%");
            });
        }

        if ($request->filled('category_id')) {
            $query->whereHas('item', function ($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Order by total cost desc
        $query->orderByRaw('total_cost DESC');

        // Paginate results
        $itemSummaries = $query->paginate(15)->withQueryString();

        // Add FOC available quantities and item details to each summary
        foreach ($itemSummaries as $summary) {
            if ($summary->item) {
                $summary->item_code = $summary->item->item_code;
                $summary->item_description = $summary->item->item_description;
                $summary->category_name = $summary->item->category->category_name ?? 'No Category';

                // Get unit name properly from the relationship
                $summary->unit = $summary->item->unit->unit_symbol ?? $summary->item->unit->unit_name ?? 'units';

                // FOC: quantity_available IS the available quantity
                $summary->foc_available = \App\Models\FocInventory::getTotalAvailableForItem($summary->item_id);

                // Get all FOC inventory for dispatched calculation
                $focInventory = \App\Models\FocInventory::where('item_id', $summary->item_id)->get();

                // Calculate FOC dispatched
                $focDispatched = 0;
                foreach ($focInventory as $foc) {
                    if ($foc->adjustments) {
                        $dispatchedQty = $foc->adjustments()
                            ->where('adjustment_type', 'dispatch')
                            ->sum('quantity_change'); // This will be negative
                        $focDispatched += abs($dispatchedQty);
                    }
                }
                $summary->foc_dispatched = $focDispatched;
            }
        }

        // Calculate overall statistics
        $totalStats = \App\Models\DamagedItem::selectRaw('
            COUNT(*) as total_incidents,
            COUNT(DISTINCT item_id) as unique_items,
            SUM(quantity) as total_quantity,
            SUM(total_cost) as total_cost,
            SUM(recoverable_amount) as total_recoverable,
            SUM(total_cost - recoverable_amount) as total_loss,
            COUNT(CASE WHEN status = "reported" THEN 1 END) as pending_count,
            COUNT(CASE WHEN status = "approved" THEN 1 END) as approved_count,
            SUM(CASE WHEN status = "approved" THEN quantity ELSE 0 END) as approved_quantity
        ')->first();

        // Get filter options
        $categories = \App\Models\Warehouse\ItemCategory::orderBy('category_name')->get();

        // Extract individual stats for compatibility with view
        $totalRecords = $totalStats->total_incidents ?? 0;
        $totalQuantity = $totalStats->total_quantity ?? 0;
        $totalCost = $totalStats->total_cost ?? 0;
        $totalLoss = $totalStats->total_loss ?? 0;
        $totalFocAvailable = \App\Models\FocInventory::sum('quantity_available') ?? 0;

        // Get projects for filter
        $projects = \App\Models\Warehouse\Project::orderBy('project_name')->get();

        return view('warehouse.reports.damaged-items-summary', compact(
            'itemSummaries',
            'totalRecords',
            'totalQuantity',
            'totalCost',
            'totalLoss',
            'totalFocAvailable',
            'categories',
            'projects'
        ));
    }

    public function damagedItemsDetails(Request $request, $itemId)
    {

        // Get the item info
        $item = \App\Models\Warehouse\Item::with(['category', 'unit'])->findOrFail($itemId);

        $query = \App\Models\DamagedItem::with(['item', 'item.category', 'item.unit', 'project', 'user'])
            ->where('item_id', $itemId);

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('source_reference', 'like', "%{$search}%")
                  ->orWhere('damage_description', 'like', "%{$search}%");
            });
        }

        if ($request->filled('source_type')) {
            $query->where('source_type', $request->source_type);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('severity')) {
            $query->where('damage_severity', $request->severity);
        }

        if ($request->filled('project_id')) {
            $query->where('project_id', $request->project_id);
        }

        if ($request->filled('cost_responsibility')) {
            $query->where('cost_responsibility', $request->cost_responsibility);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Order by created_at desc
        $query->orderBy('created_at', 'desc');

        // Paginate results
        $damagedItems = $query->paginate(25)->withQueryString();

        // Calculate statistics for this item
        $totalQuantity = $damagedItems->getCollection()->sum('quantity');
        $totalCost = $damagedItems->getCollection()->sum('total_cost');
        $totalLoss = $damagedItems->getCollection()->sum(function ($damagedItem) {
            return $damagedItem->total_cost - $damagedItem->recoverable_amount;
        });
        $pendingCount = $damagedItems->getCollection()->where('status', 'reported')->count();
        $approvedCount = $damagedItems->getCollection()->where('status', 'approved')->count();

        // Get comprehensive FOC inventory data for this item
        $focInventory = \App\Models\FocInventory::forItem($itemId)->get(); // Get all FOC records (not just available)
        $totalFocAvailable = $focInventory->sum('quantity_available');

        // Calculate total FOC received (including dispatched items)
        $totalFocReceived = 0;
        $totalFocDispatched = 0;
        $totalFocReturned = 0;

        foreach ($focInventory as $foc) {
            // For each FOC record, get the total quantity originally received
            $originalQuantity = $foc->quantity_available;

            // Add any dispatched amounts from adjustments
            if ($foc->adjustments) {
                $dispatchedQty = $foc->adjustments()
                    ->where('adjustment_type', 'dispatch')
                    ->sum('quantity_change'); // This will be negative
                $originalQuantity += abs($dispatchedQty);
                $totalFocDispatched += abs($dispatchedQty);
            }

            // Check if this FOC record is from a site return (FOC return)
            if ($foc->source_type === 'site_return' && strpos($foc->source_reference, 'SR-') === 0) {
                $totalFocReturned += $originalQuantity;
            }

            $totalFocReceived += $originalQuantity;
        }

        // Get filter options
        $projects = \App\Models\Warehouse\Project::orderBy('project_name')->get();
        $sourceTypes = [
            'inventory_conversion' => 'Inventory Conversion',
            'site_return' => 'Site Return'
        ];
        $statuses = [
            'reported' => 'Reported',
            'investigating' => 'Investigating',
            'approved' => 'Approved',
            'settled' => 'Settled',
            'written_off' => 'Written Off'
        ];
        $severities = [
            'minor' => 'Minor',
            'moderate' => 'Moderate',
            'major' => 'Major',
            'severe' => 'Severe'
        ];
        $responsibilities = [
            'company' => 'Company',
            'supplier' => 'Supplier',
            'project' => 'Project',
            'contractor' => 'Contractor',
            'external' => 'External Party',
            'unknown' => 'Unknown'
        ];

        return view('warehouse.reports.damaged-items-details', compact(
            'item',
            'damagedItems',
            'totalQuantity',
            'totalCost',
            'totalLoss',
            'pendingCount',
            'approvedCount',
            'focInventory',
            'totalFocAvailable',
            'totalFocReceived',
            'totalFocDispatched',
            'totalFocReturned',
            'projects',
            'sourceTypes',
            'statuses',
            'severities',
            'responsibilities'
        ));
    }

    public function showDamagedItemDetails(Request $request, $damagedItemId)
    {
        // Get the damaged item with all relationships
        $damagedItem = \App\Models\DamagedItem::with([
            'item', 'item.category', 'item.unit',
            'project', 'user'
        ])->findOrFail($damagedItemId);

        // Get related FOC inventory if any
        $focInventory = \App\Models\FocInventory::where('source_damaged_item_id', $damagedItemId)
            ->with(['adjustments'])
            ->get();

        // Return a partial view for modal content
        return view('warehouse.reports.partials.damaged-item-modal-content', compact(
            'damagedItem',
            'focInventory'
        ));
    }

    /**
     * Division Delivery Report - shows total deliveries vs current stock vs issued quantities by division
     */
    public function divisionDeliveries(Request $request)
    {
        $query = \App\Models\Warehouse\DivisionDelivery::with(['item', 'item.category', 'division']);

        // Apply filters
        if ($request->filled('division_id')) {
            $query->where('division_id', $request->division_id);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('item', function($q) use ($search) {
                $q->where('item_code', 'like', "%{$search}%")
                  ->orWhere('item_description', 'like', "%{$search}%");
            });
        }

        if ($request->filled('category_id')) {
            $query->whereHas('item', function($q) use ($request) {
                $q->where('category_id', $request->category_id);
            });
        }

        // Handle Export requests
        if ($request->has('export')) {
            return $this->exportDivisionDeliveries($request, $query);
        }

        // Handle AJAX requests for DataTable
        if ($request->expectsJson() || $request->has('draw')) {
            $start = $request->input('start', 0);
            $length = $request->input('length', 10);
            $searchValue = $request->input('search.value');

            // Apply search if provided
            if (!empty($searchValue)) {
                $query->whereHas('item', function($q) use ($searchValue) {
                    $q->where('item_code', 'like', "%{$searchValue}%")
                      ->orWhere('item_description', 'like', "%{$searchValue}%");
                });
            }

            // Get total count before pagination
            $totalRecords = $query->count();

            // Apply pagination and ordering
            $deliveries = $query->orderBy('total_quantity_delivered', 'desc')
                               ->skip($start)
                               ->take($length)
                               ->get();

            $data = [];
            foreach ($deliveries as $index => $delivery) {
                // Calculate returned quantity from site returns
                $totalReturned = 0;

                try {
                    // Calculate material request returns
                    $returnedQuantity = IncomingOperationItem::join('incoming_operations', 'incoming_operation_items.incoming_operation_id', '=', 'incoming_operations.id')
                        ->leftJoin('outgoing_items', 'incoming_operation_items.outgoing_item_id', '=', 'outgoing_items.id')
                        ->where('incoming_operations.operation_type', 'site_return')
                        ->where('incoming_operations.status', 'completed')
                        ->where('incoming_operation_items.item_id', $delivery->item_id)
                        ->where('outgoing_items.division_id', $delivery->division_id)
                        ->sum('incoming_operation_items.quantity_delivered');

                    // Calculate direct delivery returns
                    $directReturnedQuantity = IncomingOperationItem::join('incoming_operations', 'incoming_operation_items.incoming_operation_id', '=', 'incoming_operations.id')
                        ->leftJoin('direct_delivery_items', 'incoming_operation_items.direct_delivery_item_id', '=', 'direct_delivery_items.id')
                        ->leftJoin('direct_deliveries', 'direct_delivery_items.direct_delivery_id', '=', 'direct_deliveries.id')
                        ->where('incoming_operations.operation_type', 'site_return')
                        ->where('incoming_operations.status', 'completed')
                        ->where('incoming_operation_items.item_id', $delivery->item_id)
                        ->where('direct_deliveries.project_division_id', $delivery->division_id)
                        ->sum('incoming_operation_items.quantity_delivered');

                    $totalReturned = $returnedQuantity + $directReturnedQuantity;
                } catch (\Exception $e) {
                    \Log::error('Error calculating returned quantity', [
                        'delivery_id' => $delivery->id,
                        'error' => $e->getMessage()
                    ]);
                    $totalReturned = 0;
                }

                // Calculate availability percentage for styling
                $percentage = $delivery->total_quantity_delivered > 0
                    ? ($delivery->current_available_quantity / $delivery->total_quantity_delivered) * 100
                    : 0;

                if ($percentage >= 80) {
                    $class = 'text-success';
                    $icon = 'check_circle';
                } elseif ($percentage >= 50) {
                    $class = 'text-warning';
                    $icon = 'warning';
                } else {
                    $class = 'text-danger';
                    $icon = 'error';
                }

                $data[] = [
                    'DT_RowIndex' => $start + $index + 1,
                    'item_code' => '<code class="text-primary">' . $delivery->item->item_code . '</code>',
                    'item_description' => $delivery->item->item_description,
                    'division_name' => '<span class="badge badge-outline-primary">' . $delivery->division->division_name . '</span>',
                    'total_delivered' => '<strong class="' . ($delivery->total_quantity_delivered > 1000 ? 'text-success' : 'text-info') . '">' . number_format($delivery->total_quantity_delivered, 2) . '</strong>',
                    'current_available' => '<div class="d-flex align-items-center">
                                              <i class="material-icons text-xs mr-1 ' . $class . '">' . $icon . '</i>
                                              <span class="' . $class . '">' . number_format($delivery->current_available_quantity, 2) . '</span>
                                           </div>',
                    'total_issued' => $delivery->total_quantity_issued > 0
                        ? '<span class="text-primary">' . number_format($delivery->total_quantity_issued, 2) . '</span>'
                        : '<span class="text-muted">0.00</span>',
                    'total_returned' => $totalReturned > 0
                        ? '<span class="text-success">' . number_format($totalReturned, 2) . '</span>'
                        : '<span class="text-muted">0.00</span>',
                    'total_value' => '<span class="text-info">AED ' . number_format($delivery->total_value, 2) . '</span>',
                    'last_delivery' => $delivery->last_delivery_date
                        ? $delivery->last_delivery_date->format('d M, Y')
                        : '-',
                    'actions' => '<button type="button" class="btn btn-sm btn-outline-primary view-details-btn"
                                    data-division-id="' . $delivery->division_id . '"
                                    data-item-id="' . $delivery->item_id . '"
                                    data-item-code="' . htmlspecialchars($delivery->item->item_code) . '"
                                    data-item-description="' . htmlspecialchars($delivery->item->item_description) . '"
                                    data-division-name="' . htmlspecialchars($delivery->division->division_name) . '"
                                    title="View Project Details">
                                    <i class="material-icons">visibility</i>
                                  </button>'
                ];
            }

            return response()->json([
                'draw' => intval($request->input('draw')),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $data
            ]);
        }

        // Get data for the main view
        $deliveries = $query->paginate(15);

        // Calculate summary statistics
        $totalDeliveries = \App\Models\Warehouse\DivisionDelivery::query()
            ->when($request->filled('division_id'), function($q) use ($request) {
                $q->where('division_id', $request->division_id);
            })
            ->sum('total_quantity_delivered');

        $totalAvailable = \App\Models\Warehouse\DivisionDelivery::query()
            ->when($request->filled('division_id'), function($q) use ($request) {
                $q->where('division_id', $request->division_id);
            })
            ->sum('current_available_quantity');

        $totalIssued = \App\Models\Warehouse\DivisionDelivery::query()
            ->when($request->filled('division_id'), function($q) use ($request) {
                $q->where('division_id', $request->division_id);
            })
            ->sum('total_quantity_issued');

        $totalValue = \App\Models\Warehouse\DivisionDelivery::query()
            ->when($request->filled('division_id'), function($q) use ($request) {
                $q->where('division_id', $request->division_id);
            })
            ->sum('total_value');

        $totalItems = \App\Models\Warehouse\DivisionDelivery::query()
            ->when($request->filled('division_id'), function($q) use ($request) {
                $q->where('division_id', $request->division_id);
            })
            ->count();

        $utilizationRate = $totalDeliveries > 0 ? ($totalIssued / $totalDeliveries) * 100 : 0;

        // Get filter options
        $divisions = \App\Models\Warehouse\ProjectDivision::where('status', 'active')->orderBy('division_name')->get();
        $categories = ItemCategory::orderBy('category_name')->get();

        return view('warehouse.reports.division-deliveries', compact(
            'deliveries',
            'totalDeliveries',
            'totalAvailable',
            'totalIssued',
            'totalValue',
            'totalItems',
            'utilizationRate',
            'divisions',
            'categories'
        ));
    }

    /**
     * Get project-wise details for a specific division and item
     */
    public function getDivisionItemDetails(Request $request)
    {
        $divisionId = $request->division_id;
        $itemId = $request->item_id;

        if (!$divisionId || !$itemId) {
            return response()->json(['error' => 'Division ID and Item ID are required'], 400);
        }

        try {
            // Get projects where this item from this division has been issued
            $projectDetails = [];

            // 1. Material Request Issues
            $materialRequestIssues = OutgoingItem::with(['transaction.project', 'transaction.requester'])
                ->where('division_id', $divisionId)
                ->where('item_id', $itemId)
                ->whereHas('transaction', function($q) {
                    $q->where('status', 'successful');
                })
                ->get();

            foreach ($materialRequestIssues as $issue) {
                $projectId = $issue->transaction->project_id;
                if (!isset($projectDetails[$projectId])) {
                    $projectDetails[$projectId] = [
                        'project_id' => $projectId,
                        'project_name' => $issue->transaction->project->project_name ?? 'Unknown Project',
                        'project_number' => $issue->transaction->project->project_number ?? 'N/A',
                        'total_issued' => 0,
                        'total_returned' => 0,
                        'current_balance' => 0,
                        'issues' => []
                    ];
                }

                $projectDetails[$projectId]['total_issued'] += $issue->quantity_released;
                $projectDetails[$projectId]['issues'][] = [
                    'type' => 'Material Request',
                    'reference' => $issue->transaction->material_request_number,
                    'quantity' => $issue->quantity_released,
                    'date' => $issue->transaction->transaction_date->format('M d, Y'),
                    'requester' => $issue->transaction->requester->name ?? 'Unknown'
                ];
            }

            // 2. Direct Delivery Issues
            $directDeliveries = \App\Models\DirectDeliveryItem::with(['directDelivery.project'])
                ->whereHas('directDelivery', function($q) use ($divisionId) {
                    $q->where('project_division_id', $divisionId)->whereIn('status', ['delivered', 'completed']);
                })
                ->where('item_id', $itemId)
                ->get();

            foreach ($directDeliveries as $delivery) {
                $projectId = $delivery->directDelivery->project_id;
                if (!isset($projectDetails[$projectId])) {
                    $projectDetails[$projectId] = [
                        'project_id' => $projectId,
                        'project_name' => $delivery->directDelivery->project->project_name ?? 'Unknown Project',
                        'project_number' => $delivery->directDelivery->project->project_number ?? 'N/A',
                        'total_issued' => 0,
                        'total_returned' => 0,
                        'current_balance' => 0,
                        'issues' => []
                    ];
                }

                $deliveredQty = $delivery->delivered_quantity ?? $delivery->ordered_quantity;
                $projectDetails[$projectId]['total_issued'] += $deliveredQty;
                $projectDetails[$projectId]['issues'][] = [
                    'type' => 'Direct Delivery',
                    'reference' => $delivery->directDelivery->delivery_number,
                    'quantity' => $deliveredQty,
                    'date' => $delivery->directDelivery->delivery_date->format('M d, Y'),
                    'supplier' => $delivery->directDelivery->supplier->name ?? 'Unknown'
                ];
            }

            // 3. Calculate returned quantities for each project
            foreach ($projectDetails as $projectId => &$details) {
                // Material request returns
                $materialReturned = IncomingOperationItem::join('incoming_operations', 'incoming_operation_items.incoming_operation_id', '=', 'incoming_operations.id')
                    ->leftJoin('outgoing_items', 'incoming_operation_items.outgoing_item_id', '=', 'outgoing_items.id')
                    ->where('incoming_operations.operation_type', 'site_return')
                    ->where('incoming_operations.status', 'completed')
                    ->where('incoming_operations.project_id', $projectId)
                    ->where('incoming_operation_items.item_id', $itemId)
                    ->where('outgoing_items.division_id', $divisionId)
                    ->sum('incoming_operation_items.quantity_delivered');

                // Direct delivery returns
                $directReturned = IncomingOperationItem::join('incoming_operations', 'incoming_operation_items.incoming_operation_id', '=', 'incoming_operations.id')
                    ->leftJoin('direct_delivery_items', 'incoming_operation_items.direct_delivery_item_id', '=', 'direct_delivery_items.id')
                    ->leftJoin('direct_deliveries', 'direct_delivery_items.direct_delivery_id', '=', 'direct_deliveries.id')
                    ->where('incoming_operations.operation_type', 'site_return')
                    ->where('incoming_operations.status', 'completed')
                    ->where('incoming_operations.project_id', $projectId)
                    ->where('incoming_operation_items.item_id', $itemId)
                    ->where('direct_deliveries.project_division_id', $divisionId)
                    ->sum('incoming_operation_items.quantity_delivered');

                $details['total_returned'] = $materialReturned + $directReturned;
                $details['current_balance'] = $details['total_issued'] - $details['total_returned'];
            }

            // Get division and item info
            $division = \App\Models\Warehouse\ProjectDivision::find($divisionId);
            $item = \App\Models\Warehouse\Item::find($itemId);

            return response()->json([
                'success' => true,
                'data' => [
                    'division' => [
                        'id' => $division->id,
                        'name' => $division->division_name,
                        'code' => $division->division_code
                    ],
                    'item' => [
                        'id' => $item->id,
                        'code' => $item->item_code,
                        'description' => $item->item_description,
                        'unit' => $item->unit_of_measure
                    ],
                    'projects' => array_values($projectDetails)
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Error fetching division item details', [
                'division_id' => $divisionId,
                'item_id' => $itemId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Unable to fetch project details'
            ], 500);
        }
    }

    /**
     * Export Division Deliveries Report
     */
    private function exportDivisionDeliveries(Request $request, $query)
    {
        $deliveries = $query->orderBy('total_quantity_delivered', 'desc')->get();
        $exportType = $request->input('export');

        if ($exportType === 'excel') {
            return $this->exportDivisionDeliveriesToExcel($deliveries, $request);
        } elseif ($exportType === 'pdf') {
            return $this->exportDivisionDeliveriesToPdf($deliveries, $request);
        }

        return response()->json(['error' => 'Invalid export type'], 400);
    }

    /**
     * Export Division Deliveries to Excel
     */
    private function exportDivisionDeliveriesToExcel($deliveries, Request $request)
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Division Deliveries Report');

        // Set headers
        $headers = [
            'A' => 'Item Code',
            'B' => 'Description',
            'C' => 'Division',
            'D' => 'Total Delivered',
            'E' => 'Available',
            'F' => 'Total Issued',
            'G' => 'Utilization %',
            'H' => 'Total Value (AED)',
            'I' => 'Last Delivery Date'
        ];

        $row = 1;
        foreach ($headers as $col => $header) {
            $sheet->setCellValue($col . $row, $header);
            $sheet->getStyle($col . $row)->getFont()->setBold(true);
        }

        // Add data
        $row = 2;
        foreach ($deliveries as $delivery) {
            $utilizationRate = $delivery->total_quantity_delivered > 0
                ? ($delivery->total_quantity_issued / $delivery->total_quantity_delivered) * 100
                : 0;

            $sheet->setCellValue('A' . $row, $delivery->item->item_code);
            $sheet->setCellValue('B' . $row, $delivery->item->item_description);
            $sheet->setCellValue('C' . $row, $delivery->division->division_name);
            $sheet->setCellValue('D' . $row, number_format($delivery->total_quantity_delivered, 2));
            $sheet->setCellValue('E' . $row, number_format($delivery->current_available_quantity, 2));
            $sheet->setCellValue('F' . $row, number_format($delivery->total_quantity_issued, 2));
            $sheet->setCellValue('G' . $row, number_format($utilizationRate, 1) . '%');
            $sheet->setCellValue('H' . $row, number_format($delivery->total_value, 2));
            $sheet->setCellValue('I' . $row, $delivery->last_delivery_date ? $delivery->last_delivery_date->format('d M, Y') : '-');

            $row++;
        }

        // Auto-size columns
        foreach (range('A', 'I') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        $filename = 'division-deliveries-report-' . now()->format('Y-m-d-H-i-s') . '.xlsx';
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);

        return response()->stream(function() use ($writer) {
            $writer->save('php://output');
        }, 200, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
            'Cache-Control' => 'max-age=0',
        ]);
    }

    /**
     * Export Division Deliveries to PDF
     */
    private function exportDivisionDeliveriesToPdf($deliveries, Request $request)
    {
        $data = [
            'deliveries' => $deliveries,
            'filters' => [
                'division_id' => $request->input('division_id'),
                'category_id' => $request->input('category_id'),
                'search' => $request->input('search'),
            ],
            'generated_at' => now()->format('d M, Y H:i:s')
        ];

        $pdf = app('dompdf.wrapper');
        $pdf->loadView('warehouse.reports.pdf.division-deliveries', $data);
        $pdf->setPaper('a4', 'landscape');

        $filename = 'division-deliveries-report-' . now()->format('Y-m-d-H-i-s') . '.pdf';
        return $pdf->download($filename);
    }

}