<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Casts\Attribute;

class SiteSetting extends Model
{
    protected $fillable = [
        'key',
        'value', 
        'type',
        'group',
        'label',
        'description',
        'is_active',
        'sort_order'
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer'
    ];

    public static function getValue($key, $default = null)
    {
        $setting = static::where('key', $key)
            ->where('is_active', true)
            ->first();
            
        if ($setting) {
            return $setting->value;
        }
        
        // If setting not found, try to get from defaults
        $defaultValue = static::getDefaultValue($key);
        if ($defaultValue !== null) {
            return $defaultValue;
        }
        
        return $default;
    }
    
    public static function getDefaultValue($key)
    {
        $defaults = static::getDefaultSettings();
        return $defaults[$key] ?? null;
    }
    
    public static function ensureDefaults()
    {
        $defaults = static::getDefaultSettings();
        $fileSettings = ['site_logo', 'site_favicon', 'admin_favicon']; // Settings that should be file type
        
        foreach ($defaults as $key => $value) {
            static::firstOrCreate(
                ['key' => $key],
                [
                    'value' => $value,
                    'type' => in_array($key, $fileSettings) ? 'file' : 'text',
                    'group' => static::getSettingGroup($key),
                    'label' => static::getSettingLabel($key),
                    'description' => static::getSettingDescription($key),
                    'is_active' => true,
                    'sort_order' => 0
                ]
            );
        }
    }
    
    private static function getSettingGroup($key)
    {
        $groups = [
            'site_name' => 'general',
            'site_description' => 'general',
            'site_logo' => 'general',
            'site_favicon' => 'general',
            'admin_favicon' => 'admin',
            'contact_email' => 'contact',
            'contact_phone' => 'contact',
            'contact_address' => 'contact',
            'whatsapp_number' => 'whatsapp',
            'whatsapp_enabled' => 'whatsapp',
            'footer_text' => 'footer',
        ];
        
        return $groups[$key] ?? 'general';
    }
    
    private static function getSettingLabel($key)
    {
        $labels = [
            'site_logo' => 'Site Logo',
            'site_favicon' => 'Site Favicon',
            'admin_favicon' => 'Admin Panel Favicon',
            'whatsapp_number' => 'WhatsApp Number',
            'whatsapp_enabled' => 'Enable WhatsApp',
            'footer_text' => 'Footer Copyright Text',
        ];
        
        return $labels[$key] ?? ucwords(str_replace('_', ' ', $key));
    }
    
    private static function getSettingDescription($key)
    {
        $descriptions = [
            'site_logo' => 'Upload your site logo (PNG, JPG, SVG)',
            'site_favicon' => 'Upload your site favicon (ICO, PNG)',
            'admin_favicon' => 'Upload admin panel favicon (ICO, PNG) - separate from site favicon',
            'whatsapp_number' => 'WhatsApp contact number (with country code)',
            'whatsapp_enabled' => 'Enable/disable WhatsApp contact widget',
        ];
        
        return $descriptions[$key] ?? 'Default setting for ' . $key;
    }
    
    private static function getDefaultSettings()
    {
        return [
            'site_name' => 'UFG - United Float Glass',
            'site_description' => 'Leading manufacturer of high-quality float glass products',
            'site_logo' => 'assets/images/aufg-logo.png',
            'site_favicon' => 'images/favicon.ico',
            'admin_favicon' => 'images/admin-favicon.ico',
            'contact_email' => 'info@ufg.com',
            'contact_phone' => '+1 (555) 123-4567',
            'contact_address' => '123 Glass Street, Industrial District, City, Country',
            'whatsapp_number' => '+1234567890',
            'whatsapp_enabled' => '1',
            'footer_text' => '© 2025 UFG - United Float Glass. All rights reserved.',
        ];
    }

    public static function getByGroup($group)
    {
        return static::where('group', $group)
            ->where('is_active', true)
            ->orderBy('sort_order')
            ->get();
    }

    protected function value(): Attribute
    {
        return Attribute::make(
            get: function ($value) {
                return match($this->type) {
                    'json' => json_decode($value, true),
                    'boolean' => (bool) $value,
                    'integer' => (int) $value,
                    default => $value
                };
            },
            set: function ($value) {
                return match($this->type) {
                    'json' => json_encode($value),
                    'boolean' => $value ? '1' : '0',
                    default => (string) $value
                };
            }
        );
    }
}
