<?php

namespace App\Models\Warehouse;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class DivisionDelivery extends Model
{
    use HasFactory;

    protected $fillable = [
        'division_id',
        'item_id',
        'total_quantity_delivered',
        'current_available_quantity',
        'total_quantity_issued',
        'average_unit_price',
        'total_value',
        'first_delivery_date',
        'last_delivery_date',
        'last_updated'
    ];

    protected $casts = [
        'total_quantity_delivered' => 'decimal:2',
        'current_available_quantity' => 'decimal:2',
        'total_quantity_issued' => 'decimal:2',
        'average_unit_price' => 'decimal:2',
        'total_value' => 'decimal:2',
        'first_delivery_date' => 'date',
        'last_delivery_date' => 'date',
        'last_updated' => 'datetime'
    ];

    /**
     * Get the division that this delivery belongs to
     */
    public function division(): BelongsTo
    {
        return $this->belongsTo(ProjectDivision::class, 'division_id');
    }

    /**
     * Get the item for this delivery record
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * Update delivery totals when new stock arrives
     */
    public static function recordDelivery($divisionId, $itemId, $quantity, $unitPrice, $deliveryDate = null)
    {
        $deliveryDate = $deliveryDate ?: now()->format('Y-m-d');

        $record = self::firstOrNew([
            'division_id' => $divisionId,
            'item_id' => $itemId
        ]);

        if ($record->exists) {
            // Update existing record
            $totalDeliveredBefore = $record->total_quantity_delivered;
            $totalValueBefore = $record->total_value;

            $record->total_quantity_delivered += $quantity;
            $record->current_available_quantity += $quantity;
            $record->total_value += ($quantity * $unitPrice);
            $record->average_unit_price = $record->total_value / $record->total_quantity_delivered;
            $record->last_delivery_date = $deliveryDate;
        } else {
            // Create new record
            $record->total_quantity_delivered = $quantity;
            $record->current_available_quantity = $quantity;
            $record->total_quantity_issued = 0;
            $record->average_unit_price = $unitPrice;
            $record->total_value = $quantity * $unitPrice;
            $record->first_delivery_date = $deliveryDate;
            $record->last_delivery_date = $deliveryDate;
        }

        $record->last_updated = now();
        $record->save();

        return $record;
    }

    /**
     * Update quantities when items are issued
     */
    public static function recordIssue($divisionId, $itemId, $issuedQuantity)
    {
        $record = self::where('division_id', $divisionId)
            ->where('item_id', $itemId)
            ->first();

        if ($record) {
            $record->current_available_quantity = max(0, $record->current_available_quantity - $issuedQuantity);
            $record->total_quantity_issued += $issuedQuantity;
            $record->last_updated = now();
            $record->save();
        }

        return $record;
    }

    /**
     * Get delivery summary for a division
     */
    public static function getDivisionSummary($divisionId)
    {
        return self::where('division_id', $divisionId)
            ->with('item')
            ->orderBy('total_quantity_delivered', 'desc')
            ->get();
    }

    /**
     * Get delivery summary for an item across all divisions
     */
    public static function getItemSummary($itemId)
    {
        return self::where('item_id', $itemId)
            ->with('division')
            ->orderBy('total_quantity_delivered', 'desc')
            ->get();
    }
}