<?php

namespace App\Models\Warehouse;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class MaterialDeliveryNote extends Model
{
    use HasFactory;

    protected $fillable = [
        'mdn_number',
        'outgoing_transaction_id',
        'generated_date',
        'delivered_to',
        'vehicle_number',
        'driver_name',
        'driver_contact',
        'notes'
    ];

    protected $casts = [
        'generated_date' => 'date',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($deliveryNote) {
            if (!$deliveryNote->mdn_number) {
                $deliveryNote->mdn_number = 'MDN-' . date('Ymd') . '-' . str_pad(static::count() + 1, 4, '0', STR_PAD_LEFT);
            }

            if (!$deliveryNote->generated_date) {
                $deliveryNote->generated_date = now()->toDateString();
            }
        });
    }

    /**
     * Get the outgoing transaction that owns the delivery note
     */
    public function outgoingTransaction(): BelongsTo
    {
        return $this->belongsTo(OutgoingTransaction::class);
    }

    /**
     * Get the project from the outgoing transaction
     */
    public function getProjectAttribute()
    {
        return $this->outgoingTransaction->project;
    }

    /**
     * Get the division from the outgoing transaction
     */
    public function getDivisionAttribute()
    {
        return $this->outgoingTransaction->division;
    }

    /**
     * Get the items from the outgoing transaction
     */
    public function getItemsAttribute()
    {
        return $this->outgoingTransaction->items;
    }

    /**
     * Get the requester from the outgoing transaction
     */
    public function getRequesterAttribute()
    {
        return $this->outgoingTransaction->requester;
    }

    /**
     * Get the receiver name from the outgoing transaction
     */
    public function getReceiverNameAttribute()
    {
        return $this->outgoingTransaction->receiver_name;
    }

    /**
     * Get total items count
     */
    public function getTotalItemsCountAttribute()
    {
        return $this->outgoingTransaction->items()->count();
    }

    /**
     * Get total quantity delivered
     */
    public function getTotalQuantityDeliveredAttribute()
    {
        return $this->outgoingTransaction->items()->sum('quantity_released');
    }

    /**
     * Get total value of delivery
     */
    public function getTotalDeliveryValueAttribute()
    {
        return $this->outgoingTransaction->items()->sum('total_price');
    }

    /**
     * Check if delivery is completed
     */
    public function getIsCompletedAttribute()
    {
        return $this->outgoingTransaction->status === 'completed';
    }

    /**
     * Get delivery status badge color
     */
    public function getStatusBadgeColorAttribute()
    {
        return match ($this->outgoingTransaction->status) {
            'pending' => 'warning',
            'approved' => 'info',
            'completed' => 'success',
            'cancelled' => 'danger',
            default => 'secondary'
        };
    }

    /**
     * Get formatted delivery address
     */
    public function getFormattedDeliveryAddressAttribute()
    {
        $address = [];

        if ($this->delivered_to) {
            $address[] = $this->delivered_to;
        }

        if ($this->project) {
            $address[] = $this->project->project_name;
            if ($this->project->location) {
                $address[] = $this->project->location;
            }
        }

        return implode(', ', $address);
    }

    /**
     * Generate delivery note PDF
     */
    public function generatePDF()
    {
        // This would contain the logic to generate PDF
        // Can be implemented using packages like barryvdh/laravel-dompdf
        return "PDF generation logic would go here";
    }

    /**
     * Send delivery note via email
     */
    public function sendEmail($to)
    {
        // This would contain the logic to send email
        // Can be implemented using Laravel Mail
        return "Email sending logic would go here";
    }

    /**
     * Scope for delivery notes by date range
     */
    public function scopeDateBetween($query, $startDate, $endDate)
    {
        return $query->whereBetween('generated_date', [$startDate, $endDate]);
    }

    /**
     * Scope for delivery notes by project
     */
    public function scopeForProject($query, $projectId)
    {
        return $query->whereHas('outgoingTransaction', function ($q) use ($projectId) {
            $q->where('project_id', $projectId);
        });
    }

    /**
     * Scope for delivery notes by division
     */
    public function scopeForDivision($query, $divisionId)
    {
        return $query->whereHas('outgoingTransaction', function ($q) use ($divisionId) {
            $q->where('division_id', $divisionId);
        });
    }
}