<?php

namespace App\Models\Warehouse;

use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;

class MaterialTransferRequest extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'transfer_request_number',
        'request_date',
        'transferrer_project_id',
        'receiver_project_id',
        'transfer_reason',
        'transfer_notes',
        'priority',
        'required_date',
        'status',
        'requested_by',
        'transferrer_approved_by',
        'transferrer_approved_at',
        'transferrer_approval_notes',
        'receiver_approved_by',
        'receiver_approved_at',
        'receiver_approval_notes',
        'rejected_by',
        'rejected_at',
        'rejection_reason',
        'total_estimated_value',
        'total_items_count'
        // Removed: 'delivery_type' and 'direct_delivery_id' to simplify material requests
        // Focus on item selection management only
    ];

    protected $casts = [
        'request_date' => 'date',
        'required_date' => 'date',
        'transferrer_approved_at' => 'datetime',
        'receiver_approved_at' => 'datetime',
        'rejected_at' => 'datetime',
        'total_estimated_value' => 'decimal:2',
        'total_items_count' => 'integer'
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($request) {
            if (empty($request->transfer_request_number)) {
                $request->transfer_request_number = static::generateRequestNumber();
            }
        });

        static::saving(function ($request) {
            // Recalculate totals from items
            if ($request->exists && $request->items()->exists()) {
                $request->total_estimated_value = $request->items()->sum('estimated_total_value');
                $request->total_items_count = $request->items()->count();
            }
        });
    }

    /**
     * Generate unique transfer request number
     */
    public static function generateRequestNumber()
    {
        $year = now()->format('Y');
        $sequence = static::whereYear('created_at', $year)->count() + 1;
        return 'MTR-' . $year . '-' . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Relationships
     */
    public function transferrerProject(): BelongsTo
    {
        return $this->belongsTo(Project::class, 'transferrer_project_id');
    }

    public function receiverProject(): BelongsTo
    {
        return $this->belongsTo(Project::class, 'receiver_project_id');
    }

    public function requestedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'requested_by');
    }

    public function transferrerApprovedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'transferrer_approved_by');
    }

    public function receiverApprovedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'receiver_approved_by');
    }

    public function rejectedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'rejected_by');
    }

    public function items(): HasMany
    {
        return $this->hasMany(MaterialTransferRequestItem::class, 'transfer_request_id');
    }

    public function transferIssue(): HasOne
    {
        return $this->hasOne(MaterialTransferIssue::class, 'transfer_request_id');
    }

    /**
     * Scopes
     */
    public function scopePending($query)
    {
        return $query->whereIn('status', ['draft', 'pending_transferrer_approval']);
    }

    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    public function scopeForTransferrerProject($query, $projectId)
    {
        return $query->where('transferrer_project_id', $projectId);
    }

    public function scopeForReceiverProject($query, $projectId)
    {
        return $query->where('receiver_project_id', $projectId);
    }

    /**
     * Status Management
     */
    public function submitForApproval()
    {
        $this->status = 'pending_transferrer_approval';
        $this->save();
    }

    public function approveByTransferrer($userId, $notes = null)
    {
        $this->transferrer_approved_by = $userId;
        $this->transferrer_approved_at = now();
        $this->transferrer_approval_notes = $notes;
        $this->status = 'approved';
        $this->save();
    }

    public function partiallyApproveByTransferrer($userId, $notes = null)
    {
        $this->transferrer_approved_by = $userId;
        $this->transferrer_approved_at = now();
        $this->transferrer_approval_notes = $notes;
        $this->status = 'partially_approved';
        $this->save();
    }


    public function reject($userId, $reason)
    {
        $this->rejected_by = $userId;
        $this->rejected_at = now();
        $this->rejection_reason = $reason;
        $this->status = 'rejected';
        $this->save();
    }

    /**
     * Status Checks
     */
    public function isPending()
    {
        return in_array($this->status, ['draft', 'pending_transferrer_approval']);
    }

    public function isApproved()
    {
        return $this->status === 'approved';
    }

    public function isRejected()
    {
        return $this->status === 'rejected';
    }

    public function canBeApprovedByTransferrer()
    {
        return in_array($this->status, ['pending_transferrer_approval', 'partially_approved']);
    }


    /**
     * Check if transfer can be issued
     */
    public function canCreateTransferIssue()
    {
        return $this->isApproved() && !$this->transferIssue;
    }

    /**
     * Get priority badge color
     */
    public function getPriorityColorAttribute()
    {
        return match($this->priority) {
            'urgent' => 'danger',
            'high' => 'warning',
            'medium' => 'info',
            'low' => 'secondary',
            default => 'secondary'
        };
    }

    /**
     * Get status badge color
     */
    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'draft' => 'secondary',
            'pending_transferrer_approval' => 'warning',
            'partially_approved' => 'info',
            'approved' => 'success',
            'rejected' => 'danger',
            'cancelled' => 'dark',
            default => 'secondary'
        };
    }
}