<?php

namespace App\Models\Warehouse;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ReturnItem extends Model
{
    use HasFactory;

    protected $fillable = [
        'return_transaction_id',
        'item_id',
        'quantity_returned',
        'condition',
        'unit_price',
        'total_value',
        'action_taken'
    ];

    protected $casts = [
        'quantity_returned' => 'decimal:2',
        'unit_price' => 'decimal:2',
        'total_value' => 'decimal:2',
        'condition' => 'string',
        'action_taken' => 'string',
        'created_at' => 'datetime',
        'updated_at' => 'datetime'
    ];

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($item) {
            // Auto-calculate total value
            $item->total_value = $item->quantity_returned * $item->unit_price;

            // Auto-determine action based on condition if not set
            if (!$item->action_taken) {
                $item->action_taken = match ($item->condition) {
                    'good' => 'restocked',
                    'damaged' => 'discarded',
                    'expired' => 'discarded',
                    default => 'restocked'
                };
            }
        });
    }

    /**
     * Get the transaction that owns the item
     */
    public function transaction(): BelongsTo
    {
        return $this->belongsTo(ReturnTransaction::class, 'return_transaction_id');
    }

    /**
     * Get the item details
     */
    public function item(): BelongsTo
    {
        return $this->belongsTo(Item::class);
    }

    /**
     * Check if item can be restocked
     */
    public function getCanBeRestockedAttribute()
    {
        return $this->condition === 'good';
    }

    /**
     * Check if item should be discarded
     */
    public function getShouldBeDiscardedAttribute()
    {
        return in_array($this->condition, ['damaged', 'expired']);
    }

    /**
     * Get condition badge color for UI
     */
    public function getConditionBadgeColorAttribute()
    {
        return match ($this->condition) {
            'good' => 'success',
            'damaged' => 'warning',
            'expired' => 'danger',
            default => 'secondary'
        };
    }

    /**
     * Get action badge color for UI
     */
    public function getActionBadgeColorAttribute()
    {
        return match ($this->action_taken) {
            'restocked' => 'success',
            'discarded' => 'danger',
            'returned_to_supplier' => 'info',
            default => 'secondary'
        };
    }

    /**
     * Scope for items in good condition
     */
    public function scopeGoodCondition($query)
    {
        return $query->where('condition', 'good');
    }

    /**
     * Scope for damaged items
     */
    public function scopeDamaged($query)
    {
        return $query->where('condition', 'damaged');
    }

    /**
     * Scope for expired items
     */
    public function scopeExpired($query)
    {
        return $query->where('condition', 'expired');
    }

    /**
     * Scope for restocked items
     */
    public function scopeRestocked($query)
    {
        return $query->where('action_taken', 'restocked');
    }
}