<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Gate;
use App\Services\Auth\RoleBasedAccessService;
use App\Policies\Warehouse\ItemPolicy;
use App\Policies\Warehouse\InventoryPolicy;
use App\Policies\Warehouse\TransactionPolicy;
use App\Models\Warehouse\Item;
use App\Models\Warehouse\Inventory;
use App\Models\Warehouse\IncomingTransaction;
use App\Models\Warehouse\OutgoingTransaction;
use App\Models\Warehouse\ReturnTransaction;

class AppServiceProvider extends ServiceProvider
{
    /**
     * The policy mappings for the application.
     *
     * @var array<class-string, class-string>
     */
    protected $policies = [
        Item::class => ItemPolicy::class,
        Inventory::class => InventoryPolicy::class,
        IncomingTransaction::class => TransactionPolicy::class,
        OutgoingTransaction::class => TransactionPolicy::class,
        ReturnTransaction::class => TransactionPolicy::class,
    ];

    /**
     * Register any application services.
     */
    public function register(): void
    {
        // Register the RoleBasedAccessService as singleton
        $this->app->singleton(RoleBasedAccessService::class, function ($app) {
            return new RoleBasedAccessService();
        });

        // Register repository bindings
        $this->app->bind(
            \App\Repositories\Warehouse\ItemRepository::class,
            function ($app) {
                return new \App\Repositories\Warehouse\ItemRepository(
                    new \App\Models\Warehouse\Item()
                );
            }
        );

        $this->app->bind(
            \App\Repositories\Warehouse\InventoryRepository::class,
            function ($app) {
                return new \App\Repositories\Warehouse\InventoryRepository(
                    new \App\Models\Warehouse\Inventory()
                );
            }
        );

        $this->app->bind(
            \App\Repositories\Warehouse\ItemCategoryRepository::class,
            function ($app) {
                return new \App\Repositories\Warehouse\ItemCategoryRepository(
                    new \App\Models\Warehouse\ItemCategory()
                );
            }
        );

        $this->app->bind(
            \App\Repositories\Warehouse\StockMovementRepository::class,
            function ($app) {
                return new \App\Repositories\Warehouse\StockMovementRepository(
                    new \App\Models\Warehouse\StockMovement()
                );
            }
        );

        $this->app->bind(
            \App\Repositories\Warehouse\SupplierRepository::class,
            function ($app) {
                return new \App\Repositories\Warehouse\SupplierRepository(
                    new \App\Models\Warehouse\Supplier()
                );
            }
        );
    }

    /**
     * Bootstrap any application services.
     */
    public function boot(): void
    {
        // Register policies
        foreach ($this->policies as $model => $policy) {
            Gate::policy($model, $policy);
        }

        // Register warehouse permission gates
        Gate::define('warehouse.access', function ($user) {
            $accessService = app(RoleBasedAccessService::class);

            // Super admins and admins always have warehouse access
            if ($accessService->isSuperAdmin($user) || $accessService->isAdmin($user)) {
                return true;
            }

            return $accessService->hasWarehouseAccess($user);
        });

        Gate::define('warehouse.admin', function ($user) {
            return app(RoleBasedAccessService::class)->isWarehouseAdmin($user);
        });

        Gate::define('warehouse.manager', function ($user) {
            return app(RoleBasedAccessService::class)->isWarehouseManagerOrHigher($user);
        });

        Gate::define('warehouse.supervisor', function ($user) {
            return app(RoleBasedAccessService::class)->isWarehouseSupervisorOrHigher($user);
        });

        // Define all warehouse permission gates
        $warehousePermissions = [
            'warehouse.dashboard.view',
            'warehouse.suppliers.view',
            'warehouse.suppliers.create',
            'warehouse.suppliers.edit',
            'warehouse.suppliers.delete',
            'warehouse.projects.view',
            'warehouse.projects.create',
            'warehouse.projects.edit',
            'warehouse.projects.delete',
            'warehouse.project-divisions.view',
            'warehouse.project-divisions.create',
            'warehouse.project-divisions.edit',
            'warehouse.project-divisions.delete',
            'warehouse.items.view',
            'warehouse.items.create',
            'warehouse.items.edit',
            'warehouse.items.delete',
            'warehouse.item-categories.view',
            'warehouse.item-categories.create',
            'warehouse.item-categories.edit',
            'warehouse.item-categories.delete',
            'warehouse.inventory.view',
            'warehouse.inventory.adjust',
            'warehouse.inventory.transfer',
            'warehouse.inventory.count',
            'warehouse.incoming.view',
            'warehouse.incoming.create',
            'warehouse.incoming.edit',
            'warehouse.incoming.complete',
            'warehouse.outgoing.view',
            'warehouse.outgoing.create',
            'warehouse.outgoing.approve',
            'warehouse.outgoing.fulfill',
            'warehouse.returns.view',
            'warehouse.returns.create',
            'warehouse.returns.process',
            'warehouse.reports.view',
            'warehouse.reports.export',
            'warehouse.reports.advanced',
            'warehouse.analytics.view',
            'warehouse.settings.view',
            'warehouse.settings.manage',
            'warehouse.users.view',
            'warehouse.users.create',
            'warehouse.users.edit',
            'warehouse.users.delete',
            'warehouse.activity-log.view',
            'warehouse.locations.view',
            'warehouse.locations.create',
            'warehouse.locations.edit',
            'warehouse.locations.delete'
        ];

        foreach ($warehousePermissions as $permission) {
            Gate::define($permission, function ($user) use ($permission) {
                $accessService = app(RoleBasedAccessService::class);
                return $accessService->hasPermission($user, $permission);
            });
        }
    }
}
