<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::table('direct_delivery_items', function (Blueprint $table) {
            // Only add columns that don't exist yet
            if (!Schema::hasColumn('direct_delivery_items', 'currency')) {
                $table->enum('currency', ['AED', 'USD', 'EUR'])->default('AED')->after('total_price');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'unit_price_aed')) {
                $table->decimal('unit_price_aed', 10, 2)->after('currency');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'total_price_aed')) {
                $table->decimal('total_price_aed', 15, 2)->after('unit_price_aed');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'serial_number')) {
                $table->string('serial_number')->nullable()->after('batch_number');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'manufacturing_date')) {
                $table->date('manufacturing_date')->nullable()->after('serial_number');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'country_of_origin')) {
                $table->string('country_of_origin')->nullable()->after('manufacturer');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'quality_checklist')) {
                $table->json('quality_checklist')->nullable()->after('quality_notes');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'storage_conditions')) {
                $table->string('storage_conditions')->nullable()->after('storage_location');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'hazardous_material')) {
                $table->boolean('hazardous_material')->default(false)->after('storage_conditions');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'hazard_classifications')) {
                $table->json('hazard_classifications')->nullable()->after('hazardous_material');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'warranty_period_months')) {
                $table->integer('warranty_period_months')->nullable()->after('hazard_classifications');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'warranty_expiry')) {
                $table->date('warranty_expiry')->nullable()->after('warranty_period_months');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'requires_maintenance')) {
                $table->boolean('requires_maintenance')->default(false)->after('warranty_expiry');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'maintenance_interval_days')) {
                $table->integer('maintenance_interval_days')->nullable()->after('requires_maintenance');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'allocated_project_id')) {
                $table->foreignId('allocated_project_id')
                    ->nullable()
                    ->after('maintenance_interval_days')
                    ->constrained('projects')
                    ->nullOnDelete();
            }

            if (!Schema::hasColumn('direct_delivery_items', 'project_phase')) {
                $table->string('project_phase')->nullable()->after('allocated_project_id');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'work_area')) {
                $table->string('work_area')->nullable()->after('project_phase');
            }

            if (!Schema::hasColumn('direct_delivery_items', 'status')) {
                $table->enum('status', ['pending', 'delivered', 'accepted', 'rejected', 'returned'])
                    ->default('pending')
                    ->after('work_area');
            }

            // Add indexes for new columns
            if (!Schema::hasColumn('direct_delivery_items', 'allocated_project_id')) {
                $table->index(['allocated_project_id']);
            }
            if (!Schema::hasColumn('direct_delivery_items', 'status')) {
                $table->index(['status']);
            }
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('direct_delivery_items', function (Blueprint $table) {
            $columnsToCheck = [
                'currency', 'unit_price_aed', 'total_price_aed', 'serial_number',
                'manufacturing_date', 'country_of_origin', 'quality_checklist',
                'storage_conditions', 'hazardous_material', 'hazard_classifications',
                'warranty_period_months', 'warranty_expiry', 'requires_maintenance',
                'maintenance_interval_days', 'allocated_project_id', 'project_phase',
                'work_area', 'status'
            ];

            foreach ($columnsToCheck as $column) {
                if (Schema::hasColumn('direct_delivery_items', $column)) {
                    if ($column === 'allocated_project_id') {
                        $table->dropForeign(['allocated_project_id']);
                        $table->dropIndex(['allocated_project_id']);
                    }
                    if ($column === 'status') {
                        $table->dropIndex(['status']);
                    }
                    $table->dropColumn($column);
                }
            }
        });
    }
};