<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('foc_inventory', function (Blueprint $table) {
            $table->id();

            // Basic item information
            $table->foreignId('item_id')->constrained('items');
            $table->unsignedBigInteger('division_id')->nullable(); // No FK constraint since divisions table may not exist

            // Quantity and availability
            $table->decimal('quantity_available', 15, 4)->default(0);
            $table->decimal('quantity_reserved', 15, 4)->default(0); // For pending dispatches
            $table->string('unit', 20)->nullable();

            // Source information (from damaged items/returns)
            $table->enum('source_type', ['site_return', 'inventory_conversion', 'project_return', 'damaged_item']);
            $table->string('source_reference')->nullable(); // Reference to original operation
            $table->foreignId('source_damaged_item_id')->nullable()->constrained('damaged_items');
            $table->foreignId('source_project_id')->nullable()->constrained('projects');

            // Item condition and details
            $table->enum('condition', ['damaged', 'excess', 'unused', 'serviceable'])->default('damaged');
            $table->text('condition_notes')->nullable();
            $table->enum('damage_severity', ['minor', 'moderate', 'severe'])->nullable();

            // Batch and tracking information
            $table->string('batch_number')->nullable();
            $table->string('supplier_batch_number')->nullable();
            $table->date('production_date')->nullable();
            $table->date('expiry_date')->nullable();

            // Location and storage
            $table->string('storage_location')->nullable();
            $table->string('bin_location')->nullable();

            // Status and quality
            $table->enum('status', ['available', 'reserved', 'dispatched', 'disposed'])->default('available');
            $table->enum('quality_status', ['approved', 'pending', 'rejected'])->default('approved');
            $table->text('quality_notes')->nullable();

            // Cost tracking (useful for reports)
            $table->decimal('estimated_value', 15, 2)->nullable(); // Original estimated value
            $table->decimal('recoverable_value', 15, 2)->nullable(); // Current estimated recoverable value

            // Timestamps and tracking
            $table->timestamp('received_at')->nullable(); // When received as FOC
            $table->timestamp('last_dispatched_at')->nullable();
            $table->foreignId('created_by')->nullable()->constrained('users');
            $table->foreignId('updated_by')->nullable()->constrained('users');

            $table->timestamps();
            $table->softDeletes();

            // Indexes for performance
            $table->index(['item_id', 'status']);
            $table->index(['source_type', 'source_reference']);
            $table->index(['condition', 'status']);
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('foc_inventory');
    }
};
