<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('foc_inventory_adjustments', function (Blueprint $table) {
            $table->id();

            // Reference to FOC inventory
            $table->foreignId('foc_inventory_id')->constrained('foc_inventory')->onDelete('cascade');
            $table->foreignId('item_id')->constrained('items'); // For easy querying

            // Adjustment details
            $table->enum('adjustment_type', [
                'stock_increase',      // Adding stock
                'stock_decrease',      // Reducing stock
                'condition_change',    // Changing condition (damaged -> serviceable)
                'quality_change',      // Quality status change
                'disposal',           // Writing off stock
                'dispatch',           // Dispatching for material issue
                'reserve',           // Reserving stock
                'unreserve',         // Unreserving stock
                'correction'         // General correction
            ]);

            // Quantity changes
            $table->decimal('quantity_before', 15, 4); // Stock before adjustment
            $table->decimal('quantity_after', 15, 4);  // Stock after adjustment
            $table->decimal('quantity_change', 15, 4); // Change amount (+/-)

            // Condition changes
            $table->enum('condition_before', ['damaged', 'excess', 'unused', 'serviceable'])->nullable();
            $table->enum('condition_after', ['damaged', 'excess', 'unused', 'serviceable'])->nullable();

            // Status changes
            $table->enum('status_before', ['available', 'reserved', 'dispatched', 'disposed'])->nullable();
            $table->enum('status_after', ['available', 'reserved', 'dispatched', 'disposed'])->nullable();

            // Reason and reference
            $table->string('reason')->nullable(); // Reason for adjustment
            $table->text('notes')->nullable();   // Additional notes
            $table->string('reference_type')->nullable(); // Type of reference (material_issue, site_return, etc.)
            $table->string('reference_id')->nullable();   // ID of the reference
            $table->string('reference_number')->nullable(); // Human readable reference

            // Cost impact
            $table->decimal('cost_impact', 15, 2)->nullable(); // Financial impact of adjustment

            // Approval workflow
            $table->enum('approval_status', ['pending', 'approved', 'rejected'])->default('pending');
            $table->foreignId('requested_by')->constrained('users');
            $table->foreignId('approved_by')->nullable()->constrained('users');
            $table->timestamp('approved_at')->nullable();

            $table->timestamps();

            // Indexes
            $table->index(['foc_inventory_id', 'adjustment_type']);
            $table->index(['item_id', 'created_at']);
            $table->index(['reference_type', 'reference_id']);
            $table->index('approval_status');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::dropIfExists('foc_inventory_adjustments');
    }
};
