<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // First, create department records that match project divisions
        $projectDivisions = DB::table('project_divisions')->get(['id', 'division_name', 'division_code']);

        foreach ($projectDivisions as $division) {
            // Check if department already exists with this ID
            $existingDept = DB::table('departments')->where('id', $division->id)->first();

            if (!$existingDept) {
                // Create department record with same ID as project division
                DB::table('departments')->insert([
                    'id' => $division->id,
                    'name' => $division->division_name,
                    'description' => 'Auto-created from project division',
                    'code' => $division->division_code,
                    'manager_name' => null,
                    'manager_email' => null,
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }

        // Now update inventory records with division_id based on their source incoming operations
        $query = "
            UPDATE inventory i
            INNER JOIN incoming_operation_items ioi ON i.item_id = ioi.item_id
            INNER JOIN incoming_operations io ON ioi.incoming_operation_id = io.id
            SET i.division_id = io.division
            WHERE i.division_id IS NULL
            AND io.division IS NOT NULL
            AND io.division REGEXP '^[0-9]+$'
            AND EXISTS (SELECT 1 FROM departments d WHERE d.id = io.division)
        ";

        $affectedRows = DB::update($query);

        // Log the results
        \Log::info("Updated {$affectedRows} inventory records with division_id from incoming operations");

        // For inventory records that still don't have division_id, we could try other approaches:
        $remainingNull = DB::table('inventory')
            ->whereNull('division_id')
            ->where('quantity_available', '>', 0)
            ->count();

        if ($remainingNull > 0) {
            \Log::info("There are still {$remainingNull} inventory records with NULL division_id that need manual assignment");
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // Revert inventory division_id to NULL
        // Only revert records that were updated by this migration
        DB::table('inventory')->update(['division_id' => null]);
        \Log::info("Reverted all inventory division_id values to NULL");
    }
};