<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\ProductCategory;
use App\Models\ProductStandard;
use App\Models\ProductSubcategory;
use App\Models\ProductApplication;
use App\Models\Product;

class PatternGlassSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create Pattern Glass Category
        $patternGlass = ProductCategory::firstOrCreate(
            ['slug' => 'pattern-glass'],
            [
                'name' => 'Pattern Glass',
                'description' => 'Decorative glass with various patterns and textures',
                'sort_order' => 2,
                'is_active' => true
            ]
        );

        // Create Standards
        $enStandard = ProductStandard::firstOrCreate(
            ['category_id' => $patternGlass->id, 'code' => 'EN'],
            [
                'name' => 'European Standard',
                'description' => 'European standards for pattern glass',
                'sort_order' => 1,
                'is_active' => true
            ]
        );

        $sasoStandard = ProductStandard::firstOrCreate(
            ['category_id' => $patternGlass->id, 'code' => 'SASO'],
            [
                'name' => 'Saudi Standards Organization',
                'description' => 'Saudi standards for pattern glass',
                'sort_order' => 2,
                'is_active' => true
            ]
        );

        // EN Standard Subcategories and Applications
        $this->createENSubcategories($enStandard);
        
        // SASO Standard Subcategories and Applications
        $this->createSASOSubcategories($sasoStandard);
    }

    private function createENSubcategories($enStandard)
    {
        // 1. Decorative
        $decorative = ProductSubcategory::firstOrCreate(
            ['standard_id' => $enStandard->id, 'slug' => 'decorative'],
            [
                'name' => 'Decorative',
                'description' => 'Decorative pattern glass',
                'sort_order' => 1,
                'is_active' => true
            ]
        );

        $decorativeApp = ProductApplication::firstOrCreate(
            ['subcategory_id' => $decorative->id, 'name' => 'Decorative'],
            [
                'description' => 'General decorative applications',
                'sort_order' => 1,
                'is_active' => true
            ]
        );

        // Create decorative products
        $decorativeProducts = ['Nashji', 'Dewani', 'Milenium', 'Delta'];
        foreach ($decorativeProducts as $index => $productName) {
            Product::firstOrCreate(
                ['application_id' => $decorativeApp->id, 'name' => $productName],
                [
                    'slug' => strtolower(str_replace(' ', '-', $productName)),
                    'description' => "High-quality {$productName} pattern glass for decorative applications",
                    'sort_order' => $index + 1,
                    'is_featured' => false,
                    'is_active' => true
                ]
            );
        }

        // 2. Bronze
        $bronze = ProductSubcategory::firstOrCreate(
            ['standard_id' => $enStandard->id, 'slug' => 'bronze'],
            [
                'name' => 'Bronze',
                'description' => 'Bronze tinted pattern glass',
                'sort_order' => 2,
                'is_active' => true
            ]
        );

        // Bronze Wired
        $bronzeWired = ProductApplication::firstOrCreate(
            ['subcategory_id' => $bronze->id, 'name' => 'Bronze Wired'],
            [
                'description' => 'Bronze pattern glass with wire reinforcement',
                'sort_order' => 1,
                'is_active' => true
            ]
        );

        Product::firstOrCreate(
            ['application_id' => $bronzeWired->id, 'name' => 'Nashji'],
            [
                'slug' => 'bronze-wired-nashji',
                'description' => 'Bronze wired Nashji pattern glass',
                'sort_order' => 1,
                'is_featured' => false,
                'is_active' => true
            ]
        );

        // Bronze Non Wired
        $bronzeNonWired = ProductApplication::firstOrCreate(
            ['subcategory_id' => $bronze->id, 'name' => 'Bronze Non Wired'],
            [
                'description' => 'Bronze pattern glass without wire reinforcement',
                'sort_order' => 2,
                'is_active' => true
            ]
        );

        Product::firstOrCreate(
            ['application_id' => $bronzeNonWired->id, 'name' => 'Nashji'],
            [
                'slug' => 'bronze-non-wired-nashji',
                'description' => 'Bronze non-wired Nashji pattern glass',
                'sort_order' => 1,
                'is_featured' => false,
                'is_active' => true
            ]
        );

        // 3. Clear
        $clear = ProductSubcategory::firstOrCreate(
            ['standard_id' => $enStandard->id, 'slug' => 'clear'],
            [
                'name' => 'Clear',
                'description' => 'Clear pattern glass',
                'sort_order' => 3,
                'is_active' => true
            ]
        );

        // Clear Non Wired
        $clearNonWired = ProductApplication::firstOrCreate(
            ['subcategory_id' => $clear->id, 'name' => 'Clear Non Wired'],
            [
                'description' => 'Clear pattern glass without wire reinforcement',
                'sort_order' => 1,
                'is_active' => true
            ]
        );

        $clearNonWiredProducts = ['Nashji', 'Milenium', 'Dewani', 'Delta', 'Matrix', 'Pyramid', 'Reeded'];
        foreach ($clearNonWiredProducts as $index => $productName) {
            Product::firstOrCreate(
                ['application_id' => $clearNonWired->id, 'name' => $productName],
                [
                    'slug' => 'clear-non-wired-' . strtolower(str_replace(' ', '-', $productName)),
                    'description' => "Clear non-wired {$productName} pattern glass",
                    'sort_order' => $index + 1,
                    'is_featured' => false,
                    'is_active' => true
                ]
            );
        }

        // Clear Wired (no products mentioned in your structure)
        ProductApplication::firstOrCreate(
            ['subcategory_id' => $clear->id, 'name' => 'Clear Wired'],
            [
                'description' => 'Clear pattern glass with wire reinforcement',
                'sort_order' => 2,
                'is_active' => true
            ]
        );
    }

    private function createSASOSubcategories($sasoStandard)
    {
        // 1. Clear
        $clear = ProductSubcategory::firstOrCreate(
            ['standard_id' => $sasoStandard->id, 'slug' => 'clear'],
            [
                'name' => 'Clear',
                'description' => 'Clear pattern glass under SASO standards',
                'sort_order' => 1,
                'is_active' => true
            ]
        );

        $clearNonWired = ProductApplication::firstOrCreate(
            ['subcategory_id' => $clear->id, 'name' => 'Pattern Non Wired'],
            [
                'description' => 'Clear pattern glass without wire reinforcement',
                'sort_order' => 1,
                'is_active' => true
            ]
        );

        $clearProducts = ['Nashji', 'Milenium', 'Dewani', 'Delta', 'Matrix', 'Pyramid', 'Reeded'];
        foreach ($clearProducts as $index => $productName) {
            Product::firstOrCreate(
                ['application_id' => $clearNonWired->id, 'name' => $productName],
                [
                    'slug' => 'saso-clear-non-wired-' . strtolower(str_replace(' ', '-', $productName)),
                    'description' => "SASO standard clear non-wired {$productName} pattern glass",
                    'sort_order' => $index + 1,
                    'is_featured' => false,
                    'is_active' => true
                ]
            );
        }

        // 2. Bronze
        $bronze = ProductSubcategory::firstOrCreate(
            ['standard_id' => $sasoStandard->id, 'slug' => 'bronze'],
            [
                'name' => 'Bronze',
                'description' => 'Bronze pattern glass under SASO standards',
                'sort_order' => 2,
                'is_active' => true
            ]
        );

        ProductApplication::firstOrCreate(
            ['subcategory_id' => $bronze->id, 'name' => 'Pattern Non Wired'],
            [
                'description' => 'Bronze pattern glass without wire reinforcement',
                'sort_order' => 1,
                'is_active' => true
            ]
        );
        // Note: No products mentioned for SASO Bronze in your structure
    }
}