/**
 * Warehouse Reports Common JavaScript
 */

class WarehouseReports {
    constructor() {
        this.init();
    }

    init() {
        this.initializeSelect2();
        this.initializeLoadingOverlays();
        this.initializeTableFeatures();
        this.initializeExportFeatures();
        
        console.log('🏭 Warehouse Reports initialized');
    }

    /**
     * Initialize Select2 dropdowns
     */
    initializeSelect2() {
        if (typeof $.fn.select2 !== 'undefined') {
            $('.warehouse-select2, .searchable-dropdown').select2({
                placeholder: function() {
                    return $(this).find('option:first').text();
                },
                allowClear: true,
                width: '100%',
                theme: 'default',
                minimumResultsForSearch: 0 // Always show search box
            });
            console.log('✅ Select2 searchable dropdowns initialized');
        } else {
            console.warn('⚠️ Select2 not loaded, trying to load...');
            this.loadSelect2();
        }
    }

    /**
     * Load Select2 if not available
     */
    loadSelect2() {
        if (!document.querySelector('link[href*="select2"]')) {
            const link = document.createElement('link');
            link.rel = 'stylesheet';
            link.href = 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css';
            document.head.appendChild(link);
        }

        if (typeof $.fn.select2 === 'undefined') {
            const script = document.createElement('script');
            script.src = 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js';
            script.onload = () => {
                this.initializeSelect2();
            };
            document.head.appendChild(script);
        }
    }

    /**
     * Initialize loading overlays for forms and pagination
     */
    initializeLoadingOverlays() {
        // Form submissions
        $(document).on('submit', '.warehouse-filter-form, #filterForm', (e) => {
            this.showTableLoading();
        });

        // Pagination clicks
        $(document).on('click', '.warehouse-pagination a, .pagination a', (e) => {
            e.preventDefault();
            this.showTableLoading();
            window.location.href = $(e.currentTarget).attr('href');
        });

        console.log('✅ Loading overlays initialized');
    }

    /**
     * Show table loading overlay
     */
    showTableLoading() {
        const overlay = $('.warehouse-loading-overlay, #tableLoadingOverlay');
        if (overlay.length) {
            overlay.addClass('active').show();
        }
    }

    /**
     * Hide table loading overlay
     */
    hideTableLoading() {
        const overlay = $('.warehouse-loading-overlay, #tableLoadingOverlay');
        if (overlay.length) {
            overlay.removeClass('active').hide();
        }
    }

    /**
     * Initialize table features
     */
    initializeTableFeatures() {
        // Sticky header on scroll
        this.initializeStickyHeaders();
        
        // Row hover effects
        $('.warehouse-table tbody tr').hover(
            function() { $(this).addClass('table-active'); },
            function() { $(this).removeClass('table-active'); }
        );

        console.log('✅ Table features initialized');
    }

    /**
     * Initialize sticky table headers
     */
    initializeStickyHeaders() {
        const tables = $('.warehouse-table');
        
        tables.each(function() {
            const $table = $(this);
            const $wrapper = $table.closest('.warehouse-table-wrapper');
            
            if ($wrapper.length) {
                $wrapper.on('scroll', function() {
                    const scrollTop = $(this).scrollTop();
                    const $thead = $table.find('thead th');
                    
                    if (scrollTop > 0) {
                        $thead.addClass('shadow-sm');
                    } else {
                        $thead.removeClass('shadow-sm');
                    }
                });
            }
        });
    }

    /**
     * Initialize export features
     */
    initializeExportFeatures() {
        // Excel export
        $(document).on('click', '.warehouse-export-excel, [data-export="excel"]', (e) => {
            e.preventDefault();
            this.exportToExcel();
        });

        // PDF export
        $(document).on('click', '.warehouse-export-pdf, [data-export="pdf"]', (e) => {
            e.preventDefault();
            this.exportToPdf();
        });

        // Print
        $(document).on('click', '.warehouse-print, [data-action="print"]', (e) => {
            e.preventDefault();
            this.printReport();
        });

        console.log('✅ Export features initialized');
    }

    /**
     * Export report to Excel
     */
    exportToExcel() {
        this.showTableLoading();
        
        const params = new URLSearchParams(window.location.search);
        params.append('export', 'excel');
        
        const exportUrl = window.location.pathname + '?' + params.toString();
        
        // Create invisible download link
        const link = document.createElement('a');
        link.href = exportUrl;
        link.download = '';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        // Hide loading after delay
        setTimeout(() => {
            this.hideTableLoading();
        }, 2000);
    }

    /**
     * Export report to PDF
     */
    exportToPdf() {
        this.showTableLoading();
        
        const params = new URLSearchParams(window.location.search);
        params.append('export', 'pdf');
        
        const exportUrl = window.location.pathname + '?' + params.toString();
        window.open(exportUrl, '_blank');
        
        setTimeout(() => {
            this.hideTableLoading();
        }, 1000);
    }

    /**
     * Print report
     */
    printReport() {
        window.print();
    }

    /**
     * Refresh table data
     */
    refreshTable() {
        this.showTableLoading();
        window.location.reload();
    }

    /**
     * Clear all filters
     */
    clearFilters() {
        const form = $('.warehouse-filter-form, #filterForm');
        if (form.length) {
            form[0].reset();
            $('.warehouse-select2, .searchable-dropdown').val(null).trigger('change');
        }
    }

    /**
     * Apply filters
     */
    applyFilters() {
        const form = $('.warehouse-filter-form, #filterForm');
        if (form.length) {
            form.submit();
        }
    }

    /**
     * Initialize date pickers
     */
    initializeDatePickers() {
        if (typeof flatpickr !== 'undefined') {
            flatpickr('.warehouse-datepicker', {
                dateFormat: 'Y-m-d',
                allowInput: true
            });
            console.log('✅ Date pickers initialized');
        }
    }

    /**
     * Show notification
     */
    showNotification(message, type = 'info') {
        // Simple notification system
        const alertClass = type === 'error' ? 'alert-danger' : 
                          type === 'success' ? 'alert-success' : 
                          type === 'warning' ? 'alert-warning' : 'alert-info';
        
        const notification = $(`
            <div class="alert ${alertClass} alert-dismissible fade show position-fixed" 
                 style="top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
                ${message}
                <button type="button" class="close" data-dismiss="alert">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
        `);
        
        $('body').append(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            notification.alert('close');
        }, 5000);
    }
}

// Initialize when document is ready
$(document).ready(function() {
    window.warehouseReports = new WarehouseReports();
});

// Global utility functions
window.WarehouseReports = {
    showLoading: () => window.warehouseReports?.showTableLoading(),
    hideLoading: () => window.warehouseReports?.hideTableLoading(),
    refresh: () => window.warehouseReports?.refreshTable(),
    clearFilters: () => window.warehouseReports?.clearFilters(),
    exportExcel: () => window.warehouseReports?.exportToExcel(),
    exportPdf: () => window.warehouseReports?.exportToPdf(),
    print: () => window.warehouseReports?.printReport()
};