/* Progressive Tab Morphing Script */
document.addEventListener('DOMContentLoaded', function() {
    // Progressive Filter State
    let currentStep = 'categories'; // categories, standards, applications
    let selectedCategory = 'all';
    let selectedStandard = '';
    let selectedApplication = '';
    
    // DOM Elements
    const scrollContainer = document.getElementById('productCarousel');
    const scrollPrev = document.getElementById('scrollPrev');
    const scrollNext = document.getElementById('scrollNext');
    const backButton = document.getElementById('back-button');
    const tabsContainer = document.getElementById('filter-tabs-container');
    
    // Check if elements exist before proceeding
    if (!scrollContainer || !tabsContainer) {
        return; // Exit if required elements don't exist
    }
    
    // Scroll navigation functionality
    const scrollAmount = 340;
    
    function updateScrollButtons() {
        if (!scrollContainer) return;
        const isAtStart = scrollContainer.scrollLeft <= 0;
        const isAtEnd = scrollContainer.scrollLeft >= scrollContainer.scrollWidth - scrollContainer.clientWidth - 10;
        if (scrollPrev) scrollPrev.classList.toggle('disabled', isAtStart);
        if (scrollNext) scrollNext.classList.toggle('disabled', isAtEnd);
    }
    
    if (scrollPrev) {
        scrollPrev.addEventListener('click', () => {
            if (scrollContainer) scrollContainer.scrollBy({ left: -scrollAmount, behavior: 'smooth' });
        });
    }
    
    if (scrollNext) {
        scrollNext.addEventListener('click', () => {
            if (scrollContainer) scrollContainer.scrollBy({ left: scrollAmount, behavior: 'smooth' });
        });
    }
    
    if (scrollContainer) {
        scrollContainer.addEventListener('scroll', updateScrollButtons);
        updateScrollButtons();
    }
    
    // Update carousel with filtered products
    function updateCarousel() {
        if (!scrollContainer || !window.homePageData) return;
        
        let products = window.homePageData.allProducts;
        
        // Apply filters progressively
        if (selectedCategory !== 'all') {
            products = products.filter(product => {
                if (product.application?.standard?.category) {
                    return product.application.standard.category.slug === selectedCategory;
                }
                if (product.standard?.category) {
                    return product.standard.category.slug === selectedCategory;
                }
                return false;
            });
        }
        
        if (selectedStandard) {
            products = products.filter(product => {
                const standardCode = product.application?.standard?.code || product.standard?.code;
                return standardCode && standardCode.toLowerCase().replace(/[^a-z]/g, '') === selectedStandard;
            });
        }
        
        if (selectedApplication) {
            products = products.filter(product => {
                return product.application && 
                       product.application.name.toLowerCase().replace(/\s+/g, '-') === selectedApplication;
            });
        }
        
        // Update carousel
        if (products.length === 0) {
            scrollContainer.innerHTML = `
                <div class="no-products-message">
                    <div class="no-products-content">
                        <svg width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                            <circle cx="11" cy="11" r="8"/>
                            <path d="m21 21-4.35-4.35"/>
                        </svg>
                        <h3>No Products Found</h3>
                        <p>No products are available for the selected criteria. Try a different selection.</p>
                    </div>
                </div>
            `;
        } else {
            scrollContainer.innerHTML = products.map(product => {
                let categorySlug = '';
                if (product.application?.standard?.category) {
                    categorySlug = product.application.standard.category.slug;
                } else if (product.standard?.category) {
                    categorySlug = product.standard.category.slug;
                }
                return `
                <a href="/products/${categorySlug ? categorySlug : ''}${categorySlug && product.slug ? '/' : ''}${product.slug || ''}" class="product-card" style="text-decoration: none; color: inherit; display: block;">
                    <div class="product-image">
                        <img src="${product.image ? (product.image.startsWith('http') ? product.image : (product.image.startsWith('assets/') ? '/' + product.image : '/storage/' + product.image)) : '/assets/2025/08/22dafa95b499b0bb8e5259edab864dd8.png'}" alt="${product.name}">
                    </div>
                    <div class="product-content">
                        <h3 class="product-title">${product.name.toUpperCase()}</h3>
                        <p class="product-subtitle">ARALUX ® ${(product.application?.standard?.category?.name || product.standard?.category?.name || product.name).toUpperCase()}</p>
                    </div>
                </a>
            `}).join('');
        }
        
        scrollContainer.scrollLeft = 0;
        updateScrollButtons();
    }
    
    // Morph tabs with animation
    function morphTabs(newTabs, newStep) {
        // Fade out current tabs
        const currentTabs = tabsContainer.querySelectorAll('.filter-tab');
        currentTabs.forEach(tab => tab.classList.add('fade-out'));
        
        setTimeout(() => {
            tabsContainer.innerHTML = newTabs;
            currentStep = newStep;
            
            // Fade in new tabs
            const newTabElements = tabsContainer.querySelectorAll('.filter-tab');
            newTabElements.forEach(tab => {
                tab.classList.add('fade-in');
                setTimeout(() => tab.classList.remove('fade-in'), 400);
            });
            
            // Add event listeners to new tabs
            addTabEventListeners();
        }, 200);
    }
    
    // Show standards for selected category
    function showStandards(categorySlug) {
        if (!window.homePageData) return;
        
        const categoryStandards = window.homePageData.standards.filter(standard => 
            standard.category && standard.category.slug === categorySlug
        );
        
        const standardTabs = categoryStandards.map(standard => 
            `<button class="filter-tab" data-standard="${standard.code.toLowerCase().replace(/[^a-z]/g, '')}">${standard.code}</button>`
        ).join('');
        
        morphTabs(standardTabs, 'standards');
        if (backButton) backButton.classList.add('show');
    }
    
    // Show applications for selected standard
    function showApplications(standardCode) {
        if (!window.homePageData) return;
        
        const standardApplications = window.homePageData.applications.filter(application =>
            application.standard && 
            application.standard.code.toLowerCase().replace(/[^a-z]/g, '') === standardCode
        );
        
        const applicationTabs = standardApplications.map(application =>
            `<button class="filter-tab" data-application="${application.name.toLowerCase().replace(/\s+/g, '-')}">${application.name}</button>`
        ).join('');
        
        morphTabs(applicationTabs, 'applications');
    }
    
    // Go back to previous step
    function goBack() {
        if (!window.homePageData) return;
        
        if (currentStep === 'standards') {
            // Back to categories
            selectedStandard = '';
            selectedApplication = '';
            const categoryTabs = `
                <button class="filter-tab ${selectedCategory === 'all' ? 'active' : ''}" data-category="all">All</button>
                ${window.homePageData.categories.map(category => 
                    `<button class="filter-tab ${selectedCategory === category.slug ? 'active' : ''}" data-category="${category.slug}">${category.name}</button>`
                ).join('')}
            `;
            morphTabs(categoryTabs, 'categories');
            if (backButton) backButton.classList.remove('show');
        } else if (currentStep === 'applications') {
            // Back to standards
            selectedApplication = '';
            showStandards(selectedCategory);
        }
        updateCarousel();
    }
    
    // Add event listeners to current tabs
    function addTabEventListeners() {
        // Category tabs
        tabsContainer.querySelectorAll('[data-category]').forEach(tab => {
            tab.addEventListener('click', function() {
                selectedCategory = this.dataset.category;
                selectedStandard = '';
                selectedApplication = '';
                
                // Update active state
                tabsContainer.querySelectorAll('.filter-tab').forEach(t => t.classList.remove('active'));
                this.classList.add('active');
                
                if (selectedCategory === 'all') {
                    updateCarousel();
                } else {
                    showStandards(selectedCategory);
                    updateCarousel();
                }
            });
        });
        
        // Standard tabs
        tabsContainer.querySelectorAll('[data-standard]').forEach(tab => {
            tab.addEventListener('click', function() {
                selectedStandard = this.dataset.standard;
                selectedApplication = '';
                
                // Update active state
                tabsContainer.querySelectorAll('.filter-tab').forEach(t => t.classList.remove('active'));
                this.classList.add('active');
                
                showApplications(selectedStandard);
                updateCarousel();
            });
        });
        
        // Application tabs
        tabsContainer.querySelectorAll('[data-application]').forEach(tab => {
            tab.addEventListener('click', function() {
                selectedApplication = this.dataset.application;
                
                // Update active state
                tabsContainer.querySelectorAll('.filter-tab').forEach(t => t.classList.remove('active'));
                this.classList.add('active');
                
                updateCarousel();
            });
        });
    }
    
    // Back button event listener
    if (backButton) {
        backButton.addEventListener('click', goBack);
        backButton.classList.remove('show');
    }
    
    // Initialize
    addTabEventListeners();
    updateCarousel();
});