// Mobile Products Section Builder
document.addEventListener('DOMContentLoaded', function() {
    console.log('Mobile Products Builder: DOMContentLoaded fired');
    console.log('Current window width:', window.innerWidth);
    
    // Only run on mobile devices
    function isMobile() {
        return window.innerWidth <= 768;
    }
    
    if (!isMobile()) {
        console.log('Desktop detected - skipping mobile products builder');
        return;
    }
    
    console.log('Mobile detected - Building mobile products section...');
    
    // Add a visible debug indicator
    const debugDiv = document.createElement('div');
    debugDiv.style.cssText = `
        position: fixed;
        top: 10px;
        right: 10px;
        background: red;
        color: white;
        padding: 5px 10px;
        font-size: 12px;
        z-index: 99999;
        border-radius: 5px;
    `;
    debugDiv.textContent = 'Mobile Products Loading...';
    document.body.appendChild(debugDiv);
    
    // Remove debug indicator after 3 seconds
    setTimeout(() => {
        if (debugDiv.parentNode) {
            debugDiv.remove();
        }
    }, 3000);
    
    // Product data
    const products = [
        {
            id: 'clear-float',
            category: 'float',
            title: 'Clear Float',
            subtitle: 'ARALUX® Clear Float Glass',
            description: 'Premium quality clear float glass for modern applications',
            image: 'images/Float/DSC00418.jpg',
            icon: 'CF'
        },
        {
            id: 'silver-mirror',
            category: 'mirror',
            title: 'Silver Mirror',
            subtitle: 'ARALUX® Silver Mirror',
            description: 'High-quality silver mirror with superior reflectivity',
            image: 'images/Mirror/DSC00544.png',
            icon: 'SM'
        },
        {
            id: 'pattern-glass',
            category: 'pattern',
            title: 'Pattern Glass',
            subtitle: 'ARALUX® Pattern Glass',
            description: 'Decorative pattern glass for aesthetic applications',
            image: 'images/Pattern/Bronze/DSC00545 (1).png',
            icon: 'PG'
        },
        {
            id: 'decorative-mirror',
            category: 'mirror',
            title: 'Decorative Mirror',
            subtitle: 'ARALUX® Decorative Mirror',
            description: 'Artistic decorative mirrors for premium spaces',
            image: 'images/Pattern/Bronze/DSC00545 (1).png',
            icon: 'DM'
        }
    ];
    
    // Filter categories
    const categories = [
        { id: 'all', name: 'All', count: products.length },
        { id: 'float', name: 'Float Glass', count: products.filter(p => p.category === 'float').length },
        { id: 'pattern', name: 'Pattern Glass', count: products.filter(p => p.category === 'pattern').length },
        { id: 'mirror', name: 'Mirror', count: products.filter(p => p.category === 'mirror').length }
    ];
    
    // Build mobile products HTML
    function buildMobileProductsHTML() {
        const categoriesHTML = categories.map(cat => `
            <div class="mobile-filter-tab ${cat.id === 'all' ? 'active' : ''}" data-category="${cat.id}">
                ${cat.name}
            </div>
        `).join('');
        
        const productsHTML = products.map(product => `
            <div class="mobile-product-card fade-in" data-category="${product.category}" data-product="${product.id}">
                <div class="mobile-product-icon">${product.icon}</div>
                <div class="mobile-product-image">
                    <img src="${product.image}" alt="${product.title}" loading="lazy">
                </div>
                <div class="mobile-product-content">
                    <h3 class="mobile-product-title">${product.title}</h3>
                    <p class="mobile-product-subtitle">${product.subtitle}</p>
                    <p class="mobile-product-description">${product.description}</p>
                </div>
            </div>
        `).join('');
        
        return `
            <section class="mobile-products-section">
                <div class="mobile-products-header">
                    <div class="mobile-products-badge">Masterpieces</div>
                    <h2 class="mobile-products-title">Our Products</h2>
                    <p class="mobile-products-subtitle">Premium glass solutions for every need</p>
                </div>
                
                <div class="mobile-filter-tabs">
                    ${categoriesHTML}
                </div>
                
                <div class="mobile-products-grid">
                    ${productsHTML}
                </div>
            </section>
        `;
    }
    
    // Insert mobile products section
    function insertMobileProducts() {
        console.log('Attempting to insert mobile products section...');
        
        // Try multiple selectors to find insertion point
        let insertionPoint = document.querySelector('.products-section-header');
        
        if (!insertionPoint) {
            // Try alternative selectors
            insertionPoint = document.querySelector('.products-section');
        }
        
        if (!insertionPoint) {
            // Try to find any main content area
            insertionPoint = document.querySelector('.main-content');
        }
        
        if (!insertionPoint) {
            // Last resort - insert into main content
            insertionPoint = document.querySelector('main') || document.body;
        }
        
        if (insertionPoint) {
            console.log('Insertion point found:', insertionPoint);
            const mobileHTML = buildMobileProductsHTML();
            
            if (insertionPoint.tagName === 'MAIN' || insertionPoint.tagName === 'BODY') {
                insertionPoint.insertAdjacentHTML('beforeend', mobileHTML);
            } else {
                insertionPoint.insertAdjacentHTML('afterend', mobileHTML);
            }
            
            console.log('Mobile products section inserted successfully');
            
            // Update debug indicator
            if (debugDiv.parentNode) {
                debugDiv.textContent = 'Mobile Products Inserted!';
                debugDiv.style.background = 'green';
            }
            
            // Wait a bit for DOM to settle, then initialize
            setTimeout(() => {
                initializeMobileProducts();
            }, 100);
        } else {
            console.error('Could not find any insertion point for mobile products');
        }
    }
    
    // Initialize mobile products functionality
    function initializeMobileProducts() {
        const tabs = document.querySelectorAll('.mobile-filter-tab');
        const productCards = document.querySelectorAll('.mobile-product-card');
        
        // Tab filtering
        tabs.forEach(tab => {
            tab.addEventListener('click', function() {
                const category = this.dataset.category;
                
                // Update active tab
                tabs.forEach(t => t.classList.remove('active'));
                this.classList.add('active');
                
                // Filter products with animation
                filterMobileProducts(category, productCards);
                
                // Add touch feedback
                this.style.transform = 'scale(0.95)';
                setTimeout(() => {
                    this.style.transform = 'scale(1)';
                }, 150);
            });
        });
        
        // Add touch feedback and interactions to product cards
        productCards.forEach((card, index) => {
            // Touch feedback
            card.addEventListener('touchstart', function() {
                this.style.transform = 'scale(0.98)';
            }, { passive: true });
            
            card.addEventListener('touchend', function() {
                this.style.transform = 'scale(1)';
            }, { passive: true });
            
            // Click handler for future expansion (product details, etc.)
            card.addEventListener('click', function() {
                const productId = this.dataset.product;
                console.log(`Product clicked: ${productId}`);
                
                // Add ripple effect
                this.style.position = 'relative';
                this.style.overflow = 'hidden';
                
                const ripple = document.createElement('span');
                ripple.style.cssText = `
                    position: absolute;
                    border-radius: 50%;
                    transform: scale(0);
                    animation: ripple 0.6s linear;
                    background-color: rgba(3, 139, 185, 0.3);
                    left: 50%;
                    top: 50%;
                    width: 100px;
                    height: 100px;
                    margin-left: -50px;
                    margin-top: -50px;
                `;
                
                this.appendChild(ripple);
                
                setTimeout(() => {
                    ripple.remove();
                }, 600);
            });
            
            // Staggered initial animation
            setTimeout(() => {
                card.classList.add('fade-in');
            }, index * 150);
        });
        
        console.log('Mobile products functionality initialized');
    }
    
    // Filter products with smooth animation
    function filterMobileProducts(category, productCards) {
        productCards.forEach((card, index) => {
            const productCategory = card.dataset.category;
            const shouldShow = category === 'all' || productCategory === category;
            
            if (shouldShow) {
                // Show with staggered animation
                setTimeout(() => {
                    card.classList.remove('hidden');
                    card.style.display = 'flex';
                    
                    // Trigger reflow
                    card.offsetHeight;
                    
                    card.classList.add('fade-in');
                }, index * 100);
            } else {
                // Hide with smooth animation
                card.style.transition = 'all 0.3s ease';
                card.style.opacity = '0';
                card.style.transform = 'translateY(-10px)';
                
                setTimeout(() => {
                    card.classList.add('hidden');
                    card.classList.remove('fade-in');
                }, 300);
            }
        });
        
        console.log(`Filtered products for category: ${category}`);
    }
    
    // Handle window resize
    function handleResize() {
        if (!isMobile()) {
            // Remove mobile section if switching to desktop
            const mobileSection = document.querySelector('.mobile-products-section');
            if (mobileSection) {
                mobileSection.remove();
                console.log('Mobile products section removed on desktop switch');
            }
        } else {
            // Add mobile section if switching to mobile
            const mobileSection = document.querySelector('.mobile-products-section');
            if (!mobileSection) {
                insertMobileProducts();
            }
        }
    }
    
    // Debounced resize handler
    let resizeTimeout;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(handleResize, 250);
    });
    
    // Initialize with retries
    function initWithRetries(attempt = 1) {
        console.log(`Mobile products initialization attempt ${attempt}`);
        
        if (document.querySelector('.mobile-products-section')) {
            console.log('Mobile products section already exists');
            return;
        }
        
        try {
            insertMobileProducts();
        } catch (error) {
            console.error('Error inserting mobile products:', error);
            
            if (attempt < 3) {
                console.log(`Retrying in ${attempt * 500}ms...`);
                setTimeout(() => initWithRetries(attempt + 1), attempt * 500);
            } else {
                console.error('Failed to insert mobile products after 3 attempts');
            }
        }
    }
    
    initWithRetries();
    
    console.log('Mobile products builder completed');
});