// Products Section Entrance Animations
document.addEventListener('DOMContentLoaded', function() {
    console.log('Setting up Products section entrance animations...');
    
    // Find the products section elements
    const masterpiecesTitle = document.querySelector('.products-masterpieces-title');
    const ourProductsTitle = document.querySelector('.products-main-title');
    const descriptionText = document.querySelector('.products-description');
    const productsSection = document.querySelector('.products-section'); // Parent container
    
    // Array of elements to animate
    const animateElements = [
        { element: masterpiecesTitle, name: 'Masterpieces title' },
        { element: ourProductsTitle, name: 'Our Products title' },
        { element: descriptionText, name: 'Description text' }
    ].filter(item => item.element); // Only include found elements
    
    if (animateElements.length === 0) {
        console.log('No products section elements found for animation');
        return;
    }
    
    console.log(`Found ${animateElements.length} elements to animate`);
    
    // Add the animation class to the parent section
    if (productsSection) {
        productsSection.classList.add('products-section-animate');
    }
    
    // Create intersection observer with optimized settings
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            const element = entry.target;
            const intersectionRatio = entry.intersectionRatio;
            
            console.log(`Element ${element.dataset.id} intersection ratio:`, intersectionRatio);
            
            if (entry.isIntersecting && intersectionRatio >= 0.2) {
                console.log(`Triggering entrance animation for element ${element.dataset.id}`);
                element.classList.add('animate-in');
                
                // Add in-view class to parent section
                if (productsSection) {
                    productsSection.classList.add('in-view');
                }
                
                // Unobserve after animation to prevent re-triggering
                observer.unobserve(element);
            }
        });
    }, {
        threshold: [0, 0.1, 0.2, 0.3, 0.4, 0.5], // Multiple thresholds for smooth detection
        rootMargin: '0px 0px -10% 0px' // Trigger when element is 10% from bottom of viewport
    });
    
    // Start observing all elements
    animateElements.forEach(({ element, name }) => {
        console.log(`Starting to observe: ${name}`);
        observer.observe(element);
        
        // Fallback: trigger animation on scroll if intersection observer fails
        let hasTriggered = false;
        window.addEventListener('scroll', () => {
            if (hasTriggered) return;
            
            const rect = element.getBoundingClientRect();
            const windowHeight = window.innerHeight;
            
            // Trigger when element is 20% visible from bottom of viewport
            if (rect.top <= windowHeight * 0.8 && rect.bottom >= 0) {
                console.log(`Fallback scroll trigger for: ${name}`);
                element.classList.add('animate-in');
                
                if (productsSection) {
                    productsSection.classList.add('in-view');
                }
                
                hasTriggered = true;
            }
        }, { passive: true });
    });
    
    // Enhanced performance: preload animations on page load
    setTimeout(() => {
        animateElements.forEach(({ element }) => {
            element.style.willChange = 'transform, opacity';
        });
        
        // Clean up will-change after animations complete
        setTimeout(() => {
            animateElements.forEach(({ element }) => {
                element.style.willChange = 'auto';
            });
        }, 2000);
    }, 100);
    
    console.log('Products section entrance animations initialized successfully');
});