// Who We Are Section Fade Animation & Modern Video Player
document.addEventListener('DOMContentLoaded', function() {
    console.log('Setting up Who We Are fade animation...');
    
    // Find the Who We Are section
    const whoWeAreSection = document.querySelector('.who-we-are-section');
    
    if (whoWeAreSection) {
        console.log('Found Who We Are section');
        
        // Create intersection observer with smoother thresholds
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                const intersectionRatio = entry.intersectionRatio;
                console.log('Intersection ratio:', intersectionRatio);
                
                if (entry.isIntersecting && intersectionRatio >= 0.1) {
                    console.log('Who We Are section is visible, triggering fade-in');
                    entry.target.classList.add('fade-in-visible');
                } else if (!entry.isIntersecting || intersectionRatio < 0.05) {
                    console.log('Who We Are section is not visible enough, removing fade-in');
                    entry.target.classList.remove('fade-in-visible');
                }
            });
        }, {
            threshold: [0, 0.05, 0.1, 0.15, 0.2, 0.25], // Multiple thresholds for smoother detection
            rootMargin: '50px 0px -50px 0px' // More generous margins
        });
        
        // Start observing the section
        observer.observe(whoWeAreSection);
    } else {
        console.log('Who We Are section not found');
    }
    
    // Modern Video Player Setup
    console.log('Setting up modern video player...');
    
    const videoWidget = document.querySelector('.video-player-section');
    if (videoWidget) {
        console.log('Found video widget');
        
        // Add loading state initially
        videoWidget.classList.add('video-loading');
        
        // Wait for video to be embedded
        setTimeout(() => {
            const iframe = videoWidget.querySelector('iframe');
            if (iframe) {
                console.log('Video iframe found, setting up modern player');
                
                // Enable autoplay and mute for YouTube videos
                const src = iframe.src;
                if (src.includes('youtube.com') || src.includes('youtu.be')) {
                    // Add comprehensive parameters to hide all YouTube controls and branding
                    let newSrc = src;
                    const params = [
                        'autoplay=1',
                        'mute=1', 
                        'controls=0',
                        'showinfo=0',
                        'rel=0',
                        'modestbranding=1',
                        'iv_load_policy=3',
                        'fs=0',
                        'disablekb=1',
                        'playsinline=1'
                    ];
                    
                    params.forEach(param => {
                        const [key] = param.split('=');
                        if (!newSrc.includes(`${key}=`)) {
                            const separator = newSrc.includes('?') ? '&' : '?';
                            newSrc += separator + param;
                        }
                    });
                    
                    console.log('Updated video src:', newSrc);
                    iframe.src = newSrc;
                }
                
                // Create custom controls
                createCustomVideoControls(videoWidget);
                
                // Remove loading state when video is ready
                iframe.addEventListener('load', () => {
                    setTimeout(() => {
                        videoWidget.classList.remove('video-loading');
                        videoWidget.classList.add('video-loaded');
                        
                        // Aggressively hide YouTube controls after load
                        hideYouTubeControls(iframe);
                        
                        console.log('Video loaded successfully');
                    }, 1000);
                });
                
            } else {
                console.log('No iframe found, checking for embedded video...');
                // Alternative check for other video types
                setTimeout(() => {
                    videoWidget.classList.remove('video-loading');
                    videoWidget.classList.add('video-loaded');
                }, 2000);
            }
        }, 1500);
    
    function createCustomVideoControls(videoWidget) {
        console.log('Creating custom video controls');
        
        const controlsHTML = `
            <div class="custom-video-controls">
                <div class="video-controls-row">
                    <div class="video-play-btn" title="Play/Pause"></div>
                    <div class="video-progress" title="Seek">
                        <div class="video-progress-bar"></div>
                    </div>
                    <div class="video-time">0:00 / 0:00</div>
                    <div class="video-volume-btn" title="Mute/Unmute"></div>
                    <div class="video-fullscreen-btn" title="Fullscreen"></div>
                </div>
            </div>
        `;
        
        videoWidget.insertAdjacentHTML('beforeend', controlsHTML);
        
        // Get control elements
        const playBtn = videoWidget.querySelector('.video-play-btn');
        const volumeBtn = videoWidget.querySelector('.video-volume-btn');
        const fullscreenBtn = videoWidget.querySelector('.video-fullscreen-btn');
        const progressBar = videoWidget.querySelector('.video-progress');
        const timeDisplay = videoWidget.querySelector('.video-time');
        
        let isPlaying = true; // Start as playing since autoplay is on
        let isMuted = true; // Start as muted
        
        // Play/Pause functionality
        if (playBtn) {
            playBtn.addEventListener('click', () => {
                const iframe = videoWidget.querySelector('iframe');
                if (iframe) {
                    if (isPlaying) {
                        // Pause video (YouTube API would be needed for full control)
                        playBtn.classList.remove('playing');
                        isPlaying = false;
                        console.log('Video paused');
                    } else {
                        // Play video
                        playBtn.classList.add('playing');
                        isPlaying = true;
                        console.log('Video playing');
                    }
                }
            });
            
            // Set initial state
            playBtn.classList.add('playing');
        }
        
        // Volume control
        if (volumeBtn) {
            volumeBtn.addEventListener('click', () => {
                if (isMuted) {
                    volumeBtn.classList.add('unmuted');
                    isMuted = false;
                    console.log('Video unmuted');
                } else {
                    volumeBtn.classList.remove('unmuted');
                    isMuted = true;
                    console.log('Video muted');
                }
            });
        }
        
        // Fullscreen control
        if (fullscreenBtn) {
            fullscreenBtn.addEventListener('click', () => {
                if (videoWidget.requestFullscreen) {
                    videoWidget.requestFullscreen();
                } else if (videoWidget.webkitRequestFullscreen) {
                    videoWidget.webkitRequestFullscreen();
                } else if (videoWidget.msRequestFullscreen) {
                    videoWidget.msRequestFullscreen();
                }
                console.log('Fullscreen requested');
            });
        }
        
        // Progress bar click
        if (progressBar) {
            progressBar.addEventListener('click', (e) => {
                const rect = progressBar.getBoundingClientRect();
                const clickX = e.clientX - rect.left;
                const percentage = (clickX / rect.width) * 100;
                
                // Update progress bar visual
                const progressFill = progressBar.querySelector('::after');
                console.log(`Seeking to ${percentage}%`);
                
                // YouTube API would be needed for actual seeking
            });
        }
        
        console.log('Custom controls created successfully');
    }
    
    function hideYouTubeControls(iframe) {
        console.log('Aggressively hiding YouTube controls');
        
        // Add CSS to hide YouTube controls dynamically
        const hideControlsCSS = `
            <style>
                .ytp-chrome-bottom,
                .ytp-chrome-top,
                .ytp-chrome-controls,
                .ytp-gradient-bottom,
                .ytp-gradient-top,
                .ytp-progress-bar-container,
                .ytp-pause-overlay,
                .ytp-watermark,
                .ytp-contextmenu,
                .ytp-share-panel,
                .ytp-endscreen-element,
                .ytp-ce-element,
                .html5-endscreen,
                .videowall-endscreen {
                    display: none !important;
                    opacity: 0 !important;
                    visibility: hidden !important;
                    pointer-events: none !important;
                }
                
                .html5-video-player {
                    background: transparent !important;
                }
                
                .html5-video-container {
                    background: transparent !important;
                }
            </style>
        `;
        
        try {
            // Try to access iframe content and inject CSS
            if (iframe.contentDocument) {
                const head = iframe.contentDocument.head;
                if (head) {
                    head.insertAdjacentHTML('beforeend', hideControlsCSS);
                    console.log('Injected hide controls CSS into iframe');
                }
            }
        } catch (error) {
            console.log('Cannot access iframe content due to CORS, using alternative method');
        }
        
        // Alternative: Use mutation observer to detect and hide controls
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.type === 'childList') {
                    mutation.addedNodes.forEach((node) => {
                        if (node.nodeType === 1) { // Element node
                            // Hide YouTube control elements
                            const controlSelectors = [
                                '.ytp-chrome-bottom',
                                '.ytp-chrome-top', 
                                '.ytp-chrome-controls',
                                '.ytp-pause-overlay',
                                '.ytp-contextmenu'
                            ];
                            
                            controlSelectors.forEach(selector => {
                                const elements = node.querySelectorAll ? node.querySelectorAll(selector) : [];
                                elements.forEach(el => {
                                    el.style.display = 'none';
                                    el.style.opacity = '0';
                                    el.style.visibility = 'hidden';
                                });
                                
                                if (node.matches && node.matches(selector)) {
                                    node.style.display = 'none';
                                    node.style.opacity = '0';
                                    node.style.visibility = 'hidden';
                                }
                            });
                        }
                    });
                }
            });
        });
        
        // Start observing the video widget for changes
        observer.observe(videoWidget, {
            childList: true,
            subtree: true
        });
        
        console.log('Set up mutation observer to hide YouTube controls');
    }
    
    // UFG Counters Animation Setup
    console.log('Setting up UFG counter animations...');
    
    const counterSection = document.querySelector('.stats-counter-section');
    if (counterSection) {
        console.log('Found UFG counter section');
        
        // Counter elements with their target values (no suffix needed - Elementor handles it)
        const counters = [
            { 
                element: document.querySelector('.counter-years'), 
                number: document.querySelector('.counter-years .counter-number'),
                target: 19, 
                delay: 100 
            },
            { 
                element: document.querySelector('.counter-clients'), 
                number: document.querySelector('.counter-clients .counter-number'),
                target: 250000, 
                delay: 300 
            },
            { 
                element: document.querySelector('.counter-countries'), 
                number: document.querySelector('.counter-countries .counter-number'),
                target: 30, 
                delay: 500 
            },
            { 
                element: document.querySelector('.counter-projects'), 
                number: document.querySelector('.counter-projects .counter-number'),
                target: 30, 
                delay: 700 
            }
        ];
        
        // Intersection observer for the counter section
        const counterObserver = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    console.log('UFG counter section is visible, starting animations');
                    entry.target.classList.add('section-visible');
                    
                    // Animate each counter with staggered timing
                    counters.forEach((counter, index) => {
                        if (counter.element && counter.number) {
                            setTimeout(() => {
                                console.log(`Starting counter ${index + 1} animation`);
                                counter.element.classList.add('counter-visible');
                                
                                // Start the number counting animation with slower speed
                                setTimeout(() => {
                                    animateSlowCounter(counter.number, counter.target);
                                }, 500); // Wait for fade-in to start
                                
                            }, counter.delay);
                        }
                    });
                    
                } else {
                    console.log('UFG counter section is not visible, removing animations');
                    entry.target.classList.remove('section-visible');
                    
                    // Reset counters
                    counters.forEach(counter => {
                        if (counter.element && counter.number) {
                            counter.element.classList.remove('counter-visible');
                            counter.number.textContent = '0';
                            counter.number.removeAttribute('data-animated');
                        }
                    });
                }
            });
        }, {
            threshold: 0.3,
            rootMargin: '0px 0px -100px 0px'
        });
        
        counterObserver.observe(counterSection);
        
    } else {
        console.log('UFG counter section not found');
    }
    
    // Slow counter animation function (no suffix - Elementor handles it)
    function animateSlowCounter(element, target) {
        if (element.hasAttribute('data-animated')) {
            return; // Already animated
        }
        
        console.log(`Animating counter to ${target}`);
        element.setAttribute('data-animated', 'true');
        
        let current = 0;
        const duration = 4000; // Slower: 4 seconds instead of 2
        const startTime = performance.now();
        const useCommas = target >= 1000;
        
        function updateCounter(timestamp) {
            const elapsed = timestamp - startTime;
            const progress = Math.min(elapsed / duration, 1);
            
            // Slower easing function for more dramatic effect
            const eased = 1 - Math.pow(1 - progress, 4); // Even slower curve
            current = Math.floor(target * eased);
            
            // Format the number (no suffix - Elementor handles it separately)
            let displayValue = current;
            if (useCommas && current >= 1000) {
                displayValue = current.toLocaleString();
            }
            
            element.textContent = displayValue;
            
            if (progress < 1) {
                requestAnimationFrame(updateCounter);
            } else {
                // Final value (no suffix)
                let finalValue = target;
                if (useCommas && target >= 1000) {
                    finalValue = target.toLocaleString();
                }
                element.textContent = finalValue;
                console.log(`Counter animation completed: ${finalValue}`);
            }
        }
        
        // Start the animation (no suffix)
        element.textContent = '0';
        requestAnimationFrame(updateCounter);
    }
        
        // Background video autoplay setup
        const bgVideoContainer = document.querySelector('.hero-section .background-video-container');
        if (bgVideoContainer) {
            console.log('Found background video container');
            
            // Ensure background video autoplays
            setTimeout(() => {
                const bgIframe = bgVideoContainer.querySelector('iframe');
                if (bgIframe) {
                    const bgSrc = bgIframe.src;
                    if (bgSrc.includes('youtube.com') || bgSrc.includes('youtu.be')) {
                        if (!bgSrc.includes('autoplay=1')) {
                            const separator = bgSrc.includes('?') ? '&' : '?';
                            bgIframe.src = bgSrc + separator + 'autoplay=1&mute=1&loop=1';
                        }
                    }
                }
            }, 1000);
        }
        
    } else {
        console.log('Video widget not found');
    }
});