@extends('layouts.admin-simple')

@section('title', 'Division Deliveries Report')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.index') }}">Reports</a></li>
    <li class="breadcrumb-item active">Division Deliveries</li>
@endsection

@section('page-title')
    <div class="page-title-left">
        <h3 class="mb-1">Division Deliveries Report</h3>
        <p class="text-muted mb-0" style="font-size: 13px;">Track total deliveries, current stock, and utilization by division</p>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4 warehouse-stats-row">
        <div class="col-xl-2 col-md-4 mb-3">
            <x-warehouse.stats-card
                title="Total Items"
                :value="$totalItems"
                icon="inventory_2"
                color="primary"
                subtitle="Items tracked"
            />
        </div>

        <div class="col-xl-2 col-md-4 mb-3">
            <x-warehouse.stats-card
                title="Total Delivered"
                :value="number_format($totalDeliveries, 0)"
                icon="local_shipping"
                color="success"
                subtitle="Units delivered"
            />
        </div>

        <div class="col-xl-2 col-md-4 mb-3">
            <x-warehouse.stats-card
                title="Currently Available"
                :value="number_format($totalAvailable, 0)"
                icon="warehouse"
                color="info"
                subtitle="Units in stock"
            />
        </div>

        <div class="col-xl-2 col-md-4 mb-3">
            <x-warehouse.stats-card
                title="Total Issued"
                :value="number_format($totalIssued, 0)"
                icon="output"
                color="warning"
                subtitle="Units issued"
            />
        </div>

        <div class="col-xl-2 col-md-4 mb-3">
            <x-warehouse.stats-card
                title="Utilization Rate"
                :value="number_format($utilizationRate, 1) . '%'"
                icon="trending_up"
                color="{{ $utilizationRate >= 70 ? 'success' : ($utilizationRate >= 30 ? 'warning' : 'secondary') }}"
                subtitle="Usage efficiency"
            />
        </div>

        <div class="col-xl-2 col-md-4 mb-3">
            <x-warehouse.stats-card
                title="Total Value"
                :value="number_format($totalValue, 2)"
                icon="attach_money"
                color="info"
                subtitle="Total delivery value"
                :currency="true"
            />
        </div>
    </div>

    <!-- Filters -->
    <div class="card mb-4">
        <div class="card-header">
            <h6 class="card-title mb-0">
                <i class="material-icons mr-2">filter_list</i>
                Filters
            </h6>
        </div>
        <div class="card-body">
            <form method="GET" action="{{ route('warehouse.reports.division-deliveries') }}" id="filtersForm">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group mb-3">
                            <label for="division_id" class="form-label">Division</label>
                            <select name="division_id" id="division_id" class="form-control" data-placeholder="All Divisions">
                                <option value="">All Divisions</option>
                                @foreach($divisions as $division)
                                    <option value="{{ $division->id }}" {{ request('division_id') == $division->id ? 'selected' : '' }}>
                                        {{ $division->division_name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <div class="col-md-3">
                        <div class="form-group mb-3">
                            <label for="category_id" class="form-label">Category</label>
                            <select name="category_id" id="category_id" class="form-control" data-placeholder="All Categories">
                                <option value="">All Categories</option>
                                @foreach($categories as $category)
                                    <option value="{{ $category->id }}" {{ request('category_id') == $category->id ? 'selected' : '' }}>
                                        {{ $category->category_name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    </div>

                    <div class="col-md-4">
                        <div class="form-group mb-3">
                            <label for="search" class="form-label">Search Items</label>
                            <input type="text" name="search" id="search" class="form-control"
                                   placeholder="Search by item code or description..."
                                   value="{{ request('search') }}">
                        </div>
                    </div>

                    <div class="col-md-2">
                        <div class="form-group mb-3">
                            <label class="form-label">&nbsp;</label>
                            <div>
                                <button type="submit" class="btn btn-primary btn-sm">
                                    <i class="material-icons mr-1" style="font-size: 16px;">search</i>
                                    Filter
                                </button>
                                <a href="{{ route('warehouse.reports.division-deliveries') }}" class="btn btn-outline-secondary btn-sm ml-1">
                                    <i class="material-icons mr-1" style="font-size: 16px;">clear</i>
                                    Clear
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Export Buttons -->
    <div class="row warehouse-export-buttons">
        <div class="col-12 text-right">
            <button class="btn btn-success warehouse-export-btn" data-export="excel">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">download</i>
                Export Excel
            </button>
            <button class="btn btn-info warehouse-export-btn" data-export="pdf">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">picture_as_pdf</i>
                Export PDF
            </button>
            <button class="btn btn-secondary warehouse-export-btn" data-action="print">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">print</i>
                Print
            </button>
        </div>
    </div>

    <!-- Data Table -->
    <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h6 class="card-title mb-0">
                <i class="material-icons mr-2">table_chart</i>
                Division Deliveries
                @if(request()->hasAny(['division_id', 'category_id', 'search']))
                    <small class="text-muted">(Filtered Results)</small>
                @endif
            </h6>

            <div class="card-tools">
                <button type="button" class="btn btn-sm btn-outline-primary" id="refreshTable">
                    <i class="material-icons mr-1" style="font-size: 16px;">refresh</i>
                    Refresh
                </button>
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-sm table-striped" id="deliveriesTable">
                    <thead class="thead-light">
                        <tr>
                            <th width="5%">#</th>
                            <th width="12%">Item Code</th>
                            <th width="20%">Description</th>
                            <th width="10%">Division</th>
                            <th width="10%">Total Delivered</th>
                            <th width="10%">Available</th>
                            <th width="10%">Total Issued</th>
                            <th width="10%">Total Returned</th>
                            <th width="10%">Total Value</th>
                            <th width="8%">Last Delivery</th>
                            <th width="8%">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- DataTable will populate this -->
                    </tbody>
                </table>
            </div>
        </div>
    </div>
@endsection

@section('modals')
    <!-- Project Details Modal -->
    <div class="modal fade" id="projectDetailsModal" tabindex="-1" role="dialog" aria-labelledby="projectDetailsModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="projectDetailsModalLabel">
                        <i class="material-icons mr-2">assessment</i>
                        Project-wise Item Details
                    </h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="row mb-3">
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Item Information</label>
                            <div class="form-control-plaintext border rounded p-2 bg-light">
                                <strong id="modal-item-code" class="text-primary"></strong> - <span id="modal-item-description"></span>
                                <br><small class="text-muted">Unit: <span id="modal-item-unit"></span></small>
                            </div>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label class="form-label">Division</label>
                            <div class="form-control-plaintext border rounded p-2 bg-light">
                                <span class="badge badge-primary" id="modal-division-name"></span>
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-12">
                            <label class="form-label">Project-wise Breakdown</label>
                            <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
                                <table class="table table-sm table-hover">
                                    <thead class="thead-light">
                                        <tr>
                                            <th style="min-width: 150px;">
                                                <i class="material-icons mr-1" style="font-size: 16px;">folder</i>
                                                Project
                                            </th>
                                            <th style="min-width: 80px;">
                                                <i class="material-icons mr-1" style="font-size: 16px;">publish</i>
                                                Issued
                                            </th>
                                            <th style="min-width: 80px;">
                                                <i class="material-icons mr-1" style="font-size: 16px;">undo</i>
                                                Returned
                                            </th>
                                            <th style="min-width: 80px;">
                                                <i class="material-icons mr-1" style="font-size: 16px;">account_balance</i>
                                                Balance
                                            </th>
                                            <th style="min-width: 200px;">
                                                <i class="material-icons mr-1" style="font-size: 16px;">history</i>
                                                Issue History
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody id="modal-projects-tbody">
                                        <!-- Will be populated by JavaScript -->
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">
                        <i class="material-icons mr-1" style="font-size: 16px;">cancel</i>
                        Close
                    </button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
<link href="{{ asset('admin-assets/css/warehouse-reports.css') }}" rel="stylesheet" />
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
<link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.min.css" rel="stylesheet" />
<style>
.select2-container--bootstrap-5 .select2-selection {
    min-height: 38px;
    border: 1px solid #ced4da;
}
.select2-container--bootstrap-5 .select2-selection--single .select2-selection__rendered {
    line-height: 36px;
}
.select2-container--bootstrap-5 .select2-selection--single .select2-selection__arrow {
    height: 36px;
}
.select2-container {
    width: 100% !important;
}
.warehouse-stats-row .card {
    border: none;
    box-shadow: 0 0 10px rgba(0,0,0,0.1);
}

.warehouse-stats-row .stats-card {
    transition: transform 0.2s;
}

.warehouse-stats-row .stats-card:hover {
    transform: translateY(-2px);
}

#deliveriesTable th {
    background-color: #f8f9fa;
    font-weight: 600;
    font-size: 0.875rem;
    border-bottom: 2px solid #e9ecef;
}

#deliveriesTable td {
    font-size: 0.875rem;
    vertical-align: middle;
}

.badge {
    font-size: 0.75rem;
}

.badge-outline-primary {
    color: #007bff;
    background-color: transparent;
    border: 1px solid #007bff;
}

.material-icons.text-xs {
    font-size: 14px;
}

.warehouse-export-buttons {
    margin-bottom: 20px;
}

.warehouse-export-btn {
    margin-left: 5px;
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script src="{{ asset('admin-assets/js/warehouse-reports.js') }}"></script>
<script>
$(document).ready(function() {
    // Initialize Select2 for filter dropdowns
    $('#division_id, #category_id').select2({
        theme: 'bootstrap-5',
        allowClear: true,
        placeholder: function() {
            return $(this).data('placeholder') || 'Select...';
        }
    });

    let table = $('#deliveriesTable').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: "{{ route('warehouse.reports.division-deliveries') }}",
            data: function(d) {
                d.division_id = $('#division_id').val();
                d.category_id = $('#category_id').val();
                d.search = $('#search').val();
            }
        },
        columns: [
            {data: 'DT_RowIndex', name: 'DT_RowIndex', orderable: false, searchable: false},
            {data: 'item_code', name: 'item.item_code'},
            {data: 'item_description', name: 'item.item_description'},
            {data: 'division_name', name: 'division.division_name'},
            {data: 'total_delivered', name: 'total_quantity_delivered'},
            {data: 'current_available', name: 'current_available_quantity'},
            {data: 'total_issued', name: 'total_quantity_issued'},
            {data: 'total_returned', name: 'total_returned', orderable: false},
            {data: 'total_value', name: 'total_value'},
            {data: 'last_delivery', name: 'last_delivery_date'},
            {data: 'actions', name: 'actions', orderable: false, searchable: false}
        ],
        order: [[4, 'desc']], // Order by total delivered
        pageLength: 25,
        responsive: true,
        language: {
            emptyTable: "No division deliveries found"
        }
    });

    // Refresh table
    $('#refreshTable').click(function() {
        table.ajax.reload();
    });

    // Auto-submit form on filter change
    $('#division_id, #category_id').change(function() {
        $('#filtersForm').submit();
    });

    // Handle view details button click
    $(document).on('click', '.view-details-btn', function() {
        const divisionId = $(this).data('division-id');
        const itemId = $(this).data('item-id');
        const itemCode = $(this).data('item-code');
        const itemDescription = $(this).data('item-description');
        const divisionName = $(this).data('division-name');

        // Update modal header info
        $('#modal-item-code').text(itemCode);
        $('#modal-item-description').text(itemDescription);
        $('#modal-division-name').text(divisionName);

        // Show loading state
        $('#modal-projects-tbody').html('<tr><td colspan="5" class="text-center py-4"><div class="spinner-border spinner-border-sm text-primary mr-2" role="status"></div> Loading project details...</td></tr>');

        // Show modal
        $('#projectDetailsModal').modal('show');

        // Fetch project details
        $.get('{{ route("warehouse.reports.division-item-details") }}', {
            division_id: divisionId,
            item_id: itemId
        })
        .done(function(response) {
            if (response.success && response.data) {
                // Update item unit information
                if (response.data.item && response.data.item.unit) {
                    $('#modal-item-unit').text(response.data.item.unit);
                }

                let tbody = '';

                if (!response.data.projects || response.data.projects.length === 0) {
                    tbody = `<tr>
                        <td colspan="5" class="text-center py-4">
                            <i class="material-icons text-muted mb-2" style="font-size: 48px;">folder_open</i>
                            <p class="text-muted mb-0">No project issues found for this item in this division.</p>
                        </td>
                    </tr>`;
                } else {
                    response.data.projects.forEach(function(project) {
                        const balanceClass = project.current_balance > 0 ? 'text-success' :
                                           project.current_balance < 0 ? 'text-danger' : 'text-muted';

                        const balanceIcon = project.current_balance > 0 ? 'trending_up' :
                                          project.current_balance < 0 ? 'trending_down' : 'trending_flat';

                        // Build issue history with better styling
                        let issueHistory = '';
                        project.issues.forEach(function(issue) {
                            const typeIcon = issue.type === 'Material Request' ? 'assignment' : 'local_shipping';
                            const typeClass = issue.type === 'Material Request' ? 'badge-primary' : 'badge-success';

                            issueHistory += `<div class="mb-1">
                                <span class="badge ${typeClass} mr-1">
                                    <i class="material-icons" style="font-size: 12px;">${typeIcon}</i>
                                    ${issue.type}
                                </span>
                                <strong class="text-info">${issue.quantity}</strong>
                                <small class="text-muted d-block">${issue.reference} - ${issue.date}</small>
                            </div>`;
                        });

                        tbody += `<tr>
                            <td>
                                <div class="d-flex align-items-center">
                                    <i class="material-icons text-primary mr-2">folder</i>
                                    <div>
                                        <strong class="text-dark">${project.project_name}</strong>
                                        <br><small class="text-muted">${project.project_number}</small>
                                    </div>
                                </div>
                            </td>
                            <td>
                                <span class="badge badge-outline-primary px-2 py-1">
                                    ${parseFloat(project.total_issued).toFixed(2)}
                                </span>
                            </td>
                            <td>
                                <span class="badge badge-outline-success px-2 py-1">
                                    ${parseFloat(project.total_returned).toFixed(2)}
                                </span>
                            </td>
                            <td>
                                <div class="d-flex align-items-center ${balanceClass}">
                                    <i class="material-icons mr-1" style="font-size: 16px;">${balanceIcon}</i>
                                    <strong>${parseFloat(project.current_balance).toFixed(2)}</strong>
                                </div>
                            </td>
                            <td style="max-width: 300px;">
                                <div style="max-height: 150px; overflow-y: auto;">
                                    ${issueHistory || '<span class="text-muted">No issue history</span>'}
                                </div>
                            </td>
                        </tr>`;
                    });
                }

                $('#modal-projects-tbody').html(tbody);
            } else {
                $('#modal-projects-tbody').html(`<tr>
                    <td colspan="5" class="text-center py-4">
                        <i class="material-icons text-danger mb-2" style="font-size: 48px;">error</i>
                        <p class="text-danger mb-0">Error loading project details.</p>
                    </td>
                </tr>`);
            }
        })
        .fail(function() {
            $('#modal-projects-tbody').html(`<tr>
                <td colspan="5" class="text-center py-4">
                    <i class="material-icons text-danger mb-2" style="font-size: 48px;">wifi_off</i>
                    <p class="text-danger mb-0">Failed to load project details. Please try again.</p>
                </td>
            </tr>`);
        });
    });
});
</script>
@endpush