@extends('layouts.admin-simple')

@section('title', 'Material Request Report')

@section('breadcrumb')
    <li class="breadcrumb-item">Warehouse</li>
    <li class="breadcrumb-item"><a href="{{ route('warehouse.reports.index') }}">Reports</a></li>
    <li class="breadcrumb-item active">Material Request</li>
@endsection

@section('page-title')
    <div class="page-title-left">
        <h3 class="mb-1">Material Request Report</h3>
        <p class="text-muted mb-0" style="font-size: 13px;">Material request tracking and fulfillment status</p>
    </div>
@endsection

@section('content')
    <!-- Statistics Cards -->
    <div class="row mb-4 warehouse-stats-row">
        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Requests"
                :value="$outgoingRequests->total()"
                icon="assignment"
                color="primary"
                subtitle="Material requests created"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Requested"
                :value="number_format($outgoingRequests->getCollection()->sum('total_requested'), 2)"
                icon="inventory_2"
                color="info"
                subtitle="Total requested from warehouse"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Total Released"
                :value="number_format($outgoingRequests->getCollection()->sum('total_released'), 2)"
                icon="outbound"
                color="success"
                subtitle="Total fulfilled from requests"
            />
        </div>

        <div class="col-xl-3 col-md-6 mb-3">
            <x-warehouse.stats-card
                title="Pending Balance"
                :value="number_format($outgoingRequests->getCollection()->sum('total_balance'), 2)"
                icon="pending_actions"
                color="warning"
                subtitle="Total pending fulfillment"
            />
        </div>
    </div>

    <!-- Filters Panel -->
    <div class="card mb-4 warehouse-filters-panel">
        <div class="card-header">
            <h5 class="mb-0">
                <i class="material-icons align-middle mr-2">filter_list</i>
                Filters
            </h5>
        </div>
        <div class="card-body">
            <form id="filterForm" class="warehouse-filter-form" method="GET" action="{{ route('warehouse.reports.outgoing') }}">
                <div class="row">
                    <div class="col-md-2 mb-3">
                        <label class="form-label">Search</label>
                        <input type="text" name="search" value="{{ request('search') }}" class="form-control" 
                               placeholder="MR number, project">
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Project</label>
                        <select name="project_id" class="form-control warehouse-select2" data-placeholder="All Projects">
                            <option value="">All Projects</option>
                            @foreach($projects as $project)
                                <option value="{{ $project->id }}" {{ request('project_id') == $project->id ? 'selected' : '' }}>
                                    {{ $project->project_name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Division</label>
                        <select name="division_id" class="form-control warehouse-select2" data-placeholder="All Divisions">
                            <option value="">All Divisions</option>
                            @foreach($divisions as $division)
                                <option value="{{ $division->id }}" {{ request('division_id') == $division->id ? 'selected' : '' }}>
                                    {{ $division->division_name }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-control warehouse-select2" data-placeholder="All Statuses">
                            <option value="">All Statuses</option>
                            @foreach($statuses as $status)
                                <option value="{{ $status }}" {{ request('status') == $status ? 'selected' : '' }}>
                                    {{ ucfirst(str_replace('_', ' ', $status)) }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Fulfillment</label>
                        <select name="fulfillment_status" class="form-control warehouse-select2" data-placeholder="All Fulfillment">
                            <option value="">All Fulfillment</option>
                            @foreach($fulfillmentStatuses as $fulfillmentStatus)
                                <option value="{{ $fulfillmentStatus }}" {{ request('fulfillment_status') == $fulfillmentStatus ? 'selected' : '' }}>
                                    {{ ucfirst($fulfillmentStatus) }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">&nbsp;</label>
                        <button type="submit" class="btn btn-primary btn-block">
                            <i class="material-icons" style="font-size: 18px; vertical-align: middle;">search</i>
                            Apply
                        </button>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-2 mb-3">
                        <label class="form-label">Date From</label>
                        <input type="date" name="date_from" value="{{ request('date_from') }}" class="form-control warehouse-datepicker">
                    </div>

                    <div class="col-md-2 mb-3">
                        <label class="form-label">Date To</label>
                        <input type="date" name="date_to" value="{{ request('date_to') }}" class="form-control warehouse-datepicker">
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Export Buttons -->
    <div class="row warehouse-export-buttons">
        <div class="col-12 text-right">
            <button class="btn btn-success warehouse-export-btn" data-export="excel">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">download</i>
                Export Excel
            </button>
            <button class="btn btn-info warehouse-export-btn" data-export="pdf">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">picture_as_pdf</i>
                Export PDF
            </button>
            <button class="btn btn-secondary warehouse-export-btn" data-action="print">
                <i class="material-icons" style="font-size: 18px; vertical-align: middle;">print</i>
                Print
            </button>
        </div>
    </div>

    <!-- Material Request Report Table -->
    <div class="card">
        <div class="card-header warehouse-report-header">
            <h5 class="mb-0 warehouse-report-title">
                <i class="material-icons align-middle mr-2">assignment_turned_in</i>
                Material Request Report ({{ $outgoingRequests->total() }} total requests)
            </h5>
        </div>
        <div class="card-body p-0">
            <div class="warehouse-table-container">
                <!-- Loading Overlay -->
                <div class="warehouse-loading-overlay" id="tableLoadingOverlay">
                    <div class="warehouse-loading-content">
                        <div class="text-center">
                            <div class="spinner-border text-primary" role="status">
                                <span class="sr-only">Loading...</span>
                            </div>
                            <div class="mt-2">Loading request data...</div>
                        </div>
                    </div>
                </div>

                <!-- Table -->
                <div class="warehouse-table-wrapper">
                    <table class="table table-hover mb-0 warehouse-table">
                        <thead class="thead-light">
                            <tr>
                                <th style="min-width: 150px;">Request Number</th>
                                <th style="min-width: 100px;">Request Date</th>
                                <th style="min-width: 180px;">Project</th>
                                <th style="min-width: 120px;">Division</th>
                                <th style="min-width: 150px;">Requested By</th>
                                <th style="min-width: 80px;" class="text-center">Items</th>
                                <th style="min-width: 100px;" class="text-right">Requested Qty</th>
                                <th style="min-width: 100px;" class="text-right">Released Qty</th>
                                <th style="min-width: 100px;" class="text-right">Pending Qty</th>
                                <th style="min-width: 120px;">Dispatch Methods</th>
                                <th style="min-width: 120px;" class="text-center">Fulfillment Status</th>
                                <th style="min-width: 80px;" class="text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($outgoingRequests as $request)
                                <tr>
                                    <td class="warehouse-cell-nowrap">
                                        {{ $request->material_request_number ?: 'MR-' . str_pad($request->id, 6, '0', STR_PAD_LEFT) }}
                                    </td>
                                    <td class="warehouse-cell-date">
                                        {{ $request->transaction_date->format('d M Y') }}
                                    </td>
                                    <td>{{ Str::limit($request->project->project_name ?? 'No Project', 25) }}</td>
                                    <td>
                                        <span class="badge warehouse-badge warehouse-status-secondary">
                                            {{ $request->division->division_name ??
                                               ($request->project->projectDivision->division_name ?? 'Main') }}
                                        </span>
                                    </td>
                                    <td>{{ $request->requester->name ?? 'Not Specified' }}</td>
                                    <td class="text-center">
                                        <span class="badge warehouse-badge warehouse-status-info">{{ $request->items_count }} items</span>
                                    </td>
                                    <td class="text-right warehouse-cell-number">
                                        <strong>{{ number_format($request->total_requested, 2) }}</strong>
                                    </td>
                                    <td class="text-right warehouse-cell-number">
                                        <strong class="text-success">{{ number_format($request->total_released, 2) }}</strong>
                                    </td>
                                    <td class="text-right warehouse-cell-number">
                                        <strong class="{{ $request->total_balance > 0 ? 'text-warning' : 'text-success' }}">
                                            {{ number_format($request->total_balance, 2) }}
                                        </strong>
                                    </td>
                                    <td>
                                        <small>{{ $request->dispatch_methods_text ?? 'Pending' }}</small>
                                    </td>
                                    <td class="warehouse-cell-status">
                                        @if($request->fulfillment_status === 'completed')
                                            <span class="badge warehouse-badge warehouse-status-success">Completed</span>
                                        @elseif($request->fulfillment_status === 'partial')
                                            <span class="badge warehouse-badge warehouse-status-warning">Partial</span>
                                        @else
                                            <span class="badge warehouse-badge warehouse-status-info">Pending</span>
                                        @endif
                                    </td>
                                    <td class="text-center">
                                        <a href="{{ route('warehouse.reports.outgoing.request-details', $request->id) }}" 
                                           class="btn btn-sm btn-outline-primary warehouse-action-btn">
                                            <i class="material-icons">visibility</i>
                                        </a>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="12" class="warehouse-empty-state">
                                        <i class="material-icons">assignment</i>
                                        <h5>No Material Requests Found</h5>
                                        <p class="text-muted">No requests match your current filters.</p>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        @if($outgoingRequests->hasPages())
            <div class="card-footer">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        Showing {{ $outgoingRequests->firstItem() }} to {{ $outgoingRequests->lastItem() }} 
                        of {{ $outgoingRequests->total() }} requests
                    </div>
                    <div>
                        <nav class="warehouse-pagination">
                            {{ $outgoingRequests->links('pagination::bootstrap-4') }}
                        </nav>
                    </div>
                </div>
            </div>
        @endif
    </div>
@endsection

@push('styles')
    <link href="{{ asset('admin-assets/css/warehouse-reports.css') }}" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.min.css" rel="stylesheet" />
    <style>
        .select2-container--bootstrap-5 .select2-selection {
            min-height: 38px;
            border: 1px solid #ced4da;
        }
        .select2-container--bootstrap-5 .select2-selection--single .select2-selection__rendered {
            line-height: 36px;
        }
        .select2-container--bootstrap-5 .select2-selection--single .select2-selection__arrow {
            height: 36px;
        }
        .select2-container {
            width: 100% !important;
        }
    </style>
@endpush

@push('scripts')
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script src="{{ asset('admin-assets/js/warehouse-reports.js') }}"></script>
    <script>
        $(document).ready(function() {
            // Initialize Select2 for filter dropdowns
            $('.warehouse-select2').select2({
                theme: 'bootstrap-5',
                allowClear: true,
                placeholder: function() {
                    return $(this).data('placeholder') || 'Select...';
                }
            });
        });
    </script>
@endpush