@extends('layouts.admin-simple')

@section('title', 'Material Transfer Requests')


@push('styles')
<style>
/* Adjust Select2 dropdown widths for better proportions */
.select2-container {
    width: 100% !important;
}

/* Make dropdowns auto-width based on content */
.select2-dropdown {
    min-width: 200px;
    max-width: 400px;
    width: auto !important;
}

/* Ensure the selection box doesn't get too wide */
.select2-container .select2-selection--single {
    min-width: 200px;
    max-width: 100%;
}

/* Allow dropdown to expand based on content */
.select2-container--open .select2-dropdown--below {
    width: auto !important;
    min-width: 200px;
}
</style>
@endpush

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="card-title mb-0">
                        <i class="material-icons mr-2">swap_horiz</i>
                        Material Transfer Requests
                    </h4>
                    @can('warehouse.outgoing.create')
                    <a href="{{ route('warehouse.transfer-requests.create') }}" class="btn btn-primary">
                        <i class="material-icons mr-1">add</i>
                        New Transfer Request
                    </a>
                    @endcan
                </div>

                <div class="card-body">
                    <!-- Filters -->
                    <div class="row mb-4">
                        <div class="col-md-3">
                            <x-warehouse.select2-dropdown
                                name="status"
                                id="status_filter"
                                label="Status"
                                placeholder="All Status"
                                :options="[
                                    ['value' => '', 'text' => 'All Status'],
                                    ['value' => 'draft', 'text' => 'Draft'],
                                    ['value' => 'pending_transferrer_approval', 'text' => 'Pending Transferrer Approval'],
                                    ['value' => 'pending_receiver_approval', 'text' => 'Pending Receiver Approval'],
                                    ['value' => 'approved', 'text' => 'Transferred'],
                                    ['value' => 'rejected', 'text' => 'Rejected'],
                                    ['value' => 'cancelled', 'text' => 'Cancelled']
                                ]"
                                :selected="request('status')"
                                allowClear="true"
                            />
                        </div>
                        <div class="col-md-3">
                            <x-warehouse.select2-dropdown
                                name="transferrer_project_id"
                                id="transferrer_project_filter"
                                label="Transferrer Project"
                                placeholder="All Projects"
                                :options="collect([['value' => '', 'text' => 'All Projects']])->merge($projects->map(function($project) {
                                    return ['value' => $project->id, 'text' => $project->project_name];
                                }))"
                                :selected="request('transferrer_project_id')"
                                allowClear="true"
                            />
                        </div>
                        <div class="col-md-3">
                            <x-warehouse.select2-dropdown
                                name="receiver_project_id"
                                id="receiver_project_filter"
                                label="Receiver Project"
                                placeholder="All Projects"
                                :options="collect([['value' => '', 'text' => 'All Projects']])->merge($projects->map(function($project) {
                                    return ['value' => $project->id, 'text' => $project->project_name];
                                }))"
                                :selected="request('receiver_project_id')"
                                allowClear="true"
                            />
                        </div>
                        <div class="col-md-3">
                            <x-warehouse.select2-dropdown
                                name="priority"
                                id="priority_filter"
                                label="Priority"
                                placeholder="All Priorities"
                                :options="[
                                    ['value' => '', 'text' => 'All Priorities'],
                                    ['value' => 'urgent', 'text' => 'Urgent'],
                                    ['value' => 'high', 'text' => 'High'],
                                    ['value' => 'medium', 'text' => 'Medium'],
                                    ['value' => 'low', 'text' => 'Low']
                                ]"
                                :selected="request('priority')"
                                allowClear="true"
                            />
                        </div>
                    </div>

                    <!-- Date Filters -->
                    <div class="row mb-4">
                        <div class="col-md-4">
                            <label for="date_from">Date From</label>
                            <input type="date" name="date_from" id="date_from" class="form-control" value="{{ request('date_from') }}">
                        </div>
                        <div class="col-md-4">
                            <label for="date_to">Date To</label>
                            <input type="date" name="date_to" id="date_to" class="form-control" value="{{ request('date_to') }}">
                        </div>
                        <div class="col-md-4 d-flex align-items-end">
                            <button type="button" onclick="applyFilters()" class="btn btn-info mr-2">
                                <i class="material-icons mr-1">filter_list</i>
                                Apply Filters
                            </button>
                            <a href="{{ route('warehouse.transfer-requests.index') }}" class="btn btn-outline-secondary">
                                <i class="material-icons mr-1">clear</i>
                                Clear
                            </a>
                        </div>
                    </div>

                    <!-- Transfer Requests Table -->
                    <div class="table-responsive">
                        <table class="table table-striped table-hover">
                            <thead class="thead-light">
                                <tr>
                                    <th>Request #</th>
                                    <th>Date</th>
                                    <th>From Project</th>
                                    <th>To Project</th>
                                    <th>Priority</th>
                                    <th>Status</th>
                                    <th>Total Items</th>
                                    <th>Est. Value</th>
                                    <th>Required Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($transferRequests as $request)
                                <tr>
                                    <td>
                                        <strong>{{ $request->transfer_request_number }}</strong>
                                    </td>
                                    <td>{{ $request->request_date->format('M d, Y') }}</td>
                                    <td>
                                        <span class="text-primary">
                                            {{ $request->transferrerProject->project_name }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="text-success">
                                            {{ $request->receiverProject->project_name }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-{{ $request->priority_color }}">
                                            {{ ucfirst($request->priority) }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="badge badge-{{ $request->status_color }}">
                                            @if($request->status === 'approved')
                                                Transferred
                                            @elseif($request->status === 'partially_approved')
                                                Partially Transferred
                                            @else
                                                {{ ucfirst(str_replace('_', ' ', $request->status)) }}
                                            @endif
                                        </span>
                                    </td>
                                    <td>{{ $request->total_items_count }}</td>
                                    <td>AED {{ number_format($request->total_estimated_value, 2) }}</td>
                                    <td>
                                        @if($request->required_date)
                                            {{ $request->required_date->format('M d, Y') }}
                                        @else
                                            <span class="text-muted">Not specified</span>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="btn-group" role="group">
                                            <a href="{{ route('warehouse.transfer-requests.show', $request) }}"
                                               class="btn btn-sm btn-outline-primary" title="View Details">
                                                <i class="material-icons">visibility</i>
                                            </a>
                                            @if($request->status === 'draft')
                                            <a href="{{ route('warehouse.transfer-requests.edit', $request) }}"
                                               class="btn btn-sm btn-outline-warning" title="Edit">
                                                <i class="material-icons">edit</i>
                                            </a>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="10" class="text-center py-4">
                                        <div class="text-muted">
                                            <i class="material-icons mb-2" style="font-size: 48px;">swap_horiz</i>
                                            <p>No transfer requests found</p>
                                            @can('warehouse.outgoing.create')
                                            <a href="{{ route('warehouse.transfer-requests.create') }}" class="btn btn-primary">
                                                Create First Transfer Request
                                            </a>
                                            @endcan
                                        </div>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    <!-- Pagination -->
                    @if($transferRequests->hasPages())
                    <div class="d-flex justify-content-center mt-4">
                        {{ $transferRequests->appends(request()->query())->links() }}
                    </div>
                    @endif
                </div>
            </div>
        </div>
    </div>
</div>

@endsection

@push('scripts')
<script>
function applyFilters() {
    const params = new URLSearchParams();

    // Get filter values (using Select2's val() method for consistency)
    const status = $('#status_filter').val();
    const transferrerProject = $('#transferrer_project_filter').val();
    const receiverProject = $('#receiver_project_filter').val();
    const priority = $('#priority_filter').val();
    const dateFrom = document.getElementById('date_from').value;
    const dateTo = document.getElementById('date_to').value;

    // Add non-empty parameters
    if (status) params.append('status', status);
    if (transferrerProject) params.append('transferrer_project_id', transferrerProject);
    if (receiverProject) params.append('receiver_project_id', receiverProject);
    if (priority) params.append('priority', priority);
    if (dateFrom) params.append('date_from', dateFrom);
    if (dateTo) params.append('date_to', dateTo);

    // Redirect with filters
    window.location.href = '{{ route("warehouse.transfer-requests.index") }}?' + params.toString();
}
</script>
@endpush